/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Dependencies
#include <stdlib.h>
#include "stm32l4xx.h"
#include "stm32l4xx_hal.h"
#include "stm32l4xx_nucleo.h"
#include "os_port.h"
#include "boot_config.h"
#include "drivers/flash/internal/stm32l4xx_flash_driver.h"
#include "image/image.h"
#include "second_stage/boot.h"
#include "memory/memory.h"
#include "second_stage/boot_fallback.h"
#include "core/mailbox.h"
#include "version.h"
#include "debug.h"

uint8_t ledState = 0;
systime_t ledDelay = 0;
systime_t ledTimestamp = 0;

/**
  * @brief  System Clock Configuration
  *         The system Clock is configured as follows :
  *            System Clock source            = PLL (MSI)
  *            SYSCLK(Hz)                     = 80000000
  *            HCLK(Hz)                       = 80000000
  *            AHB Prescaler                  = 1
  *            APB1 Prescaler                 = 1
  *            APB2 Prescaler                 = 1
  *            MSI Frequency(Hz)              = 4000000
  *            PLL_M                          = 1
  *            PLL_N                          = 40
  *            PLL_R                          = 2
  *            PLL_P                          = 7
  *            PLL_Q                          = 4
  *            Flash Latency(WS)              = 4
  * @param  None
  * @retval None
  */
void SystemClock_Config(void) {
	RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
	RCC_OscInitTypeDef RCC_OscInitStruct = {0};

	/* MSI is enabled after System reset, activate PLL with MSI as source */
	RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_MSI;
	RCC_OscInitStruct.MSIState = RCC_MSI_ON;
	RCC_OscInitStruct.MSIClockRange = RCC_MSIRANGE_6;
	RCC_OscInitStruct.MSICalibrationValue = RCC_MSICALIBRATION_DEFAULT;
	RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
	RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_MSI;
	RCC_OscInitStruct.PLL.PLLM = 1;
	RCC_OscInitStruct.PLL.PLLN = 40;
	RCC_OscInitStruct.PLL.PLLR = 2;
	RCC_OscInitStruct.PLL.PLLP = 7;
	RCC_OscInitStruct.PLL.PLLQ = 4;
	if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
		/* Initialization Error */
		while (1);
	}

	/* Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2
	   clocks dividers */
	RCC_ClkInitStruct.ClockType = (
		RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
	RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
	RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
	RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
	RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
	if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4) != HAL_OK) {
		/* Initialization Error */
		while (1);
	}
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask() {
	if ((uint32_t) (systemTicks - ledTimestamp) >= ledDelay) {
		//Toggle LED state
		if (ledState == 0) {
			BSP_LED_On(LED2);
			ledState = 1;
			ledDelay = 100;
		} else {
			BSP_LED_Off(LED2);
			ledState = 0;
			ledDelay = 900;
		}

		ledTimestamp = systemTicks;
	}
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void) {
	BootSettings bootSettings;
	BootContext bootContext;

	cboot_error_t cerror;

	//HAL library initialization
	HAL_Init();
	//Configure the system clock
	SystemClock_Config();
	//Initialize kernel
	osInitKernel();
	//Configure debug UART
	debugInit(115200);

	//Start-up message
	TRACE_INFO("\r\n");
	TRACE_INFO("******************************************\r\n");
	TRACE_INFO("*** CycloneBOOT Single Bank Bootloader ***\r\n");
	TRACE_INFO("******************************************\r\n");
	TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
	TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
	TRACE_INFO("Target: STM32L476\r\n");
	TRACE_INFO("Version: %s\r\n", BOOT_VERSION_STRING);
	TRACE_INFO("\r\n");

	//LED configuration
	BSP_LED_Init(LED2);

	//Clear LEDs
	BSP_LED_Off(LED2);

	//Beginning of handling exception
	do {
		cerror = CBOOT_NO_ERROR;

		//Get default booloader user settings
		bootGetDefaultSettings(&bootSettings);

		//User settings primary memory configuration
		bootSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
		bootSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
		bootSettings.memories[0].driver = &stm32l4xxFlashDriver;
		bootSettings.memories[0].nbSlots = 3;
		//User settings primary memory slot 0 configuration
		bootSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
		bootSettings.memories[0].slots[0].cType = SLOT_CONTENT_BINARY;
		bootSettings.memories[0].slots[0].memParent = &bootSettings.memories[0];
		bootSettings.memories[0].slots[0].addr = 0x08010000;
		bootSettings.memories[0].slots[0].size = 0x20000;

		//User settings primary memory slot 1 configuration
		bootSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
		bootSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE | SLOT_CONTENT_BACKUP;
		bootSettings.memories[0].slots[1].memParent = &bootSettings.memories[0];
		bootSettings.memories[0].slots[1].addr = 0x8030000;
		bootSettings.memories[0].slots[1].size = 0x20000;
		//User settings primary memory slot 2 configuration
		bootSettings.memories[0].slots[2].type = SLOT_TYPE_DIRECT;
		bootSettings.memories[0].slots[2].cType = SLOT_CONTENT_UPDATE | SLOT_CONTENT_BACKUP;
		bootSettings.memories[0].slots[2].memParent = &bootSettings.memories[0];
		bootSettings.memories[0].slots[2].addr = 0x8050000;
		bootSettings.memories[0].slots[2].size = 0x20000;

		mailBoxInit();

		//Initialize bootloader
		cerror = bootInit(&bootContext, &bootSettings);
		//Is any error?
		if (cerror)
			break;
	} while (0);

	//Is any error?
	if (cerror) {
		//Debug message
		TRACE_ERROR("Bootloader configuration failed!\r\n");
		while (1);
	}

	__HAL_FLASH_SET_LATENCY(FLASH_LATENCY_4);

	while (1) {
		ledTask();
		cerror = bootFsm(&bootContext);
		//Is any error?
		if (cerror) {
			//Debug message
			TRACE_ERROR("Bootloader failure!\r\n");
		}
	}

	//This function should never return
	return 0;
}
