/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

/**
 *
 * Flashing the firmware :
 *
 * - 2nd stage bootloader address - 0x08000000
 * - Initial firmware address     - 0x08020000
 *
 * - Use ST-Link or equivalent to flash the bootloader (bootloader.bin) and initial firmware application (iap_demo_bootable.bin) to the MCU internal flash :
 *   E.g.,
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\2nd_stage_bl\bootloader.bin 0x08000000 -halt -rst
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\iap_demo_bootable.bin 0x08020000 -halt -rst
 *
 * Uploading the firmware update :
 *
 * UART/Y-Modem
 *
 *  - The trace outputs are redirected to USART. Exact configuration is found in debug.c file
 *  - Firmware updates performed over USART3. Exact configuration is found in uart_user.c file
 *  - You should obtain a simple command interpreter. Press '1' to put the MCU in update mode (reset the MCU if necessary, if the menu is not found)
 *  - If using Tera Term, select File > Transfer > YMODEM > Send ... and select the update image to be sent to the board (iap_demo_2_0_0.img)
 *  - A progress bar should appear (alongside other information such as speed) on Tera Term client. The progress bar will reach 100% and should automatically disappear after few seconds
 *  - Once the update process is complete, the MCU should restart automatically and you should obtain the greeting message reflecting the new app version (v2.0.0)
 *
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h573i_discovery.h"
#include "stm32h573i_discovery_lcd.h"
#include "stm32_lcd.h"
#include "path.h"
#include "date_time.h"
#include "resource_manager.h"
#include "version.h"
#include "debug.h"
#include "uart_user.h"

//Global variables
uint_t lcdLine = 0;
uint_t lcdColumn = 0;

//Update signature public key resource path
#define UPDT_PUBLIC_SIGN_KEY "keys/my_rsa_pub_key.pem"

//Global variables
const uint8_t *pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;

//IAP mutex handler
OsMutex updateMutex;

void MPU_Config(void);

/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/

void lcdSetCursor(uint_t line, uint_t column) {
    lcdLine = MIN(line, 12);
    lcdColumn = MIN(column, 18);
}


/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/

void lcdPutChar(char_t c) {
    if (c == '\r') {
        lcdColumn = 0;
    } else if (c == '\n') {
        lcdColumn = 0;
        lcdLine++;
    } else if (lcdLine < 12 && lcdColumn < 18) {
        //Display current character
        UTIL_LCD_DisplayChar(lcdColumn * 13 + 4, lcdLine * 20 + 2, c);

        //Advance the cursor position
        if (++lcdColumn >= 18) {
            lcdColumn = 0;
            lcdLine++;
        }
    }
}

void MPU_Config(void)
{
    HAL_MPU_Disable();

    // configure the MPU to prohibit access to after the .bss
    // this will trigger an exception if the stack grows to large
    MPU_Region_InitTypeDef MPU_InitStruct;
    MPU_InitStruct.Enable = MPU_REGION_ENABLE;
    MPU_InitStruct.Number = MPU_REGION_NUMBER0;
    MPU_InitStruct.BaseAddress = 0x08fff000;
    MPU_InitStruct.LimitAddress = 0x08ffffff;
    MPU_InitStruct.AttributesIndex = MPU_ATTRIBUTES_NUMBER0;
    MPU_InitStruct.AccessPermission = MPU_REGION_ALL_RO;
    MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
    MPU_InitStruct.IsShareable = MPU_ACCESS_OUTER_SHAREABLE;
    HAL_MPU_ConfigRegion(&MPU_InitStruct);

    HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void) {
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};
    RCC_CRSInitTypeDef RCC_CRSInitStruct = {0};

    //Configure the main internal regulator output voltage
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

    //Wait for the voltage scaling ready flag to be set
    while (!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {
    }

    //Configure PLL with HSE as source
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI48;
    RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS_DIGITAL;
    RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLM = 5;
    RCC_OscInitStruct.PLL.PLLN = 100;
    RCC_OscInitStruct.PLL.PLLP = 2;
    RCC_OscInitStruct.PLL.PLLQ = 2;
    RCC_OscInitStruct.PLL.PLLR = 2;
    RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_2;
    RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
    RCC_OscInitStruct.PLL.PLLFRACN = 0;
    HAL_RCC_OscConfig(&RCC_OscInitStruct);

    //Select PLL as system clock source and configure bus clocks dividers
    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
                                  RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
    HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

    //Select clock source for RNG peripheral
    RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
    RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
    HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);

    //Configure CRS
    RCC_CRSInitStruct.Prescaler = RCC_CRS_SYNC_DIV1;
    RCC_CRSInitStruct.Source = RCC_CRS_SYNC_SOURCE_USB;
    RCC_CRSInitStruct.Polarity = RCC_CRS_SYNC_POLARITY_RISING;
    RCC_CRSInitStruct.ReloadValue = __HAL_RCC_CRS_RELOADVALUE_CALCULATE(48000000, 1000);
    RCC_CRSInitStruct.ErrorLimitValue = 34;
    RCC_CRSInitStruct.HSI48CalibrationValue = 32;
    HAL_RCCEx_CRSConfig(&RCC_CRSInitStruct);

    //Enable CRS clock
    __HAL_RCC_CRS_CLK_ENABLE();
}

/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param) {
    //Endless loop
    while (1) {
        BSP_LED_On(LED1);
        osDelayTask(100);
        BSP_LED_Off(LED1);
        osDelayTask(900);
    }
}

/**
 * @brief Load user RSA public key from "res.c".
 * "res.c" is a C array file system representing the demo folder "resources/keys".
 **/

error_t updateUserLoadKeys(void) {
    error_t error;

    //Get server's private key
    error = resGetData(UPDT_PUBLIC_SIGN_KEY, (const uint8_t **) &pemUpdtSignPublicKey,
                       &pemUpdtSignPublicKeyLen);
    //Any error to report?
    if (error)
        return error;

    //Successful process
    return NO_ERROR;
}

/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void) {
    error_t error;

    //HAL library initialization
    HAL_Init();
    //Configure the system clock
    SystemClock_Config();

    //Enable instruction cache
    HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
    HAL_ICACHE_Enable();

    MPU_Config();

    //Configure debug UART
    debugInit(115200);

    //Start-up message
    TRACE_INFO("\r\n");
    TRACE_INFO("*****************************************************\r\n");
    TRACE_INFO("*** CycloneBOOT IAP Single-Bank UART Y-MODEM Demo ***\r\n");
    TRACE_INFO("*****************************************************\r\n");
    TRACE_INFO("Version: %s\r\n", APP_VERSION_STRING);
    TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
    TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
    TRACE_INFO("Target: STM32H573\r\n");
    TRACE_INFO("\r\n");

    //LED configuration
    BSP_LED_Init(LED1);
    BSP_LED_Init(LED2);
    BSP_LED_Init(LED3);
    BSP_LED_Init(LED4);

    //Clear LEDs
    BSP_LED_Off(LED1);
    BSP_LED_Off(LED2);
    BSP_LED_Off(LED3);
    BSP_LED_Off(LED4);

    //Initialize user button
    BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

    //Initialize LCD display
    BSP_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
    UTIL_LCD_SetFuncDriver(&LCD_Driver);
    UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_BLUE);
    UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_WHITE);
    UTIL_LCD_SetFont(&Font20);
    BSP_LCD_DisplayOn(0);

    //Clear LCD display
    UTIL_LCD_Clear(UTIL_LCD_COLOR_BLUE);

    //Welcome message
    lcdSetCursor(0, 0);
    printf("IAP UART Y-Modem\r\n");

    //Load IAP public signature key
    error = updateUserLoadKeys();
    if (error) {
        //Debug message
        TRACE_ERROR("Failed to load IAP user keys!\r\n");
        return error;
    }

    //Create IAP mutex
    if (!osCreateMutex(&updateMutex)) {
        //Debug message
        TRACE_ERROR("Failed to create IAP mutex!\r\n");
    }

    while (1) {
        uartUserTask(NULL);
    }

    //This function should never return
    return 0;
}
