/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

/**
 *
 * Flashing the firmware :
 *
 * - 2nd stage bootloader address - 0x08000000
 * - Initial firmware address     - 0x08020000
 *
 * - Use ST-Link or equivalent to flash the bootloader (bootloader.bin) and initial firmware application (iap_demo_bootable.bin) to the MCU internal flash :
 *   E.g.,
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\2nd_stage_bl\bootloader.bin 0x08000000 -halt -rst
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\iap_demo_bootable.bin 0x08020000 -halt -rst
 *
 * Uploading the firmware update :
 *
 *
 *  - The trace outputs are redirected to USART. Exact configuration is found in debug.c file
 *  - Connect USB cable to the board (make sure to use USER USB port instead of ST-Link USB port)
 *  - The blink pattern defined in main.c will indicate the current state of USB : MOUNTED, NOT MOUNTED and UNMOUNTED
 *  - Slow blink indicates the USB has been correctly enumerated and mounted by the host OS
 *  - USB CDC:
 *     - With this device class, the STM32 MCU will present itself as a Virtual COM port:
 *       - Vendor: Oryx Embedded
 *       - Product ID: CycloneBOOT Secure Boot Demo
 *     - Note the exact COM port under which the host OS has registered the device
 *     - Now we are ready to send the firmware update image to the device over USB. In a Terminal, run:
 *       python scripts/send.py <COM PORT> 115200 iap_demo_2_0_0.img. For example:
 *       python3 scripts/send.py /dev/ttyACM1 115200 iap_demo_2_0_0.img
 *     - If successful, the Python script will display: "Firmware transmission completed"
 *     - The serial terminal will show the device rebooting to install the update
 *
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h573i_discovery.h"
#include "stm32h573i_discovery_lcd.h"
#include "stm32_lcd.h"
#include "path.h"
#include "date_time.h"
#include "resource_manager.h"
#include "version.h"
#include "tusb.h"
#include "core/mailbox.h"
#include "update/update.h"
#include "drivers/flash/internal/stm32h5xx_flash_driver.h"
#include "hardware/stm32h5xx/stm32h5xx_crypto.h"
#include "debug.h"

/* Blink pattern
 * - 250 ms  : device not mounted
 * - 1000 ms : device mounted
 * - 2500 ms : device is suspended
 */
enum {
    BLINK_NOT_MOUNTED = 250,
    BLINK_MOUNTED = 1000,
    BLINK_SUSPENDED = 2500,
};

//Global variables
uint_t lcdLine = 0;
uint_t lcdColumn = 0;

//CycloneBOOT Iap Context
UpdateContext updateContext;
//CycloneBOOT Iap Settings
UpdateSettings updateSettings;
//IAP mutex handler
extern OsMutex updateMutex;
//Extern update signature public key declaration
extern const uint8_t* pemUpdtSignPublicKey;
extern size_t pemUpdtSignPublicKeyLen;

// USB related stuff
PCD_HandleTypeDef hpcd_USB_DRD_FS;
static uint32_t blink_interval_ms = BLINK_NOT_MOUNTED;
static bool blink_enable = true;

//Update signature public key resource path
#define UPDT_PUBLIC_SIGN_KEY "keys/my_rsa_pub_key.pem"

//Global variables
const uint8_t *pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;
//Buffer to put data
uint8_t data_buffer[128];

//CycloneBOOT mutex handler
OsMutex updateMutex;

void cdc_task(void);
void led_blinking_task(void);
void push_button_task(void);
static void MX_USB_PCD_Init(void);

void MPU_Config(void);

/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/

void lcdSetCursor(uint_t line, uint_t column) {
    lcdLine = MIN(line, 12);
    lcdColumn = MIN(column, 18);
}


/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/

void lcdPutChar(char_t c) {
    if (c == '\r') {
        lcdColumn = 0;
    } else if (c == '\n') {
        lcdColumn = 0;
        lcdLine++;
    } else if (lcdLine < 12 && lcdColumn < 18) {
        //Display current character
        UTIL_LCD_DisplayChar(lcdColumn * 13 + 4, lcdLine * 20 + 2, c);

        //Advance the cursor position
        if (++lcdColumn >= 18) {
            lcdColumn = 0;
            lcdLine++;
        }
    }
}

void MPU_Config(void)
{
    HAL_MPU_Disable();

    // configure the MPU to prohibit access to after the .bss
    // this will trigger an exception if the stack grows to large
    MPU_Region_InitTypeDef MPU_InitStruct;
    MPU_InitStruct.Enable = MPU_REGION_ENABLE;
    MPU_InitStruct.Number = MPU_REGION_NUMBER0;
    MPU_InitStruct.BaseAddress = 0x08fff000;
    MPU_InitStruct.LimitAddress = 0x08ffffff;
    MPU_InitStruct.AttributesIndex = MPU_ATTRIBUTES_NUMBER0;
    MPU_InitStruct.AccessPermission = MPU_REGION_ALL_RO;
    MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
    MPU_InitStruct.IsShareable = MPU_ACCESS_OUTER_SHAREABLE;
    HAL_MPU_ConfigRegion(&MPU_InitStruct);

    HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void) {
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};
    RCC_CRSInitTypeDef RCC_CRSInitStruct = {0};

    //Configure the main internal regulator output voltage
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

    //Wait for the voltage scaling ready flag to be set
    while (!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {
    }

    //Configure PLL with HSE as source
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI48;
    RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS_DIGITAL;
    RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLM = 5;
    RCC_OscInitStruct.PLL.PLLN = 100;
    RCC_OscInitStruct.PLL.PLLP = 2;
    RCC_OscInitStruct.PLL.PLLQ = 2;
    RCC_OscInitStruct.PLL.PLLR = 2;
    RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_2;
    RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
    RCC_OscInitStruct.PLL.PLLFRACN = 0;
    HAL_RCC_OscConfig(&RCC_OscInitStruct);

    //Select PLL as system clock source and configure bus clocks dividers
    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
                                  RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
    HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

    //Select clock source for RNG peripheral
    RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
    RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
    HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);

    //Configure CRS
    RCC_CRSInitStruct.Prescaler = RCC_CRS_SYNC_DIV1;
    RCC_CRSInitStruct.Source = RCC_CRS_SYNC_SOURCE_USB;
    RCC_CRSInitStruct.Polarity = RCC_CRS_SYNC_POLARITY_RISING;
    RCC_CRSInitStruct.ReloadValue = __HAL_RCC_CRS_RELOADVALUE_CALCULATE(48000000, 1000);
    RCC_CRSInitStruct.ErrorLimitValue = 34;
    RCC_CRSInitStruct.HSI48CalibrationValue = 32;
    HAL_RCCEx_CRSConfig(&RCC_CRSInitStruct);

    //Enable CRS clock
    __HAL_RCC_CRS_CLK_ENABLE();

    /* Select HSI48 as USB clock source */
    RCC_PeriphCLKInitTypeDef usb_clk = {0};
    usb_clk.PeriphClockSelection = RCC_PERIPHCLK_USB;
    usb_clk.UsbClockSelection = RCC_USBCLKSOURCE_HSI48;
    HAL_RCCEx_PeriphCLKConfig(&usb_clk);

    /* Peripheral clock enable */
    __HAL_RCC_USB_CLK_ENABLE();
}


/**
  * @brief USB Initialization Function
  * @param None
  * @retval None
  */
static void MX_USB_PCD_Init(void) {
    __HAL_RCC_GPIOA_CLK_ENABLE();
    __HAL_RCC_USB_CLK_ENABLE();

    RCC_PeriphCLKInitTypeDef PeriphClkInitStruct = {0};
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    // Configure USB DM and DP pins. This is optional, and maintained only for user guidance.
    GPIO_InitStruct.Pin = (GPIO_PIN_11 | GPIO_PIN_12);
    GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
    GPIO_InitStruct.Pull = GPIO_NOPULL;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
    HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

    hpcd_USB_DRD_FS.Instance = USB_DRD_FS;
    hpcd_USB_DRD_FS.Init.dev_endpoints = 8;
    hpcd_USB_DRD_FS.Init.speed = USBD_FS_SPEED;
    hpcd_USB_DRD_FS.Init.phy_itface = PCD_PHY_EMBEDDED;
    hpcd_USB_DRD_FS.Init.Sof_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.low_power_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.lpm_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.battery_charging_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.vbus_sensing_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.bulk_doublebuffer_enable = DISABLE;
    hpcd_USB_DRD_FS.Init.iso_singlebuffer_enable = DISABLE;
    if (HAL_PCD_Init(&hpcd_USB_DRD_FS) != HAL_OK) {
        TRACE_ERROR("Failed to initialize USB device");
    }

    if (hpcd_USB_DRD_FS.Instance == USB_DRD_FS) {
        /** Initializes the peripherals clock
        */
        PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_USB;
        PeriphClkInitStruct.UsbClockSelection = RCC_USBCLKSOURCE_HSI48;
        if (HAL_RCCEx_PeriphCLKConfig(&PeriphClkInitStruct) != HAL_OK) {
            TRACE_ERROR("Failed to initialize USB device");
        }

        /* Enable VDDUSB */
        HAL_PWREx_EnableVddUSB();
        /* USB_DRD_FS interrupt Init */
        HAL_NVIC_SetPriority(USB_DRD_FS_IRQn, 0, 0);
        HAL_NVIC_EnableIRQ(USB_DRD_FS_IRQn);
    }
}

/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param) {
    //Endless loop
    while (1) {
        BSP_LED_On(LED1);
        osDelayTask(100);
        BSP_LED_Off(LED1);
        osDelayTask(900);
    }
}

/**
 * @brief Load user RSA public key from "res.c".
 * "res.c" is a C array file system representing the demo folder "resources/keys".
 **/

error_t updateUserLoadKeys(void) {
    error_t error;

    //Get server's public key
    error = resGetData(UPDT_PUBLIC_SIGN_KEY, (const uint8_t **) &pemUpdtSignPublicKey,
                       &pemUpdtSignPublicKeyLen);
    //Any error to report?
    if (error)
        return error;

    //Successful process
    return NO_ERROR;
}

/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void) {
    error_t error;
    cboot_error_t cboot_error;
    tusb_rhport_init_t dev_init = {0};

    //HAL library initialization
    HAL_Init();
    //Configure the system clock
    SystemClock_Config();

    MPU_Config();

    HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
    HAL_ICACHE_Enable();

    //Configure debug UART
    debugInit(115200);

    //Start-up message
    TRACE_INFO("\r\n");
    TRACE_INFO("************************************************\r\n");
    TRACE_INFO("*** CycloneBOOT IAP Single-Bank USB CDC Demo ***\r\n");
    TRACE_INFO("************************************************\r\n");
    TRACE_INFO("Version: %s\r\n", APP_VERSION_STRING);
    TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
    TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
    TRACE_INFO("Target: STM32H573\r\n");
    TRACE_INFO("\r\n");

    if (mailBoxIsUpdateConfirmed()) {
        TRACE_INFO("Firmware status: CONFIRMED\n");
    } else {
        TRACE_INFO("Firmware status: CANDIDATE\n");
    }
    TRACE_INFO("\r\n");

    //LED configuration
    BSP_LED_Init(LED1);
    BSP_LED_Init(LED2);
    BSP_LED_Init(LED3);
    BSP_LED_Init(LED4);

    //Clear LEDs
    BSP_LED_Off(LED1);
    BSP_LED_Off(LED2);
    BSP_LED_Off(LED3);
    BSP_LED_Off(LED4);

    //Initialize user button
    BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

    //Initialize USB
    MX_USB_PCD_Init();

    //Initialize LCD display
    BSP_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
    UTIL_LCD_SetFuncDriver(&LCD_Driver);
    UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_BLUE);
    UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_WHITE);
    UTIL_LCD_SetFont(&Font20);
    BSP_LCD_DisplayOn(0);

    //Clear LCD display
    UTIL_LCD_Clear(UTIL_LCD_COLOR_BLUE);

    //Welcome message
    lcdSetCursor(0, 0);
    printf("IAP USB CDC\r\n");

    //Initialize hardware cryptographic accelerator
    error = stm32h5xxCryptoInit();
    //Any error to report?
    if(error)
    {
        //Debug message
        TRACE_ERROR("Failed to initialize hardware crypto accelerator!\r\n");
    }

    //Load IAP public signature key
    error = updateUserLoadKeys();
    if (error) {
        //Debug message
        TRACE_ERROR("Failed to load IAP user keys!\r\n");
        return error;
    }

    dev_init.role = TUSB_ROLE_DEVICE;
    dev_init.speed = TUSB_SPEED_AUTO;
    tusb_init(BOARD_TUD_RHPORT, &dev_init);

    //Create IAP mutex
    if (!osCreateMutex(&updateMutex)) {
        //Debug message
        TRACE_ERROR("Failed to create IAP mutex!\r\n");
    }

     //Debug message
   TRACE_INFO("\r\n");
   TRACE_INFO("Starting firmware update...\r\n");

   //Get default Update settings
   updateGetDefaultSettings(&updateSettings);

#if (UPDATE_ANTI_ROLLBACK_SUPPORT == ENABLED)
   updateSettings.appVersion = APP_VERSION;
#endif
   //User update settings security configuration
   updateSettings.imageInCrypto.verifySettings.verifyMethod    = VERIFY_METHOD_SIGNATURE;
   updateSettings.imageInCrypto.verifySettings.signAlgo        = VERIFY_SIGN_RSA;
   updateSettings.imageInCrypto.verifySettings.signHashAlgo    = SHA256_HASH_ALGO;
   updateSettings.imageInCrypto.verifySettings.signKey         = (const char_t*) pemUpdtSignPublicKey;
   updateSettings.imageInCrypto.verifySettings.signKeyLen      = pemUpdtSignPublicKeyLen;
   updateSettings.imageInCrypto.cipherAlgo                     = AES_CIPHER_ALGO;
   updateSettings.imageInCrypto.cipherMode                     = CIPHER_MODE_CBC;
   updateSettings.imageInCrypto.cipherKey                      = "aa3ff7d43cc015682c7dfd00de9379e7";
   updateSettings.imageInCrypto.cipherKeyLen                   = 32;

   //User update settings primary memory configuration
   updateSettings.memories[0].memoryRole           = MEMORY_ROLE_PRIMARY;
   updateSettings.memories[0].memoryType           = MEMORY_TYPE_FLASH;
   updateSettings.memories[0].driver               = &stm32h5xxFlashDriver;
   updateSettings.memories[0].nbSlots              = 3;
    //User settings primary memory slot 0 configuration
    updateSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
    updateSettings.memories[0].slots[0].cType = SLOT_CONTENT_APP;
    updateSettings.memories[0].slots[0].memParent = &updateSettings.memories[0];
    updateSettings.memories[0].slots[0].addr = 0x08020000;
    updateSettings.memories[0].slots[0].size = 0x80000;//0x9E000;//0xA0000;
    //User settings primary memory slot 1 configuration
    updateSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
    updateSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
    updateSettings.memories[0].slots[1].memParent = &updateSettings.memories[0];
    updateSettings.memories[0].slots[1].addr = 0x080C0000;
    updateSettings.memories[0].slots[1].size = 0x80000;//0x9E000;//0xA0000;
    //User settings primary memory slot 2 configuration
    updateSettings.memories[0].slots[2].type = SLOT_TYPE_DIRECT;
    updateSettings.memories[0].slots[2].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
    updateSettings.memories[0].slots[2].memParent = &updateSettings.memories[0];
    updateSettings.memories[0].slots[2].addr = 0x08160000;
    updateSettings.memories[0].slots[2].size = 0x80000;//0x9E000;//0xA0000;

   //Initialize IAP Application context
   cboot_error = updateInit(&updateContext, &updateSettings);
   //Is any error?
   if(cboot_error != CBOOT_NO_ERROR)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize update library!\r\n");
   }

    while (1) {
        tud_task(); // tinyusb device task
        led_blinking_task();
        push_button_task();
        cdc_task();
    }

    //This function should never return
    return 0;
}

//--------------------------------------------------------------------+
// Device callbacks
//--------------------------------------------------------------------+

// Invoked when device is mounted
void tud_mount_cb(void) {
    blink_interval_ms = BLINK_MOUNTED;
}

// Invoked when device is unmounted
void tud_umount_cb(void) {
    blink_interval_ms = BLINK_NOT_MOUNTED;
}

// Invoked when usb bus is suspended
// remote_wakeup_en : if host allow us  to perform remote wakeup
// Within 7ms, device must draw an average of current less than 2.5 mA from bus
void tud_suspend_cb(bool remote_wakeup_en) {
    (void) remote_wakeup_en;
    blink_interval_ms = BLINK_SUSPENDED;
}

// Invoked when usb bus is resumed
void tud_resume_cb(void) {
    blink_interval_ms = tud_mounted() ? BLINK_MOUNTED : BLINK_NOT_MOUNTED;
}


//--------------------------------------------------------------------+
// USB CDC
//--------------------------------------------------------------------+
#if 0
void cdc_task(void) {
    // connected() check for DTR bit
    // Most but not all terminal client set this when making connection
    // if ( tud_cdc_connected() )
    {
        // connected and there are data available
        if (tud_cdc_available()) {
            // read data
            char buf[64];
            uint32_t count = tud_cdc_read(buf, sizeof(buf));
            (void) count;

            // Echo back
            // Note: Skip echo by commenting out write() and write_flush()
            // for throughput test e.g
            //    $ dd if=/dev/zero of=/dev/ttyACM0 count=10000
            tud_cdc_write(buf, count);
            tud_cdc_write_flush();
        }

        // Press on-board button to send Uart status notification
        static cdc_notify_uart_state_t uart_state = {.value = 0};

        static uint32_t btn_prev = 0;
        const uint32_t btn = BSP_PB_GetState(BUTTON_USER);

        if ((btn_prev == 0u) && (btn != 0u)) {
            uart_state.dsr ^= 1;
            tud_cdc_notify_uart_state(&uart_state);
        }
        btn_prev = btn;
    }
}
#else
void cdc_task(void)
{
    cboot_error_t cboot_error;

    if (tud_cdc_connected())
    {
        while (tud_cdc_available())
        {
            // Read up to FW_BUFFER_SIZE bytes from host
            uint32_t count = tud_cdc_read(data_buffer, 128);

            // Process the received firmware chunk
            //Write received bytes in flash
            cboot_error = updateProcess(&updateContext, data_buffer, count);
            //Is any error?
            if(cboot_error != CBOOT_NO_ERROR)
            {
                //Debug message
                TRACE_ERROR("Failed to update firmware!\r\n");
                break;
            }
        }

        if (updateContext.imageProcessCtx.inputImage.state == IMAGE_STATE_VALIDATE_APP) {
            cboot_error = updateFinalize(&updateContext);
            if(cboot_error != CBOOT_NO_ERROR)
            {
                //Debug message
                TRACE_ERROR("Failed to finalize firmware update!\r\n");
            }
            else
            {
                cboot_error = updateReboot(&updateContext);
                if(cboot_error != CBOOT_NO_ERROR)
                {
                    //Debug message
                    TRACE_ERROR("Failed to reboot!\r\n");
                }
            }
        } else {
            // Still processing incoming data
        }
    }
}
#endif
// Invoked when cdc when line state changed e.g connected/disconnected
// Invoked when cdc when line state changed e.g connected/disconnected
void tud_cdc_line_state_cb(uint8_t itf, bool dtr, bool rts) {
    (void) itf;
    (void) rts;

    if (dtr) {
        // Terminal connected
        blink_enable = false;
        //board_led_write(true);
        HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, true);
    } else {
        // Terminal disconnected
        blink_enable = true;
    }
}

// Invoked when CDC interface received data from host
void tud_cdc_rx_cb(uint8_t itf) {
    (void) itf;
}

//--------------------------------------------------------------------+
// BLINKING TASK
//--------------------------------------------------------------------+
void led_blinking_task(void) {
    static uint32_t start_ms = 0;
    static bool led_state = false;

    if (blink_enable) {
        // Blink every interval ms
        if (osGetSystemTime() - start_ms < blink_interval_ms) {
            return; // not enough time
        }
        start_ms += blink_interval_ms;

        //BSP_LED_On(led_state);
        HAL_GPIO_WritePin(LED1_GPIO_PORT, LED1_PIN, led_state);
        led_state = !led_state;
    }
}

void push_button_task(void)
{
    if(BSP_PB_GetState(BUTTON_USER) == SET)
    {
        HAL_Delay(100);

        if(BSP_PB_GetState(BUTTON_USER) == SET)
        {
            TRACE_INFO("Confirming update...\r\n");
            //TODO: uncomment when using flag mechanism
            //Signal to bootloader that app is healthy
            mailBoxSetUpdateConfirmed(TRUE);
            mailBoxClearBootCounter();
        }
    }
}