/**
 * @file helper.c
 * @brief Generic module description
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#include <stdint.h>
#include "stm32h750b_discovery.h"
#include "stm32h750b_discovery_qspi.h"
#include "helper.h"

/**
 * @brief  Flash Memory XiP mode activated ?
 **/

static bool_t flashXiPModeActivated = 0;

/**
 * @brief Initialize flash memory.
 * @return Status code
 **/

error_t flashInit(void)
{
    uint8_t status;

    //Beginning of exceptions handling
    do
    {
        //Deinitialize QSPI external flash memory
        status = BSP_QSPI_DeInit();
        if(status != QSPI_OK)
            break;

        //Initialize QSPI external flash memory
        status = BSP_QSPI_Init();
        if(status != QSPI_OK)
            break;
    }while(0);

    //Is there any error?
    if(status != QSPI_OK)
    {
        //Debug message
        TRACE_INFO("Failed to initialize memory!\r\n");
        return ERROR_FAILURE;
    }

    flashXiPModeActivated = 0;

    //Successful process
    return NO_ERROR;
}


/**
 * @brief Initialize flash memory.
 * @return Status code
 **/

error_t flashActivateXiPMode(bool_t activateXiPMode)
{
    error_t error;
    uint8_t status;

    //Activate XiP mode?
    if(activateXiPMode)
    {
        //Is XiP mode not yet activated?
        if(!flashXiPModeActivated)
        {
            //Activate QSPi external flash memory XiP (eXecute In Place) mode
            status = BSP_QSPI_EnableMemoryMappedMode();
            if(status != QSPI_OK)
            {
                //Debug message
                TRACE_INFO("Failed to setup QSPI in memory mapped mode!\r\n");
                return ERROR_FAILURE;
            }

            //Set flash memory XiP mode activated
            flashXiPModeActivated = 1;
        }
    }
    else
    {
        //Is XiP mode already activated?
        if(flashXiPModeActivated)
        {
            //Initialize flash memory
            error = flashInit();
            //Is any error?
            if(error)
                return error;
            else
                //Set flash memory XiP mode deactivated
                    flashXiPModeActivated = 0;
        }
    }

    //Successful process
    return NO_ERROR;
}
