/**
 * @file curve448.h
 * @brief Curve448 elliptic curve (constant-time implementation)
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _CURVE448_H
#define _CURVE448_H

//Dependencies
#include "core/crypto.h"

//Speed optimization level
#ifndef CURVE448_SPEED_OPTIMIZATION_LEVEL
   #define CURVE448_SPEED_OPTIMIZATION_LEVEL 2
#elif (CURVE448_SPEED_OPTIMIZATION_LEVEL < 0)
   #error CURVE448_SPEED_OPTIMIZATION_LEVEL parameter is not valid
#endif

//Length of the elliptic curve
#define CURVE448_BIT_LEN 448
#define CURVE448_BYTE_LEN 56
#define CURVE448_WORD_LEN 14

//A24 constant
#define CURVE448_A24 39082

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif

//Curve448 related functions
void curve448SetInt(int32_t *a, int32_t b);

void curve448Add(int32_t *r, const int32_t *a, const int32_t *b);
void curve448AddInt(int32_t *r, const int32_t *a, int32_t b);

void curve448Sub(int32_t *r, const int32_t *a, const int32_t *b);
void curve448SubInt(int32_t *r, const int32_t *a, int32_t b);

void curve448Mul(int32_t *r, const int32_t *a, const int32_t *b);
void curve448MulInt(int32_t *r, const int32_t *a, int32_t b);
void curve448Sqr(int32_t *r, const int32_t *a);
void curve448Pwr2(int32_t *r, const int32_t *a, uint_t n);

void curve448Inv(int32_t *r, const int32_t *a);

uint32_t curve448Sqrt(int32_t *r, const int32_t *a, const int32_t *b);

void curve448Canonicalize(int32_t *r, const int32_t *a);

void curve448Copy(int32_t *a, const int32_t *b);
void curve448Swap(int32_t *a, int32_t *b, uint32_t c);

void curve448Select(int32_t *r, const int32_t *a, const int32_t *b,
   uint32_t c);

uint32_t curve448Comp(const int32_t *a, const int32_t *b);

void curve448Import(int32_t *a, const uint8_t *data);
void curve448Export(int32_t *a, uint8_t *data);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
