/**
 * @file eddsa.h
 * @brief EdDSA (Edwards-Curve Digital Signature Algorithm)
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _EDDSA_H
#define _EDDSA_H

//Dependencies
#include "core/crypto.h"
#include "ecc/ec.h"
#include "ecc/ec_curves.h"

//Maximum size of EdDSA public keys
#if (ED448_SUPPORT == ENABLED)
   #define EDDSA_MAX_PUBLIC_KEY_LEN 57
#else
   #define EDDSA_MAX_PUBLIC_KEY_LEN 32
#endif

//Maximum size of EdDSA private keys
#if (ED448_SUPPORT == ENABLED)
   #define EDDSA_MAX_PRIVATE_KEY_LEN 57
#else
   #define EDDSA_MAX_PRIVATE_KEY_LEN 32
#endif

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief EdDSA public key
 **/

typedef struct
{
   const EcCurve *curve;                ///<Elliptic curve parameters
   uint8_t q[EDDSA_MAX_PUBLIC_KEY_LEN]; ///<Public key
} EddsaPublicKey;


/**
 * @brief EdDSA private key
 **/

typedef struct
{
   const EcCurve *curve;                 ///<Elliptic curve parameters
   uint8_t d[EDDSA_MAX_PRIVATE_KEY_LEN]; ///<Private key
   int_t slot;                           ///<Private key slot
   EddsaPublicKey q;                     ///<Public key
} EddsaPrivateKey;


//EdDSA related functions
void eddsaInitPublicKey(EddsaPublicKey *key);
void eddsaFreePublicKey(EddsaPublicKey *key);

void eddsaInitPrivateKey(EddsaPrivateKey *key);
void eddsaFreePrivateKey(EddsaPrivateKey *key);

error_t eddsaGenerateKeyPair(const PrngAlgo *prngAlgo, void *prngContext,
   const EcCurve *curve, EddsaPrivateKey *privateKey,
   EddsaPublicKey *publicKey);

error_t eddsaGeneratePrivateKey(const PrngAlgo *prngAlgo, void *prngContext,
   const EcCurve *curve, EddsaPrivateKey *privateKey);

error_t eddsaGeneratePublicKey(const EddsaPrivateKey *privateKey,
   EddsaPublicKey *publicKey);

error_t eddsaImportPublicKey(EddsaPublicKey *key, const EcCurve *curve,
   const uint8_t *input, size_t length);

error_t eddsaExportPublicKey(const EddsaPublicKey *key, uint8_t *output,
   size_t *written);

error_t eddsaImportPrivateKey(EddsaPrivateKey *key, const EcCurve *curve,
   const uint8_t *data, size_t length);

error_t eddsaExportPrivateKey(const EddsaPrivateKey *key, uint8_t *output,
   size_t *written);

//C++ guard
#ifdef __cplusplus
}
#endif

//Ed25519 supported?
#if (ED25519_SUPPORT == ENABLED)
   #include "ecc/ed25519.h"
#endif

//Ed448 supported?
#if (ED448_SUPPORT == ENABLED)
   #include "ecc/ed448.h"
#endif

#endif
