/**
 * @file mcxe247_crypto_trng.c
 * @brief NXP MCX E247 true random number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "fsl_device_registers.h"
#include "core/crypto.h"
#include "hardware/mcxe247/mcxe247_crypto.h"
#include "hardware/mcxe247/mcxe247_crypto_trng.h"
#include "debug.h"

//Check crypto library configuration
#if (MCXE247_CRYPTO_TRNG_SUPPORT == ENABLED)

//Global variables
static uint8_t buffer[16];
static size_t bufferPos;


/**
 * @brief TRNG module initialization
 * @return Error code
 **/

error_t trngInit(void)
{
   uint32_t status;

   //Mark the buffer as empty
   bufferPos = sizeof(buffer);

   //Check for the previous CSEC command to complete
   while((FTFC->FSTAT & FTFC_FSTAT_CCIF_MASK) == 0)
   {
   }

   //Clear error flags
   FTFC->FSTAT = FTFC_FSTAT_FPVIOL_MASK | FTFC_FSTAT_ACCERR_MASK;

   //Start CSEC command
   ELA_CSEC->DATA_32[0] = CSEC_CMD_INIT_RNG;

   //Wait for the CSEC command to complete
   while((FTFC->FSTAT & FTFC_FSTAT_CCIF_MASK) == 0)
   {
   }

   //Retrieve status code
   status = ELA_CSEC->DATA_32[1] >> 16;

   //Return status code
   return (status == CSEC_ERC_NO_ERROR) ? NO_ERROR : ERROR_FAILURE;
}


/**
 * @brief Get random data from the TRNG module
 * @param[out] data Buffer where to store random data
 * @param[in] length Number of random bytes to generate
 **/

error_t trngGetRandomData(uint8_t *data, size_t length)
{
   size_t i;
   uint32_t temp;
   uint32_t status;

   //Initialize status code
   status = CSEC_ERC_NO_ERROR;

   //Acquire exclusive access to the CSEC module
   osAcquireMutex(&mcxe247CryptoMutex);

   //Generate random data
   for(i = 0; i < length && status == CSEC_ERC_NO_ERROR; i++)
   {
      //Generate a new 128-bit random value when necessary
      if(bufferPos >= sizeof(buffer))
      {
         //Check for the previous CSEC command to complete
         while((FTFC->FSTAT & FTFC_FSTAT_CCIF_MASK) == 0)
         {
         }

         //Clear error flags
         FTFC->FSTAT = FTFC_FSTAT_FPVIOL_MASK | FTFC_FSTAT_ACCERR_MASK;

         //Start CSEC command
         ELA_CSEC->DATA_32[0] = CSEC_CMD_RND;

         //Wait for the CSEC command to complete
         while((FTFC->FSTAT & FTFC_FSTAT_CCIF_MASK) == 0)
         {
         }

         //Retrieve status code
         status = ELA_CSEC->DATA_32[1] >> 16;

         //Check status code
         if(status == CSEC_ERC_NO_ERROR)
         {
            //Save the 128-bit random value
            temp = ELA_CSEC->DATA_32[4];
            STORE32BE(temp, buffer);
            temp = ELA_CSEC->DATA_32[5];
            STORE32BE(temp, buffer + 4);
            temp = ELA_CSEC->DATA_32[6];
            STORE32BE(temp, buffer + 8);
            temp = ELA_CSEC->DATA_32[7];
            STORE32BE(temp, buffer + 12);
         }

         //Rewind to the beginning of the buffer
         bufferPos = 0;
      }

      //Extract a random byte
      data[i] = buffer[bufferPos++];
   }

   //Release exclusive access to the CSEC module
   osReleaseMutex(&mcxe247CryptoMutex);

   //Return status code
   return (status == CSEC_ERC_NO_ERROR) ? NO_ERROR : ERROR_FAILURE;
}

#endif
