/**
 * @file mcxn547_crypto_trng.c
 * @brief NXP MCX N547 true random number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include <mcuxClEls.h>
#include <internal/mcuxClTrng_Internal.h>
#include "core/crypto.h"
#include "hardware/mcxn547/mcxn547_crypto.h"
#include "hardware/mcxn547/mcxn547_crypto_trng.h"
#include "debug.h"

//Global variables
static uint32_t buffer[MCUXCLTRNG_ELS_TRNG_OUTPUT_SIZE / 4];
static size_t bufferPos;

//Check crypto library configuration
#if (MCXN547_CRYPTO_TRNG_SUPPORT == ENABLED)


/**
 * @brief TRNG module initialization
 * @return Error code
 **/

error_t trngInit(void)
{
   error_t error;

   //Initialize status code
   error = NO_ERROR;

   //Mark the buffer as empty
   bufferPos = MCUXCLTRNG_ELS_TRNG_OUTPUT_SIZE;

   //Initialize TRNG
   MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(status, token, mcuxClTrng_Init());

   //Check the protection token and the return value
   if(token != MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClTrng_Init) ||
      status != MCUXCLTRNG_STATUS_OK)
   {
      error = ERROR_FAILURE;
   }

   //End of function call
   MCUX_CSSL_FP_FUNCTION_CALL_END();

   //Return status code
   return error;
}


/**
 * @brief Get random data from the TRNG module
 * @param[out] data Buffer where to store random data
 * @param[in] length Number of random bytes to generate
 **/

error_t trngGetRandomData(uint8_t *data, size_t length)
{
   error_t error;
   size_t i;
   uint32_t temp;

   //Initialize status code
   error = NO_ERROR;

   //Acquire exclusive access to the ELS module
   osAcquireMutex(&mcxn547CryptoMutex);

   //Generate random data
   for(i = 0; i < length && !error; i++)
   {
      //Generate a new 32-byte random value when necessary
      if(bufferPos >= MCUXCLTRNG_ELS_TRNG_OUTPUT_SIZE)
      {
         //Generate random bytes
         MCUX_CSSL_FP_FUNCTION_CALL_BEGIN(status, token, mcuxClTrng_getEntropyInput(
            &elsSession, buffer, MCUXCLTRNG_ELS_TRNG_OUTPUT_SIZE));

         //Check the protection token and the return value
         if(token != MCUX_CSSL_FP_FUNCTION_CALLED(mcuxClTrng_getEntropyInput) ||
            status != MCUXCLTRNG_STATUS_OK)
         {
            error = ERROR_FAILURE;
         }

         //End of function call
         MCUX_CSSL_FP_FUNCTION_CALL_END();

         //Rewind to the beginning of the buffer
         bufferPos = 0;
      }

      //Extract a random byte
      data[i] = (buffer[bufferPos / 4] >> ((bufferPos % 4) * 8)) & 0xFF;
      //Increment index
      bufferPos++;
   }

   //Release exclusive access to the ELS module
   osReleaseMutex(&mcxn547CryptoMutex);

   //Return status code
   return error;
}

#endif
