/**
 * @file mimxrt1180_crypto.c
 * @brief i.MX RT1180 hardware cryptographic accelerator (ELE)
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "fsl_device_registers.h"
#include "ele_crypto.h"
#include "ele_fw.h"
#include "core/crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto_trng.h"
#include "debug.h"

//Global variables
OsMutex mimxrt1180CryptoMutex;


/**
 * @brief Initialize hardware cryptographic accelerator
 * @return Error code
 **/

error_t mimxrt1180CryptoInit(void)
{
   error_t error;
   uint32_t version;
   status_t status;

   //Initialize status code
   error = NO_ERROR;

   //Create a mutex to prevent simultaneous access to the hardware
   //cryptographic accelerator
   if(!osCreateMutex(&mimxrt1180CryptoMutex))
   {
      //Failed to create mutex
      error = ERROR_OUT_OF_RESOURCES;
   }

   //Check status code
   if(!error)
   {
      //Debug message
      TRACE_INFO("Loading ELE firmware...\r\n");

      //Load ELE firmware
      status = ELE_LoadFw(MU_APPS_S3MUA, ele_fw);

      //Check status code
      if(status == kStatus_Success)
      {
         //Retrieve ELE firmware version
         status = ELE_GetFwVersion(MU_APPS_S3MUA, &version);
      }

      //Check status code
      if(status == kStatus_Success)
      {
         //Dump firmware version
         TRACE_INFO("ELE firmware version: 0x%08X\r\n", version);

         //Successful initialization
         error = NO_ERROR;
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }
   }

#if (MIMXRT1180_CRYPTO_TRNG_SUPPORT == ENABLED)
   //Check status code
   if(!error)
   {
      //Initialize TRNG module
      error = trngInit();
   }
#endif

   //Return status code
   return error;
}
