/**
 * @file mimxrt1180_crypto_pkc.c
 * @brief i.MX RT1180 public-key hardware accelerator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "fsl_device_registers.h"
#include "ele_crypto.h"
#include "core/crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto_pkc.h"
#include "pkc/rsa.h"
#include "debug.h"

//Check crypto library configuration
#if (MIMXRT1180_CRYPTO_PKC_SUPPORT == ENABLED)

//ELE RSA parameters
SDK_ALIGN(static EleRsaArgs eleRsaArgs, 8);


#if (RSA_SUPPORT == ENABLED)

/**
 * @brief RSA private key generation
 * @param[in] prngAlgo PRNG algorithm
 * @param[in] prngContext Pointer to the PRNG context
 * @param[in] k Required bit length of the modulus n
 * @param[in] e Public exponent (3, 5, 17, 257 or 65537)
 * @param[out] privateKey RSA private key
 * @return Error code
 **/

error_t rsaGeneratePrivateKey(const PrngAlgo *prngAlgo, void *prngContext,
   size_t k, uint_t e, RsaPrivateKey *privateKey)
{
   error_t error;
   size_t eLen;
   status_t status;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(privateKey == NULL)
      return ERROR_INVALID_PARAMETER;

   //Check the length of the modulus
   if(k > 4096)
      return ERROR_INVALID_PARAMETER;

   //Check the value of the public exponent
   if(e == 3 || e == 5 || e == 17)
   {
      eleRsaArgs.e[0] = e & 0xFF;
      eLen = 1;
   }
   else if(e == 257)
   {
      eleRsaArgs.e[0] = (e >> 8) & 0xFF;
      eleRsaArgs.e[1] = e & 0xFF;
      eLen = 2;
   }
   else if(e == 65537)
   {
      eleRsaArgs.e[0] = (e >> 16) & 0xFF;
      eleRsaArgs.e[1] = (e >> 8) & 0xFF;
      eleRsaArgs.e[2] = e & 0xFF;
      eLen = 3;
   }
   else
   {
      return ERROR_INVALID_PARAMETER;
   }

   //Acquire exclusive access to the ELE module
   osAcquireMutex(&mimxrt1180CryptoMutex);

   //Set RSA parameters
   genericRsa.key_size = k;
   genericRsa.modulus = (uint32_t) eleRsaArgs.n;
   genericRsa.modulus_size = (k + 7) / 8;
   genericRsa.priv_exponent = (uint32_t) eleRsaArgs.d;
   genericRsa.priv_exponent_size = (k + 7) / 8;
   genericRsa.pub_exponent = (uint32_t) eleRsaArgs.e;
   genericRsa.pub_exponent_size = eLen;

   //Generate an RSA private key
   status = ELE_GenericRsaKeygen(MU_APPS_S3MUA, &genericRsa);

   //Check status code
   if(status == kStatus_Success)
   {
      //Copy the public exponent
      error = mpiSetValue(&privateKey->e, e);

      //Check status code
      if(!error)
      {
         //Copy the private exponent
         error = mpiReadRaw(&privateKey->d, eleRsaArgs.d, (k + 7) / 8);
      }

      //Check status code
      if(!error)
      {
         //Copy the modulus
         error = mpiReadRaw(&privateKey->n, eleRsaArgs.n, (k + 7) / 8);
      }

      //Check status code
      if(!error)
      {
         //The first factor is not generated by the hardware
         error = mpiSetValue(&privateKey->p, 0);
      }

      //Check status code
      if(!error)
      {
         //The second factor is not generated by the hardware
         error = mpiSetValue(&privateKey->q, 0);
      }

      //Check status code
      if(!error)
      {
         //The first factor's CRT exponent is not generated by the hardware
         error = mpiSetValue(&privateKey->dp, 0);
      }

      //Check status code
      if(!error)
      {
         //The second factor's CRT exponent is not generated by the hardware
         error = mpiSetValue(&privateKey->dq, 0);
      }

      //Check status code
      if(!error)
      {
         //The CRT coefficient is not generated by the hardware
         error = mpiSetValue(&privateKey->qinv, 0);
      }
   }
   else
   {
      //Report an error
      error = ERROR_FAILURE;
   }

   //Release exclusive access to the ELE module
   osReleaseMutex(&mimxrt1180CryptoMutex);

   //Return status code
   return error;
}


/**
 * @brief RSAES-PKCS1-v1_5 encryption operation
 * @param[in] prngAlgo PRNG algorithm
 * @param[in] prngContext Pointer to the PRNG context
 * @param[in] key Recipient's RSA public key
 * @param[in] message Message to be encrypted
 * @param[in] messageLen Length of the message to be encrypted
 * @param[out] ciphertext Ciphertext resulting from the encryption operation
 * @param[out] ciphertextLen Length of the resulting ciphertext
 * @return Error code
 **/

error_t rsaesPkcs1v15Encrypt(const PrngAlgo *prngAlgo, void *prngContext,
   const RsaPublicKey *key, const uint8_t *message, size_t messageLen,
   uint8_t *ciphertext, size_t *ciphertextLen)
{
   error_t error;
   size_t nLen;
   size_t eLen;
   status_t status;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(prngAlgo == NULL || prngContext == NULL)
      return ERROR_INVALID_PARAMETER;
   if(key == NULL || message == NULL)
      return ERROR_INVALID_PARAMETER;
   if(ciphertext == NULL || ciphertextLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the public exponent, in bytes
   eLen = mpiGetByteLength(&key->e);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && eLen <= 512 && messageLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the public exponent
      mpiWriteRaw(&key->e, eleRsaArgs.e, eLen);
      //Copy the message to be encrypted
      osMemcpy(eleRsaArgs.m, message, messageLen);

      //Set RSA parameters
      genericRsa.algo = RSA_PKCS1_V1_5_CRYPT;
      genericRsa.mode = kEncryption;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.pub_exponent = (uint32_t) eleRsaArgs.e;
      genericRsa.pub_exponent_size = eLen;
      genericRsa.plaintext = (uint32_t) eleRsaArgs.m;
      genericRsa.plaintext_size = messageLen;
      genericRsa.ciphertext = (uint32_t) eleRsaArgs.c;
      genericRsa.ciphertext_size = (nLen + 7) / 8;
      genericRsa.flags = kFlagDigest;

      //Perform RSAES-PKCS1-v1_5 encryption
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Copy the resulting ciphertext
         osMemcpy(ciphertext, eleRsaArgs.c, (nLen + 7) / 8);
         //Return the length of the ciphertext
         *ciphertextLen = (nLen + 7) / 8;

         //Successful processing
         error = NO_ERROR;
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSAES-PKCS1-v1_5 decryption operation
 * @param[in] key Recipient's RSA private key
 * @param[in] ciphertext Ciphertext to be decrypted
 * @param[in] ciphertextLen Length of the ciphertext to be decrypted
 * @param[out] message Output buffer where to store the decrypted message
 * @param[in] messageSize Size of the output buffer
 * @param[out] messageLen Length of the decrypted message
 * @return Error code
 **/

error_t rsaesPkcs1v15Decrypt(const RsaPrivateKey *key,
   const uint8_t *ciphertext, size_t ciphertextLen, uint8_t *message,
   size_t messageSize, size_t *messageLen)
{
   error_t error;
   size_t nLen;
   size_t dLen;
   status_t status;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || ciphertext == NULL)
      return ERROR_INVALID_PARAMETER;
   if(message == NULL || messageSize == 0 || messageLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the private exponent, in bytes
   dLen = mpiGetByteLength(&key->d);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && dLen <= 512 && ciphertextLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the private exponent
      mpiWriteRaw(&key->d, eleRsaArgs.d, dLen);
      //Copy the ciphertext to be decrypted
      osMemcpy(eleRsaArgs.c, ciphertext, ciphertextLen);

      //Set RSA parameters
      genericRsa.algo = RSA_PKCS1_V1_5_CRYPT;
      genericRsa.mode = kDecryption;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.priv_exponent = (uint32_t) eleRsaArgs.d;
      genericRsa.priv_exponent_size = dLen;
      genericRsa.ciphertext = (uint32_t) eleRsaArgs.c;
      genericRsa.ciphertext_size = ciphertextLen;
      genericRsa.plaintext = (uint32_t) eleRsaArgs.m;
      genericRsa.plaintext_size = (nLen + 7) / 8;
      genericRsa.flags = kFlagDigest;

      //Perform RSAES-PKCS1-v1_5 decryption
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Check the length of the resulting plaintext
         if(genericRsa.out_plaintext_len <= messageSize)
         {
            //Copy the resulting message
            osMemcpy(message, eleRsaArgs.m, genericRsa.out_plaintext_len);
            //Return the length of the message
            *messageLen = genericRsa.out_plaintext_len;

            //Successful processing
            error = NO_ERROR;
         }
         else
         {
            //Report an error
            error = ERROR_BUFFER_OVERFLOW;
         }
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSAES-OAEP encryption operation
 * @param[in] prngAlgo PRNG algorithm
 * @param[in] prngContext Pointer to the PRNG context
 * @param[in] key Recipient's RSA public key
 * @param[in] hash Underlying hash function
 * @param[in] label Optional label to be associated with the message
 * @param[in] message Message to be encrypted
 * @param[in] messageLen Length of the message to be encrypted
 * @param[out] ciphertext Ciphertext resulting from the encryption operation
 * @param[out] ciphertextLen Length of the resulting ciphertext
 * @return Error code
 **/

error_t rsaesOaepEncrypt(const PrngAlgo *prngAlgo, void *prngContext,
   const RsaPublicKey *key, const HashAlgo *hash, const char_t *label,
   const uint8_t *message, size_t messageLen, uint8_t *ciphertext,
   size_t *ciphertextLen)
{
   error_t error;
   size_t nLen;
   size_t eLen;
   size_t labelLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(prngAlgo == NULL || prngContext == NULL)
      return ERROR_INVALID_PARAMETER;
   if(key == NULL || message == NULL)
      return ERROR_INVALID_PARAMETER;
   if(ciphertext == NULL || ciphertextLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-1") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA1;
   }
   else if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA224;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA256;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA384;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA512;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the public exponent, in bytes
   eLen = mpiGetByteLength(&key->e);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && eLen <= 512 && messageLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the public exponent
      mpiWriteRaw(&key->e, eleRsaArgs.e, eLen);
      //Copy the message to be encrypted
      osMemcpy(eleRsaArgs.m, message, messageLen);

      //The label is optional
      if(label != NULL)
      {
         osStrcpy(eleRsaArgs.label, label);
         labelLen = osStrlen(label);
      }
      else
      {
         labelLen = 0;
      }

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kEncryption;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.pub_exponent = (uint32_t) eleRsaArgs.e;
      genericRsa.pub_exponent_size = eLen;
      genericRsa.plaintext = (uint32_t) eleRsaArgs.m;
      genericRsa.plaintext_size = messageLen;
      genericRsa.ciphertext = (uint32_t) eleRsaArgs.c;
      genericRsa.ciphertext_size = (nLen + 7) / 8;
      genericRsa.label = (uint32_t) eleRsaArgs.label;
      genericRsa.label_size = labelLen;
      genericRsa.flags = kFlagDigest;

      //Perform RSAES-OAEP encryption
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Copy the resulting ciphertext
         osMemcpy(ciphertext, eleRsaArgs.c, (nLen + 7) / 8);
         //Return the length of the ciphertext
         *ciphertextLen = (nLen + 7) / 8;

         //Successful processing
         error = NO_ERROR;
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSAES-OAEP decryption operation
 * @param[in] key Recipient's RSA private key
 * @param[in] hash Underlying hash function
 * @param[in] label Optional label to be associated with the message
 * @param[in] ciphertext Ciphertext to be decrypted
 * @param[in] ciphertextLen Length of the ciphertext to be decrypted
 * @param[out] message Output buffer where to store the decrypted message
 * @param[in] messageSize Size of the output buffer
 * @param[out] messageLen Length of the decrypted message
 * @return Error code
 **/

error_t rsaesOaepDecrypt(const RsaPrivateKey *key, const HashAlgo *hash,
   const char_t *label, const uint8_t *ciphertext, size_t ciphertextLen,
   uint8_t *message, size_t messageSize, size_t *messageLen)
{
   error_t error;
   size_t nLen;
   size_t dLen;
   size_t labelLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || ciphertext == NULL)
      return ERROR_INVALID_PARAMETER;
   if(message == NULL || messageSize == 0 || messageLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-1") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA1;
   }
   else if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA224;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA256;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA384;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_OAEP_SHA512;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the private exponent, in bytes
   dLen = mpiGetByteLength(&key->d);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && dLen <= 512 && ciphertextLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the private exponent
      mpiWriteRaw(&key->d, eleRsaArgs.d, dLen);
      //Copy the ciphertext to be decrypted
      osMemcpy(eleRsaArgs.c, ciphertext, ciphertextLen);

      //The label is optional
      if(label != NULL)
      {
         osStrcpy(eleRsaArgs.label, label);
         labelLen = osStrlen(label);
      }
      else
      {
         labelLen = 0;
      }

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kDecryption;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.priv_exponent = (uint32_t) eleRsaArgs.d;
      genericRsa.priv_exponent_size = dLen;
      genericRsa.ciphertext = (uint32_t) eleRsaArgs.c;
      genericRsa.ciphertext_size = ciphertextLen;
      genericRsa.plaintext = (uint32_t) eleRsaArgs.m;
      genericRsa.plaintext_size = (nLen + 7) / 8;
      genericRsa.label = (uint32_t) eleRsaArgs.label;
      genericRsa.label_size = labelLen;
      genericRsa.flags = kFlagDigest;

      //Perform RSAES-OAEP decryption
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Check the length of the resulting plaintext
         if(genericRsa.out_plaintext_len <= messageSize)
         {
            //Copy the resulting message
            osMemcpy(message, eleRsaArgs.m, genericRsa.out_plaintext_len);
            //Return the length of the message
            *messageLen = genericRsa.out_plaintext_len;

            //Successful processing
            error = NO_ERROR;
         }
         else
         {
            //Report an error
            error = ERROR_BUFFER_OVERFLOW;
         }
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSASSA-PKCS1-v1_5 signature generation operation
 * @param[in] key Signer's RSA private key
 * @param[in] hash Hash function used to digest the message
 * @param[in] digest Digest of the message to be signed
 * @param[out] signature Resulting signature
 * @param[out] signatureLen Length of the resulting signature
 * @return Error code
 **/

error_t rsassaPkcs1v15Sign(const RsaPrivateKey *key, const HashAlgo *hash,
   const uint8_t *digest, uint8_t *signature, size_t *signatureLen)
{
   error_t error;
   size_t nLen;
   size_t dLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || hash == NULL || digest == NULL)
      return ERROR_INVALID_PARAMETER;
   if(signature == NULL || signatureLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA224_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA256_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA384_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA512_SIGN;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the private exponent, in bytes
   dLen = mpiGetByteLength(&key->d);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && dLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the private exponent
      mpiWriteRaw(&key->d, eleRsaArgs.d, dLen);
      //Copy the digest of the message to be signed
      osMemcpy(eleRsaArgs.digest, digest, hash->digestSize);

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kSignGen;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.priv_exponent = (uint32_t) eleRsaArgs.d;
      genericRsa.priv_exponent_size = dLen;
      genericRsa.digest = (uint32_t) eleRsaArgs.digest;
      genericRsa.digest_size = hash->digestSize;
      genericRsa.signature = (uint32_t) eleRsaArgs.signature;
      genericRsa.signature_size = (nLen + 7) / 8;
      genericRsa.flags = kFlagDigest;

      //Perform RSASSA-PKCS1-v1_5 signature generation
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Copy the resulting message
         osMemcpy(signature, eleRsaArgs.signature, (nLen + 7) / 8);
         //Return the length of the message
         *signatureLen = (nLen + 7) / 8;

         //Successful processing
         error = NO_ERROR;
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSASSA-PKCS1-v1_5 signature verification operation
 * @param[in] key Signer's RSA public key
 * @param[in] hash Hash function used to digest the message
 * @param[in] digest Digest of the message whose signature is to be verified
 * @param[in] signature Signature to be verified
 * @param[in] signatureLen Length of the signature to be verified
 * @return Error code
 **/

error_t rsassaPkcs1v15Verify(const RsaPublicKey *key, const HashAlgo *hash,
   const uint8_t *digest, const uint8_t *signature, size_t signatureLen)
{
   error_t error;
   size_t nLen;
   size_t eLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || hash == NULL || digest == NULL || signature == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA224_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA256_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA384_SIGN;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_V1_5_SHA512_SIGN;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the public exponent, in bytes
   eLen = mpiGetByteLength(&key->e);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && eLen <= 512 && signatureLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the public exponent
      mpiWriteRaw(&key->e, eleRsaArgs.e, eLen);
      //Copy the digest of the message
      osMemcpy(eleRsaArgs.digest, digest, hash->digestSize);
      //Copy the signature to be verified
      osMemcpy(eleRsaArgs.signature, signature, signatureLen);

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kVerification;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.pub_exponent = (uint32_t) eleRsaArgs.e;
      genericRsa.pub_exponent_size = eLen;
      genericRsa.digest = (uint32_t) eleRsaArgs.digest;
      genericRsa.digest_size = hash->digestSize;
      genericRsa.signature = (uint32_t) eleRsaArgs.signature;
      genericRsa.signature_size = signatureLen;
      genericRsa.flags = kFlagDigest;

      //Perform RSASSA-PKCS1-v1_5 signature verification
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Valid RSA signature?
         if(genericRsa.verify_status == kVerifySuccess)
         {
            error = NO_ERROR;
         }
         else if(genericRsa.verify_status == kVerifyFailure)
         {
            error = ERROR_INVALID_SIGNATURE;
         }
         else
         {
            error = ERROR_FAILURE;
         }
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSASSA-PSS signature generation operation
 * @param[in] prngAlgo PRNG algorithm
 * @param[in] prngContext Pointer to the PRNG context
 * @param[in] key Signer's RSA private key
 * @param[in] hash Hash function used to digest the message
 * @param[in] saltLen Length of the salt, in bytes
 * @param[in] digest Digest of the message to be signed
 * @param[out] signature Resulting signature
 * @param[out] signatureLen Length of the resulting signature
 * @return Error code
 **/

error_t rsassaPssSign(const PrngAlgo *prngAlgo, void *prngContext,
   const RsaPrivateKey *key, const HashAlgo *hash, size_t saltLen,
   const uint8_t *digest, uint8_t *signature, size_t *signatureLen)
{
   error_t error;
   size_t nLen;
   size_t dLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || hash == NULL || digest == NULL)
      return ERROR_INVALID_PARAMETER;
   if(signature == NULL || signatureLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA224;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA256;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA384;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA512;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the private exponent, in bytes
   dLen = mpiGetByteLength(&key->d);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && dLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the private exponent
      mpiWriteRaw(&key->d, eleRsaArgs.d, dLen);
      //Copy the digest of the message to be signed
      osMemcpy(eleRsaArgs.digest, digest, hash->digestSize);

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kSignGen;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.priv_exponent = (uint32_t) eleRsaArgs.d;
      genericRsa.priv_exponent_size = dLen;
      genericRsa.digest = (uint32_t) eleRsaArgs.digest;
      genericRsa.digest_size = hash->digestSize;
      genericRsa.signature = (uint32_t) eleRsaArgs.signature;
      genericRsa.signature_size = (nLen + 7) / 8;
      genericRsa.flags = kFlagDigest;

      //Perform RSASSA-PSS signature generation
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Copy the resulting message
         osMemcpy(signature, eleRsaArgs.signature, (nLen + 7) / 8);
         //Return the length of the message
         *signatureLen = (nLen + 7) / 8;

         //Successful processing
         error = NO_ERROR;
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}


/**
 * @brief RSASSA-PSS signature verification operation
 * @param[in] key Signer's RSA public key
 * @param[in] hash Hash function used to digest the message
 * @param[in] saltLen Length of the salt, in bytes
 * @param[in] digest Digest of the message whose signature is to be verified
 * @param[in] signature Signature to be verified
 * @param[in] signatureLen Length of the signature to be verified
 * @return Error code
 **/

error_t rsassaPssVerify(const RsaPublicKey *key, const HashAlgo *hash,
   size_t saltLen, const uint8_t *digest, const uint8_t *signature,
   size_t signatureLen)
{
   error_t error;
   size_t nLen;
   size_t eLen;
   status_t status;
   generic_rsa_algo_t algo;
   ele_generic_rsa_t genericRsa = {0};

   //Check parameters
   if(key == NULL || hash == NULL || digest == NULL || signature == NULL)
      return ERROR_INVALID_PARAMETER;

   //Select the relevant algorithm
   if(osStrcmp(hash->name, "SHA-224") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA224;
   }
   else if(osStrcmp(hash->name, "SHA-256") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA256;
   }
   else if(osStrcmp(hash->name, "SHA-384") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA384;
   }
   else if(osStrcmp(hash->name, "SHA-512") == 0)
   {
      algo = RSA_PKCS1_PSS_MGF1_SHA512;
   }
   else
   {
      return ERROR_UNSUPPORTED_HASH_ALGO;
   }

   //Get the length of the modulus, in bits
   nLen = mpiGetBitLength(&key->n);
   //Get the length of the public exponent, in bytes
   eLen = mpiGetByteLength(&key->e);

   //The accelerator supports operand lengths up to 4096 bits
   if(nLen <= 4096 && eLen <= 512 && signatureLen <= 512)
   {
      //Acquire exclusive access to the ELE module
      osAcquireMutex(&mimxrt1180CryptoMutex);

      //Copy the modulus
      mpiWriteRaw(&key->n, eleRsaArgs.n, (nLen + 7) / 8);
      //Copy the public exponent
      mpiWriteRaw(&key->e, eleRsaArgs.e, eLen);
      //Copy the digest of the message
      osMemcpy(eleRsaArgs.digest, digest, hash->digestSize);
      //Copy the signature to be verified
      osMemcpy(eleRsaArgs.signature, signature, signatureLen);

      //Set RSA parameters
      genericRsa.algo = algo;
      genericRsa.mode = kVerification;
      genericRsa.key_size = nLen;
      genericRsa.modulus = (uint32_t) eleRsaArgs.n;
      genericRsa.modulus_size = (nLen + 7) / 8;
      genericRsa.pub_exponent = (uint32_t) eleRsaArgs.e;
      genericRsa.pub_exponent_size = eLen;
      genericRsa.digest = (uint32_t) eleRsaArgs.digest;
      genericRsa.digest_size = hash->digestSize;
      genericRsa.signature = (uint32_t) eleRsaArgs.signature;
      genericRsa.signature_size = signatureLen;
      genericRsa.flags = kFlagDigest;

      //Perform RSASSA-PSS signature verification
      status = ELE_GenericRsa(MU_APPS_S3MUA, &genericRsa);

      //Check status code
      if(status == kStatus_Success)
      {
         //Valid RSA signature?
         if(genericRsa.verify_status == kVerifySuccess)
         {
            error = NO_ERROR;
         }
         else if(genericRsa.verify_status == kVerifyFailure)
         {
            error = ERROR_INVALID_SIGNATURE;
         }
         else
         {
            error = ERROR_FAILURE;
         }
      }
      else
      {
         //Report an error
         error = ERROR_FAILURE;
      }

      //Release exclusive access to the ELE module
      osReleaseMutex(&mimxrt1180CryptoMutex);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_LENGTH;
   }

   //Return status code
   return error;
}

#endif
#endif
