/**
 * @file mimxrt1180_crypto_trng.c
 * @brief i.MX RT1180 true random number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "fsl_device_registers.h"
#include "ele_crypto.h"
#include "core/crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto.h"
#include "hardware/mimxrt1180/mimxrt1180_crypto_trng.h"
#include "debug.h"

//Check crypto library configuration
#if (MIMXRT1180_CRYPTO_TRNG_SUPPORT == ENABLED)


/**
 * @brief TRNG module initialization
 * @return Error code
 **/

error_t trngInit(void)
{
   status_t status;
   uint32_t state;

   //Initialize TRNG context
   status = ELE_StartRng(MU_APPS_S3MUA);

   //Check status code
   if(status == kStatus_Success)
   {
      //Wait for the TRNG to be ready for use
      do
      {
         //Get TRNG state
         status = ELE_GetTrngState(MU_APPS_S3MUA, &state);

         //Check TRNG state
      } while(status == kStatus_Success && (state & 0xFF) != kELE_TRNG_ready);
   }

   //Return status code
   return (status == kStatus_Success) ? NO_ERROR : ERROR_FAILURE;
}


/**
 * @brief Get random data from the TRNG module
 * @param[out] data Buffer where to store random data
 * @param[in] length Number of random bytes to generate
 **/

error_t trngGetRandomData(uint8_t *data, size_t length)
{
   size_t i;
   uint32_t value;
   status_t status;

   //Initialize status code
   status = kStatus_Success;

   //Acquire exclusive access to the ELE module
   osAcquireMutex(&mimxrt1180CryptoMutex);

   //Generate random data
   for(i = 0; i < length; i++)
   {
      //Generate a new 32-bit random value when necessary
      if((i % 4) == 0)
      {
         //Get 32-bit random value
         status = ELE_RngGetRandom(MU_APPS_S3MUA, &value, 4, kNoReseed);
         //Check status code
         if(status != kStatus_Success)
         {
            break;
         }
      }

      //Copy random byte
      data[i] = value & 0xFF;
      //Shift the 32-bit random value
      value >>= 8;
   }

   //Release exclusive access to the ELE module
   osReleaseMutex(&mimxrt1180CryptoMutex);

   //Return status code
   return (status == kStatus_Success) ? NO_ERROR : ERROR_FAILURE;
}

#endif
