/**
 * @file sc594_crypto_trng.c
 * @brief ADSP-SC594 true random number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include <sys/platform.h>
#include <sys/adi_core.h>
#include <drivers/crypto/adi_trng.h>
#include "core/crypto.h"
#include "hardware/sc594/sc594_crypto.h"
#include "hardware/sc594/sc594_crypto_trng.h"
#include "debug.h"

//Check crypto library configuration
#if (SC594_CRYPTO_TRNG_SUPPORT == ENABLED)

//Global variables
static uint8_t trngMemory[ADI_TRNG_MEMORY_SIZE];
static ADI_TRNG_HANDLE trngHandle;


/**
 * @brief TRNG module initialization
 * @return Error code
 **/

error_t trngInit(void)
{
   ADI_TRNG_RESULT res;

   //Initialize TRNG peripheral
   res = adi_TRNG_Open(&trngHandle, trngMemory, sizeof(trngMemory));

   //Return status code
   return (res == ADI_TRNG_SUCCESS) ? NO_ERROR : ERROR_FAILURE;
}


/**
 * @brief Get random data from the TRNG module
 * @param[out] data Buffer where to store random data
 * @param[in] length Number of random bytes to generate
 **/

error_t trngGetRandomData(uint8_t *data, size_t length)
{
   size_t i;
   uint32_t value;
   ADI_TRNG_RESULT res;

   //Initialize status code
   res = ADI_TRNG_SUCCESS;

   //Acquire exclusive access to the TRNG module
   osAcquireMutex(&sc594CryptoMutex);

   //Generate random data
   for(i = 0; i < length; i++)
   {
      //Generate a new 32-bit random value when necessary
      if((i % 4) == 0)
      {
         //Get 32-bit random value
         res = adi_TRNG_Read_Output(trngHandle, &value, ADI_TRNG_32_BITS);
         //Check status code
         if(res != ADI_TRNG_SUCCESS)
         {
            break;
         }
      }

      //Copy random byte
      data[i] = value & 0xFF;
      //Shift the 32-bit random value
      value >>= 8;
   }

   //Release exclusive access to the TRNG module
   osReleaseMutex(&sc594CryptoMutex);

   //Return status code
   return (res == ADI_TRNG_SUCCESS) ? NO_ERROR : ERROR_FAILURE;
}

#endif
