/**
 * @file pem_common.h
 * @brief PEM common definitions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _PEM_COMMON_H
#define _PEM_COMMON_H

//Dependencies
#include "core/crypto.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief String representation
 **/

typedef struct
{
   const char_t *value;
   size_t length;
} PemString;


/**
 * @brief "Proc-Type" header field
 **/

typedef struct
{
   PemString version;
   PemString type;
} PemProcType;


/**
 * @brief "DEK-Info" header field
 **/

typedef struct
{
   PemString algo;
   PemString iv;
} PemDekInfo;


/**
 * @brief PEM encapsulated header
 **/

typedef struct
{
   PemProcType procType;
   PemDekInfo dekInfo;
} PemHeader;


//PEM related functions
error_t pemDecodeFile(const char_t *input, size_t inputLen, const char_t *label,
   uint8_t *output, size_t *outputLen, PemHeader *header, size_t *consumed);

error_t pemEncodeFile(const void *input, size_t inputLen, const char_t *label,
   char_t *output, size_t *outputLen);

error_t pemParseHeader(const char_t *input, size_t inputLen,
   PemHeader *header, size_t *consumed);

void pemParseHeaderField(PemString *line, PemHeader *header);

int_t pemFindTag(const char_t *input, size_t inputLen, const char_t *tag1,
   const char_t *tag2, const char_t *tag3);

int_t pemFindChar(const PemString *s, char_t c);
bool_t pemCompareString(const PemString *string, const char_t *value);
bool_t pemTokenizeString(PemString *s, char_t c, PemString *token);
void pemTrimWhitespace(PemString *s);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
