/**
 * @file pem_key_export.h
 * @brief PEM key file export functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _PEM_KEY_EXPORT_H
#define _PEM_KEY_EXPORT_H

//Dependencies
#include "core/crypto.h"
#include "pkix/pem_common.h"
#include "pkc/rsa.h"
#include "pkc/dsa.h"
#include "ecc/ec.h"
#include "ecc/eddsa.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief PEM public key formats
 **/

typedef enum
{
   PEM_PUBLIC_KEY_FORMAT_DEFAULT = 0, ///<Default format
   PEM_PUBLIC_KEY_FORMAT_PKCS1   = 1, ///<PKCS #1 format
   PEM_PUBLIC_KEY_FORMAT_RFC7468 = 2  ///<RFC 7468 format
} PemPublicKeyFormat;


/**
 * @brief PEM private key formats
 **/

typedef enum
{
   PEM_PRIVATE_KEY_FORMAT_DEFAULT  = 0, ///<Default format
   PEM_PRIVATE_KEY_FORMAT_PKCS1    = 1, ///<PKCS #1 format
   PEM_PRIVATE_KEY_FORMAT_PKCS8    = 2, ///<PKCS #8 v1 format
   PEM_PRIVATE_KEY_FORMAT_PKCS8_V2 = 3, ///<PKCS #8 v2 format
   PEM_PRIVATE_KEY_FORMAT_RFC5915  = 4  ///<RFC 5915 format
} PemPrivateKeyFormat;


//PEM related functions
error_t pemExportRsaPublicKey(const RsaPublicKey *publicKey,
   char_t *output, size_t *written, PemPublicKeyFormat format);

error_t pemExportRsaPrivateKey(const RsaPrivateKey *privateKey,
   char_t *output, size_t *written, PemPrivateKeyFormat format);

error_t pemExportRsaPssPublicKey(const RsaPublicKey *publicKey,
   char_t *output, size_t *written, PemPublicKeyFormat format);

error_t pemExportRsaPssPrivateKey(const RsaPrivateKey *privateKey,
   char_t *output, size_t *written, PemPrivateKeyFormat format);

error_t pemExportDsaPublicKey(const DsaPublicKey *publicKey,
   char_t *output, size_t *written, PemPublicKeyFormat format);

error_t pemExportDsaPrivateKey(const DsaPrivateKey *privateKey,
   char_t *output, size_t *written, PemPrivateKeyFormat format);

error_t pemExportEcPublicKey(const EcPublicKey *publicKey,
   char_t *output, size_t *written, PemPublicKeyFormat format);

error_t pemExportEcPrivateKey(const EcPrivateKey *privateKey,
   char_t *output, size_t *written, PemPrivateKeyFormat format);

error_t pemExportEddsaPublicKey(const EddsaPublicKey *publicKey,
   char_t *output, size_t *written, PemPublicKeyFormat format);

error_t pemExportEddsaPrivateKey(const EddsaPrivateKey *privateKey,
   char_t *output, size_t *written, PemPrivateKeyFormat format);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
