/**
 * @file ctr_drbg.h
 * @brief CTR_DRBG pseudorandom number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _CTR_DRBG_H
#define _CTR_DRBG_H

//Dependencies
#include "core/crypto.h"
#include "cipher/cipher_algorithms.h"

//Maximum length of the key
#define CTR_DRBG_MAX_KEY_LEN 32
//Maximum length of the seed
#define CTR_DRBG_MAX_SEED_LEN 48
//Maximum number of requests between reseeds
#define CTR_DRBG_MAX_RESEED_INTERVAL 281474976710656ULL

//Common interface for PRNG algorithms
#define CTR_DRBG_PRNG_ALGO (&ctrDrbgPrngAlgo)

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief CTR_DRBG PRNG context
 **/

typedef struct
{
   OsMutex mutex;                    ///<Mutex preventing simultaneous access to the PRNG state
   const CipherAlgo *cipherAlgo;     ///<Cipher function
   CipherContext cipherContext;      ///<Cipher context
   size_t keyLen;                    ///<Key length
   size_t securityStrength;          ///<Security strength
   bool_t df;                        ///<Use key derivation function
   size_t ctrLen;                    ///<Counter length
   size_t seedLen;                   ///<Seed length
   uint8_t v[MAX_CIPHER_BLOCK_SIZE]; ///<Value V
   uint8_t k[CTR_DRBG_MAX_KEY_LEN];  ///<Key
   uint64_t reseedCounter;           ///<Reseed counter
} CtrDrbgContext;


//CTR_DRBG related constants
extern const PrngAlgo ctrDrbgPrngAlgo;

//CTR_DRBG related functions
error_t ctrDrbgInit(CtrDrbgContext *context, const CipherAlgo *cipherAlgo,
   size_t keyLen, bool_t df);

error_t ctrDrbgSeed(CtrDrbgContext *context, const uint8_t *seed,
   size_t length);

error_t ctrDrbgSeedEx(CtrDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *nonce, size_t nonceLen,
   const uint8_t *personalizationString, size_t personalizationStringLen);

error_t ctrDrbgReseed(CtrDrbgContext *context, const uint8_t *seed,
   size_t length);

error_t ctrDrbgReseedEx(CtrDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *additionalInput,
   size_t additionalInputLen);

error_t ctrDrbgGenerate(CtrDrbgContext *context, uint8_t *output,
   size_t length);

error_t ctrDrbgGenerateEx(CtrDrbgContext *context,
   const uint8_t *additionalInput, size_t additionalInputLen, uint8_t *output,
   size_t outputLen);

void ctrDrbgDeinit(CtrDrbgContext *context);

error_t blockCipherDf(CtrDrbgContext *context, const DataChunk *input,
   uint_t inputLen, uint8_t *output, size_t outputLen);

error_t ctrDrbgBcc(CtrDrbgContext *context, const uint8_t *key,
   const DataChunk *data, uint_t dataLen, uint8_t *output);

error_t ctrDrbgUpdate(CtrDrbgContext *context, const uint8_t *providedData,
   size_t providedDataLen);

error_t ctrDrbgLoadKey(CtrDrbgContext *context, const uint8_t *key);

void ctrDrbgIncBlock(uint8_t *ctr, size_t blockLen, size_t ctrLen);
void ctrDrbgXorBlock(uint8_t *x, const uint8_t *a, const uint8_t *b, size_t n);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
