/**
 * @file hmac_drbg.c
 * @brief HMAC_DRBG pseudorandom number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "core/crypto.h"
#include "rng/hmac_drbg.h"
#include "debug.h"

//Check crypto library configuration
#if (HMAC_DRBG_SUPPORT == ENABLED)

//Common interface for PRNG algorithms
const PrngAlgo hmacDrbgPrngAlgo =
{
   "HMAC_DRBG",
   sizeof(HmacDrbgContext),
   (PrngAlgoInit) NULL,
   (PrngAlgoSeed) hmacDrbgSeed,
   (PrngAlgoReseed) hmacDrbgReseed,
   (PrngAlgoGenerate) hmacDrbgGenerate,
   (PrngAlgoDeinit) hmacDrbgDeinit
};


/**
 * @brief Initialize PRNG context
 * @param[in] context Pointer to the HMAC_DRBG context to initialize
 * @param[in] hashAlgo Approved hash function
 * @return Error code
 **/

error_t hmacDrbgInit(HmacDrbgContext *context, const HashAlgo *hashAlgo)
{
   //Check parameters
   if(context == NULL || hashAlgo == NULL)
      return ERROR_INVALID_PARAMETER;

   //Clear the internal state
   osMemset(context, 0, sizeof(HmacDrbgContext));

   //Create a mutex to prevent simultaneous access to the PRNG state
   if(!osCreateMutex(&context->mutex))
   {
      //Failed to create mutex
      return ERROR_OUT_OF_RESOURCES;
   }

   //HMAC_DRBG uses multiple occurrences of an approved keyed hash function,
   //which is based on an approved hash function
   context->hashAlgo = hashAlgo;

   //Determine the security strength (refer to SP 800-57, section 5.6.1.2)
   if(hashAlgo->digestSize <= 20)
   {
      context->securityStrength = 16;
   }
   else if(hashAlgo->digestSize <= 28)
   {
      context->securityStrength = 24;
   }
   else
   {
      context->securityStrength = 32;
   }

   //Successful initialization
   return NO_ERROR;
}


/**
 * @brief Seed the PRNG state
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] seed String of bits obtained from the randomness source
 * @param[in] length Length of the string, in bytes
 * @return Error code
 **/

error_t hmacDrbgSeed(HmacDrbgContext *context, const uint8_t *seed,
   size_t length)
{
   //Seeding process
   return hmacDrbgSeedEx(context, seed, length, NULL, 0, NULL, 0);
}


/**
 * @brief Seed the PRNG state (with nonce and personalization string)
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] entropyInput String of bits obtained from the randomness source
 * @param[in] entropyInputLen Length of the string, in bytes
 * @param[in] nonce Nonce
 * @param[in] nonceLen Length of the nonce, in bytes
 * @param[in] personalizationString Personalization string received from the
 *   consuming application
 * @param[in] personalizationStringLen Length of the personalization string,
 *   in bytes
 * @return Error code
 **/

error_t hmacDrbgSeedEx(HmacDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *nonce, size_t nonceLen,
   const uint8_t *personalizationString, size_t personalizationStringLen)
{
   size_t outLen;
   DataChunk seedMaterial[3];

   //Check parameters
   if(context == NULL || entropyInput == NULL)
      return ERROR_INVALID_PARAMETER;

   //The nonce parameter is optional
   if(nonce == NULL && nonceLen != 0)
      return ERROR_INVALID_PARAMETER;

   //The personalization_string parameter is optional
   if(personalizationString == NULL && personalizationStringLen != 0)
      return ERROR_INVALID_PARAMETER;

   //HMAC_DRBG uses multiple occurrences of an approved keyed hash function,
   //which is based on an approved hash function
   if(context->hashAlgo == NULL)
      return ERROR_PRNG_NOT_READY;

   //The entropy input shall have entropy that is equal to or greater than the
   //security strength of the instantiation
   if(entropyInputLen < context->securityStrength)
      return ERROR_INVALID_PARAMETER;

   //If an implementation utilizes a nonce in the construction of a seed during
   //instantiation, the length of the nonce shall be at least half the maximum
   //security strength supported. Per allowances in SP 800-90A, the length of a
   //nonce may be less than 1/2 the maximum security strength supported as long
   //as the entropy input length + the nonce length >= 3/2 security strength
   if((entropyInputLen + nonceLen) < (3 * context->securityStrength / 2))
      return ERROR_INVALID_PARAMETER;

   //Acquire exclusive access to the PRNG state
   osAcquireMutex(&context->mutex);

   //Determine the output block length
   outLen = context->hashAlgo->digestSize;

   //Let seed_material = entropy_input || nonce || personalization_string
   seedMaterial[0].buffer = entropyInput;
   seedMaterial[0].length = entropyInputLen;
   seedMaterial[1].buffer = nonce;
   seedMaterial[1].length = nonceLen;
   seedMaterial[2].buffer = personalizationString;
   seedMaterial[2].length = personalizationStringLen;

   //Set Key = 0x00 00...00
   osMemset(context->k, 0x00, outLen);
   //V = 0x01 01...01
   osMemset(context->v, 0x01, outLen);

   //Compute (Key, V) = HMAC_DRBG_Update(seed_material, Key, V)
   hmacDrbgUpdate(context, seedMaterial, arraysize(seedMaterial));

   //Reset reseed_counter
   context->reseedCounter = 1;

   //Release exclusive access to the PRNG state
   osReleaseMutex(&context->mutex);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Reseed the PRNG state
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] seed String of bits obtained from the randomness source
 * @param[in] length Length of the string, in bytes
 * @return Error code
 **/

error_t hmacDrbgReseed(HmacDrbgContext *context, const uint8_t *seed,
   size_t length)
{
   //Reseeding process
   return hmacDrbgReseedEx(context, seed, length, NULL, 0);
}


/**
 * @brief Reseed the PRNG state (with additional input)
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] entropyInput String of bits obtained from the randomness source
 * @param[in] entropyInputLen Length of the string, in bytes
 * @param[in] additionalInput Additional input string received from the
 *   consuming application
 * @param[in] additionalInputLen Length of the additional input string, in bytes
 * @return Error code
 **/

error_t hmacDrbgReseedEx(HmacDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *additionalInput,
   size_t additionalInputLen)
{
   DataChunk seedMaterial[2];

   //Check parameters
   if(context == NULL || entropyInput == NULL)
      return ERROR_INVALID_PARAMETER;

   //The additional_input parameter is optional
   if(additionalInput == NULL && additionalInputLen != 0)
      return ERROR_INVALID_PARAMETER;

   //HMAC_DRBG uses multiple occurrences of an approved keyed hash function,
   //which is based on an approved hash function
   if(context->hashAlgo == NULL)
      return ERROR_PRNG_NOT_READY;

   //Check whether the DRBG has been properly instantiated
   if(context->reseedCounter == 0)
      return ERROR_PRNG_NOT_READY;

   //The entropy input shall have entropy that is equal to or greater than the
   //security strength of the instantiation
   if(entropyInputLen < context->securityStrength)
      return ERROR_INVALID_PARAMETER;

   //Acquire exclusive access to the PRNG state
   osAcquireMutex(&context->mutex);

   //Let seed_material = entropy_input || additional_input
   seedMaterial[0].buffer = entropyInput;
   seedMaterial[0].length = entropyInputLen;
   seedMaterial[1].buffer = additionalInput;
   seedMaterial[1].length = additionalInputLen;

   //Compute (Key, V) = HMAC_DRBG_Update(seed_material, Key, V)
   hmacDrbgUpdate(context, seedMaterial, arraysize(seedMaterial));

   //Reset reseed_counter
   context->reseedCounter = 1;

   //Release exclusive access to the PRNG state
   osReleaseMutex(&context->mutex);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Generate pseudorandom data
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[out] output Buffer where to store the pseudorandom bytes
 * @param[in] length Requested number of bytes
 * @return Error code
 **/

error_t hmacDrbgGenerate(HmacDrbgContext *context, uint8_t *output,
   size_t length)
{
   //Generation process
   return hmacDrbgGenerateEx(context, NULL, 0, output, length);
}


/**
 * @brief Generate pseudorandom data (with additional input)
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] additionalInput Additional input string received from the
 *   consuming application
 * @param[in] additionalInputLen Length of the additional input string, in bytes
 * @param[out] output Buffer where to store the pseudorandom bytes
 * @param[in] outputLen Requested number of bytes
 * @return Error code
 **/

error_t hmacDrbgGenerateEx(HmacDrbgContext *context,
   const uint8_t *additionalInput, size_t additionalInputLen, uint8_t *output,
   size_t outputLen)
{
   size_t n;
   size_t outLen;
   DataChunk providedData[1];
   HmacContext *hmacContext;

   //Check parameters
   if(context == NULL)
      return ERROR_INVALID_PARAMETER;

   //The additional_input parameter is optional
   if(additionalInput == NULL && additionalInputLen != 0)
      return ERROR_INVALID_PARAMETER;

   //HMAC_DRBG uses multiple occurrences of an approved keyed hash function,
   //which is based on an approved hash function
   if(context->hashAlgo == NULL)
      return ERROR_PRNG_NOT_READY;

   //A DRBG shall be instantiated prior to the generation of pseudorandom bits
   if(context->reseedCounter == 0)
      return ERROR_PRNG_NOT_READY;

   //If reseed_counter > reseed_interval, then return an indication that a
   //reseed is required
   if(context->reseedCounter > HMAC_DRBG_MAX_RESEED_INTERVAL)
      return ERROR_RESEED_REQUIRED;

   //Acquire exclusive access to the PRNG state
   osAcquireMutex(&context->mutex);

   //Point to the HMAC context
   hmacContext = &context->hmacContext;
   //Determine the output block length
   outLen = context->hashAlgo->digestSize;

   //The additional input string is received from the consuming application
   providedData[0].buffer = additionalInput;
   providedData[0].length = additionalInputLen;

   //The length of the additional input string may be zero
   if(additionalInputLen > 0)
   {
      //Compute (Key, V) = HMAC_DRBG_Update(additional_input, Key, V)
      hmacDrbgUpdate(context, providedData, arraysize(providedData));
   }

   //Generate the requested number of bytes
   while(outputLen > 0)
   {
      //Number of bytes to generate at a time
      n = MIN(outputLen, outLen);

      //Compute V = HMAC(Key, V)
      hmacInit(hmacContext, context->hashAlgo, context->k, outLen);
      hmacUpdate(hmacContext, context->v, outLen);
      hmacFinal(hmacContext, context->v);

      //Copy data to the output buffer
      osMemcpy(output, context->v, n);

      //Next output block
      output += n;
      outputLen -= n;
   }

   //Compute (Key, V) = HMAC_DRBG_Update(additional_input, Key, V)
   if(additionalInputLen > 0)
   {
      hmacDrbgUpdate(context, providedData, arraysize(providedData));
   }
   else
   {
      hmacDrbgUpdate(context, NULL, 0);
   }

   //Increment reseed_counter
   context->reseedCounter++;

   //Release exclusive access to the PRNG state
   osReleaseMutex(&context->mutex);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Release PRNG context
 * @param[in] context Pointer to the HMAC_DRBG context
 **/

void hmacDrbgDeinit(HmacDrbgContext *context)
{
   //Valid HMAC_DRBG context?
   if(context != NULL)
   {
      //Free previously allocated resources
      osDeleteMutex(&context->mutex);

      //Erase the contents of the internal state
      osMemset(context, 0, sizeof(HmacDrbgContext));
   }
}


/**
 * @brief Update internal state
 * @param[in] context Pointer to the HMAC_DRBG context
 * @param[in] providedData The data to be used
 * @param[in] providedDataLen Number of data chunks representing the data
 **/

void hmacDrbgUpdate(HmacDrbgContext *context, const DataChunk *providedData,
   uint_t providedDataLen)
{
   uint8_t c;
   size_t i;
   size_t outLen;
   HmacContext *hmacContext;

   //Point to the HMAC context
   hmacContext = &context->hmacContext;
   //Determine the output block length
   outLen = context->hashAlgo->digestSize;

   //Constant byte 0x00
   c = 0;

   //Compute K = HMAC(K, V || 0x00 || provided_data)
   hmacInit(hmacContext, context->hashAlgo, context->k, outLen);
   hmacUpdate(hmacContext, context->v, outLen);
   hmacUpdate(hmacContext, &c, sizeof(c));

   for(i = 0; i < providedDataLen; i++)
   {
      hmacUpdate(hmacContext, providedData[i].buffer,
         providedData[i].length);
   }

   hmacFinal(hmacContext, context->k);

   //Compute V = HMAC(K, V)
   hmacInit(hmacContext, context->hashAlgo, context->k, outLen);
   hmacUpdate(hmacContext, context->v, outLen);
   hmacFinal(hmacContext, context->v);

   //Any data provided?
   if(providedDataLen > 0)
   {
      //Constant byte 0x01
      c = 1;

      //Compute K = HMAC(K, V || 0x01 || provided_data)
      hmacInit(hmacContext, context->hashAlgo, context->k, outLen);
      hmacUpdate(hmacContext, context->v, outLen);
      hmacUpdate(hmacContext, &c, sizeof(c));

      for(i = 0; i < providedDataLen; i++)
      {
         hmacUpdate(hmacContext, providedData[i].buffer,
            providedData[i].length);
      }

      hmacFinal(hmacContext, context->k);

      //Compute V = HMAC(K, V)
      hmacInit(hmacContext, context->hashAlgo, context->k, outLen);
      hmacUpdate(hmacContext, context->v, outLen);
      hmacFinal(hmacContext, context->v);
   }
}

#endif
