/**
 * @file hmac_drbg.h
 * @brief HMAC_DRBG pseudorandom number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _HMAC_DRBG_H
#define _HMAC_DRBG_H

//Dependencies
#include "core/crypto.h"
#include "mac/hmac.h"

//Maximum number of requests between reseeds
#define HMAC_DRBG_MAX_RESEED_INTERVAL 281474976710656ULL

//Common interface for PRNG algorithms
#define HMAC_DRBG_PRNG_ALGO (&hmacDrbgPrngAlgo)

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief HMAC_DRBG PRNG context
 **/

typedef struct
{
   OsMutex mutex;                   ///<Mutex preventing simultaneous access to the PRNG state
   const HashAlgo *hashAlgo;        ///<Hash function
   HmacContext hmacContext;         ///<HMAC context
   size_t securityStrength;         ///<Security strength
   uint8_t v[MAX_HASH_DIGEST_SIZE]; ///<Value V
   uint8_t k[MAX_HASH_DIGEST_SIZE]; ///<Key
   uint64_t reseedCounter;          ///<Reseed counter
} HmacDrbgContext;


//HMAC_DRBG related constants
extern const PrngAlgo hmacDrbgPrngAlgo;

//HMAC_DRBG related functions
error_t hmacDrbgInit(HmacDrbgContext *context, const HashAlgo *hashAlgo);

error_t hmacDrbgSeed(HmacDrbgContext *context, const uint8_t *seed,
   size_t length);

error_t hmacDrbgSeedEx(HmacDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *nonce, size_t nonceLen,
   const uint8_t *personalizationString, size_t personalizationStringLen);

error_t hmacDrbgReseed(HmacDrbgContext *context, const uint8_t *seed,
   size_t length);

error_t hmacDrbgReseedEx(HmacDrbgContext *context, const uint8_t *entropyInput,
   size_t entropyInputLen, const uint8_t *additionalInput,
   size_t additionalInputLen);

error_t hmacDrbgGenerate(HmacDrbgContext *context, uint8_t *output,
   size_t length);

error_t hmacDrbgGenerateEx(HmacDrbgContext *context,
   const uint8_t *additionalInput, size_t additionalInputLen, uint8_t *output,
   size_t outputLen);

void hmacDrbgDeinit(HmacDrbgContext *context);

void hmacDrbgUpdate(HmacDrbgContext *context, const DataChunk *providedData,
   uint_t providedDataLen);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
