/**
 * @file xdrbg.h
 * @brief XDRBG pseudorandom number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _XDRBG_H
#define _XDRBG_H

//Dependencies
#include "core/crypto.h"
#include "xof/xof_algorithms.h"

//Maximum length of the additional data
#define XDRBG_MAX_ALPHA_LEN 84
//Maximum size of the internal state
#define XDRBG_MAX_V_SIZE 64

//Common interface for PRNG algorithms
#define XDRBG_PRNG_ALGO (&xdrbgPrngAlgo)

//Encode function
#define XDRBG_ENCODE(alphaLen, n) ((n * 85) + alphaLen)

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief XDRBG PRNG context
 **/

typedef struct
{
   OsMutex mutex;               ///<Mutex preventing simultaneous access to the PRNG state
   const XofAlgo *xofAlgo;      ///<XOF algorithm
   XofContext xofContext;       ///<XOF context
   size_t securityStrength;     ///<Security strength
   size_t maxOutputLen;         ///<Maximum output length
   uint8_t v[XDRBG_MAX_V_SIZE]; ///<Internal state V
   uint64_t reseedCounter;      ///<Reseed counter
} XdrbgContext;


//XDRBG related constants
extern const PrngAlgo xdrbgPrngAlgo;

//XDRBG related functions
error_t xdrbgInit(XdrbgContext *context, const XofAlgo *xofAlgo);

error_t xdrbgSeed(XdrbgContext *context, const uint8_t *seed, size_t length);

error_t xdrbgSeedEx(XdrbgContext *context, const uint8_t *seed, size_t seedLen,
   const uint8_t *alpha, size_t alphaLen);

error_t xdrbgReseed(XdrbgContext *context, const uint8_t *seed, size_t length);

error_t xdrbgReseedEx(XdrbgContext *context, const uint8_t *seed,
   size_t seedLen, const uint8_t *alpha, size_t alphaLen);

error_t xdrbgGenerate(XdrbgContext *context, uint8_t *output, size_t length);

error_t xdrbgGenerateEx(XdrbgContext *context, const uint8_t *alpha,
   size_t alphaLen, uint8_t *output, size_t outputLen);

void xdrbgDeinit(XdrbgContext *context);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
