/**
 * @file ssh_algorithms.h
 * @brief SSH algorithm negotiation
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_ALGORITHMS_H
#define _SSH_ALGORITHMS_H

//Dependencies
#include "ssh/ssh.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif

//SSH related functions
error_t sshFormatKexAlgoList(SshConnection *connection, uint8_t *p,
   size_t *written);

error_t sshFormatHostKeyAlgoList(SshContext *context, uint8_t *p,
   size_t *written);

error_t sshFormatEncAlgoList(SshContext *context, uint8_t *p, size_t *written);
error_t sshFormatMacAlgoList(SshContext *context, uint8_t *p, size_t *written);

error_t sshFormatCompressionAlgoList(SshContext *context, uint8_t *p,
   size_t *written);

error_t sshFormatPublicKeyAlgoList(SshContext *context, uint8_t *p,
   size_t *written);

const char_t *sshSelectAlgo(SshContext *context, const SshNameList *peerAlgoList,
   const char_t *const *supportedAlgoList, uint_t supportedAlgoListLen);

const char_t *sshSelectKexAlgo(SshConnection *connection,
   const SshNameList *peerAlgoList);

const char_t *sshSelectHostKeyAlgo(SshContext *context,
   const SshNameList *peerAlgoList);

const char_t *sshSelectEncAlgo(SshContext *context,
   const SshNameList *peerAlgoList);

const char_t *sshSelectMacAlgo(SshContext *context, const char_t *encAlgo,
   const SshNameList *peerAlgoList);

const char_t *sshSelectCompressionAlgo(SshContext *context,
   const SshNameList *peerAlgoList);

const char_t *sshSelectPublicKeyAlgo(SshContext *context,
   const char_t *keyFormatId, const SshNameList *peerAlgoList);

const char_t *sshGetKeyFormatId(const SshString *publicKeyAlgo);
const char_t *sshGetSignFormatId(const SshString *publicKeyAlgo);

bool_t sshIsGuessCorrect(SshContext *context, const SshNameList *kexAlgoList,
   const SshNameList *hostKeyAlgoList);

bool_t sshIsRsaKexAlgo(const char_t *kexAlgo);
bool_t sshIsDhKexAlgo(const char_t *kexAlgo);
bool_t sshIsDhGexKexAlgo(const char_t *kexAlgo);
bool_t sshIsEcdhKexAlgo(const char_t *kexAlgo);
bool_t sshIsKemKexAlgo(const char_t *kexAlgo);
bool_t sshIsHybridKexAlgo(const char_t *kexAlgo);

bool_t sshIsCertPublicKeyAlgo(const SshString *publicKeyAlgo);
bool_t sshIsX509CertPublicKeyAlgo(const SshString *publicKeyAlgo);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
