/**
 * @file ssh_auth.h
 * @brief SSH user authentication protocol
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_AUTH_H
#define _SSH_AUTH_H

//Dependencies
#include "ssh/ssh.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief Authentication methods
 **/

typedef enum
{
   SSH_AUTH_METHOD_NONE       = 0,
   SSH_AUTH_METHOD_PASSWORD   = 1,
   SSH_AUTH_METHOD_PUBLIC_KEY = 2,
   SSH_AUTH_METHOD_HOST_BASED = 3
} SshAuthMethod;


//SSH related functions
error_t sshSendUserAuthBanner(SshConnection *connection,
   const char_t *banner);

error_t sshSendUserAuthRequest(SshConnection *connection);
error_t sshSendUserAuthSuccess(SshConnection *connection);
error_t sshSendUserAuthFailure(SshConnection *connection);

error_t sshAcceptAuthRequest(SshConnection *connection);
error_t sshRejectAuthRequest(SshConnection *connection);

error_t sshFormatUserAuthBanner(SshConnection *connection,
   const char_t *banner, uint8_t *p, size_t *length);

error_t sshFormatUserAuthRequest(SshConnection *connection, uint8_t *message,
   size_t *length);

error_t sshFormatNoneAuthParams(SshConnection *connection, uint8_t *p,
   size_t *written);

error_t sshFormatUserAuthSuccess(SshConnection *connection, uint8_t *p,
   size_t *length);

error_t sshFormatUserAuthFailure(SshConnection *connection, uint8_t *p,
   size_t *length);

error_t sshFormatUserAuthMethods(SshConnection *connection, uint8_t *p,
   size_t *written);

error_t sshParseUserAuthBanner(SshConnection *connection,
   const uint8_t *message, size_t length);

error_t sshParseUserAuthRequest(SshConnection *connection,
   const uint8_t *message, size_t length);

error_t sshParseNoneAuthParams(SshConnection *connection,
   const SshString *userName, const uint8_t *p, size_t length);

error_t sshParseUserAuthSuccess(SshConnection *connection,
   const uint8_t *message, size_t length);

error_t sshParseUserAuthFailure(SshConnection *connection,
   const uint8_t *message, size_t length);

error_t sshParseUserAuthMessage(SshConnection *connection, uint8_t type,
   const uint8_t *message, size_t length);

SshAuthMethod sshGetAuthMethod(SshConnection *connection);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
