/**
 * @file ssh_cert_export.c
 * @brief SSH certificate export functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL SSH_TRACE_LEVEL

//Dependencies
#include "ssh/ssh.h"
#include "ssh/ssh_cert_export.h"
#include "ssh/ssh_cert_parse.h"
#include "ssh/ssh_misc.h"
#include "encoding/base64.h"
#include "debug.h"

//Check SSH stack configuration
#if (SSH_SUPPORT == ENABLED && SSH_CERT_SUPPORT == ENABLED)


/**
 * @brief Export SSH certificate (OpenSSH format)
 * @param[in] input Certificate structure to encode
 * @param[in] inputLen Length of the certificate structure to encode
 * @param[out] output Resulting certificate file (optional parameter)
 * @param[out] outputLen Length of the resulting certificate file
 **/

error_t sshExportCertificate(const void *input, size_t inputLen,
   char_t *output, size_t *outputLen)
{
   error_t error;
   size_t n;
   SshCertificate cert;
   uint8_t identifier[40];

   //Check parameters
   if(input == NULL || outputLen == NULL)
      return ERROR_INVALID_PARAMETER;

   //Parse certificate structure
   error = sshParseCertificate(input, inputLen, &cert);
   //Any error to report?
   if(error)
      return error;

   //Sanity check
   if(cert.keyFormatId.length > sizeof(identifier))
      return ERROR_WRONG_IDENTIFIER;

   //Save key format identifier
   osMemcpy(identifier, cert.keyFormatId.value, cert.keyFormatId.length);

   //Encode the certificate structure using Base64
   base64Encode(input, inputLen, output, &n);

   //If the output parameter is NULL, then the function calculates the length
   //of the resulting certificate file without copying any data
   if(output != NULL)
   {
      //Make room for the identifier string
      osMemmove(output + cert.keyFormatId.length + 1, output, n + 1);
      //Copy identifier string
      osMemcpy(output, identifier, cert.keyFormatId.length);
      //The identifier must be followed by a whitespace character
      output[cert.keyFormatId.length] = ' ';
   }

   //Consider the length of the identifier string
   n += cert.keyFormatId.length + 1;

   //Total number of bytes that have been written
   *outputLen = n;

   //Successful processing
   return NO_ERROR;
}

#endif
