/**
 * @file ssh_cert_parse.h
 * @brief SSH certificate parsing
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_CERT_PARSE_H
#define _SSH_CERT_PARSE_H

//Dependencies
#include "ssh_types.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief SSH certificate types
 **/

typedef enum
{
   SSH_CERT_TYPE_USER = 1,
   SSH_CERT_TYPE_HOST = 2
} SshCertType;


/**
 * @brief RSA public key
 **/

typedef struct
{
   SshBinaryString e;
   SshBinaryString n;
} SshRsaCertPublicKey;


/**
 * @brief DSA public key
 **/

typedef struct
{
   SshBinaryString p;
   SshBinaryString q;
   SshBinaryString g;
   SshBinaryString y;
} SshDsaCertPublicKey;


/**
 * @brief ECDSA public key
 **/

typedef struct
{
   SshString curveName;
   SshBinaryString q;
} SshEcdsaCertPublicKey;


/**
 * @brief EdDSA public key
 **/

typedef struct
{
   SshBinaryString q;
} SshEddsaCertPublicKey;


/**
 * @brief Public key
 **/

typedef union
{
   SshRsaCertPublicKey rsaPublicKey;
   SshDsaCertPublicKey dsaPublicKey;
   SshEcdsaCertPublicKey ecdsaPublicKey;
   SshEddsaCertPublicKey eddsaPublicKey;
} SshCertPublicKey;


/**
 * @brief SSH certificate (OpenSSH format)
 **/

typedef struct
{
   SshString keyFormatId;
   SshBinaryString nonce;
   SshCertPublicKey publicKey;
   uint64_t serial;
   uint32_t type;
   SshString keyId;
   SshBinaryString validPrincipals;
   uint64_t validAfter;
   uint64_t validBefore;
   SshBinaryString criticalOptions;
   SshBinaryString extensions;
   SshBinaryString reserved;
   SshBinaryString signatureKey;
   SshBinaryString signature;
} SshCertificate;


//SSH certificate parsing functions
error_t sshParseCertificate(const uint8_t *data, size_t length,
   SshCertificate *cert);

error_t sshParseRsaCertPublicKey(const uint8_t *data, size_t length,
   size_t *consumed, SshRsaCertPublicKey *publicKey);

error_t sshParseDsaCertPublicKey(const uint8_t *data, size_t length,
   size_t *consumed, SshDsaCertPublicKey *publicKey);

error_t sshParseEcdsaCertPublicKey(const uint8_t *data, size_t length,
   size_t *consumed, SshEcdsaCertPublicKey *publicKey);

error_t sshParseEddsaCertPublicKey(const uint8_t *data, size_t length,
   size_t *consumed, SshEddsaCertPublicKey *publicKey);

error_t sshParseValidPrincipals(const uint8_t *data, size_t length,
   SshBinaryString *validPrincipals);

error_t sshParseCriticalOptions(const uint8_t *data, size_t length,
   SshBinaryString *criticalOptions);

error_t sshParseExtensions(const uint8_t *data, size_t length,
   SshBinaryString *extensions);

bool_t sshGetValidPrincipal(const SshCertificate *cert, uint_t index,
   SshString *name);

bool_t sshGetCriticalOption(const SshCertificate *cert, uint_t index,
   SshString *name, SshBinaryString *data);

bool_t sshGetExtension(const SshCertificate *cert, uint_t index,
   SshString *name, SshBinaryString *data);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
