/**
 * @file ssh_key_export.h
 * @brief SSH key file export functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_KEY_EXPORT_H
#define _SSH_KEY_EXPORT_H

//Dependencies
#include "ssh_types.h"
#include "pkc/rsa.h"
#include "pkc/dsa.h"
#include "ecc/ec.h"
#include "ecc/eddsa.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief SSH public key formats
 **/

typedef enum
{
   SSH_PUBLIC_KEY_FORMAT_DEFAULT = 0, ///<Default public key format
   SSH_PUBLIC_KEY_FORMAT_SSH2    = 1, ///<SSH2 public key format
   SSH_PUBLIC_KEY_FORMAT_OPENSSH = 2  ///<OpenSSH public key format
} SshPublicKeyFormat;


/**
 * @brief SSH private key formats
 **/

typedef enum
{
   SSH_PRIVATE_KEY_FORMAT_DEFAULT = 0, ///<Default private key format
   SSH_PRIVATE_KEY_FORMAT_OPENSSH = 2  ///<OpenSSH private key format
} SshPrivateKeyFormat;


//SSH key file export functions
error_t sshExportRsaPublicKey(const RsaPublicKey *publicKey,
   char_t *output, size_t *written, SshPublicKeyFormat format);

error_t sshExportDsaPublicKey(const DsaPublicKey *publicKey,
   char_t *output, size_t *written, SshPublicKeyFormat format);

error_t sshExportEcdsaPublicKey(const EcPublicKey *publicKey,
   char_t *output, size_t *written, SshPublicKeyFormat format);

error_t sshExportEd25519PublicKey(const EddsaPublicKey *publicKey,
   char_t *output, size_t *written, SshPublicKeyFormat format);

error_t sshExportEd448PublicKey(const EddsaPublicKey *publicKey,
   char_t *output, size_t *written, SshPublicKeyFormat format);

error_t sshExportRsaPrivateKey(const RsaPrivateKey *privateKey,
   char_t *output, size_t *written, SshPrivateKeyFormat format);

error_t sshExportDsaPrivateKey(const DsaPrivateKey *privateKey,
   char_t *output, size_t *written, SshPrivateKeyFormat format);

error_t sshExportEcdsaPrivateKey(const EcPrivateKey *privateKey,
   char_t *output, size_t *written, SshPrivateKeyFormat format);

error_t sshExportEd25519PrivateKey(const EddsaPrivateKey *privateKey,
   char_t *output, size_t *written, SshPrivateKeyFormat format);

error_t sshExportEd448PrivateKey(const EddsaPrivateKey *privateKey,
   char_t *output, size_t *written, SshPrivateKeyFormat format);

error_t sshExportOpenSshRsaPrivateKey(const RsaPrivateKey *privateKey,
   char_t *output, size_t *written);

error_t sshExportOpenSshDsaPrivateKey(const DsaPrivateKey *privateKey,
   char_t *output, size_t *written);

error_t sshExportOpenSshEcdsaPrivateKey(const EcPrivateKey *privateKey,
   char_t *output, size_t *written);

error_t sshExportOpenSshEd25519PrivateKey(const EddsaPrivateKey *privateKey,
   char_t *output, size_t *written);

error_t sshExportOpenSshEd448PrivateKey(const EddsaPrivateKey *privateKey,
   char_t *output, size_t *written);

error_t sshEncodePublicKeyFile(const char_t *keyFormatId, const void *input,
   size_t inputLen, char_t *output, size_t *outputLen, SshPublicKeyFormat format);

error_t sshEncodeSsh2PublicKeyFile(const void *input, size_t inputLen,
   char_t *output, size_t *outputLen);

error_t sshEncodeOpenSshPublicKeyFile(const char_t *keyFormatId,
   const void *input, size_t inputLen, char_t *output, size_t *outputLen);

error_t sshEncodeOpenSshPrivateKeyFile(const void *input, size_t inputLen,
   char_t *output, size_t *outputLen);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
