/**
 * @file ssh_key_import.h
 * @brief SSH key file import functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_KEY_IMPORT_H
#define _SSH_KEY_IMPORT_H

//Dependencies
#include "ssh_types.h"
#include "ssh_key_parse.h"
#include "pkix/x509_common.h"
#include "pkc/rsa.h"
#include "pkc/dsa.h"
#include "ecc/ec.h"
#include "ecc/eddsa.h"

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief SSH key type
 **/

typedef struct
{
   const char_t *identifier;
   X509KeyType type;
   const char_t *curveName;
} SshKeyType;


//SSH key file import functions
error_t sshImportRsaPublicKey(RsaPublicKey *publicKey, const char_t *input,
   size_t length);

error_t sshImportDsaPublicKey(DsaPublicKey *publicKey, const char_t *input,
   size_t length);

error_t sshImportEcdsaPublicKey(EcPublicKey *publicKey, const char_t *input,
   size_t length);

error_t sshImportEd25519PublicKey(EddsaPublicKey *publicKey,
   const char_t *input, size_t length);

error_t sshImportEd448PublicKey(EddsaPublicKey *publicKey,
   const char_t *input, size_t length);

error_t sshImportRsaPrivateKey(RsaPrivateKey *privateKey, const char_t *input,
   size_t length, const char_t *password);

error_t sshImportDsaPrivateKey(DsaPrivateKey *privateKey, const char_t *input,
   size_t length, const char_t *password);

error_t sshImportEcdsaPrivateKey(EcPrivateKey *privateKey, const char_t *input,
   size_t length, const char_t *password);

error_t sshImportEd25519PrivateKey(EddsaPrivateKey *privateKey,
   const char_t *input, size_t length, const char_t *password);

error_t sshImportEd448PrivateKey(EddsaPrivateKey *privateKey,
   const char_t *input, size_t length, const char_t *password);

error_t sshImportRsaHostKey(RsaPublicKey *publicKey,
   const SshRsaHostKey *hostKey);

error_t sshImportDsaHostKey(DsaPublicKey *publicKey,
   const SshDsaHostKey *hostKey);

error_t sshImportEcdsaHostKey(EcPublicKey *publicKey,
   const SshEcdsaHostKey *hostKey);

const char_t *sshGetPublicKeyType(const char_t *input, size_t length);

error_t sshDecodePublicKeyFile(const char_t *input, size_t inputLen,
   uint8_t *output, size_t *outputLen);

error_t sshDecodeSsh2PublicKeyFile(const char_t *input, size_t inputLen,
   uint8_t *output, size_t *outputLen);

error_t sshDecodeOpenSshPublicKeyFile(const char_t *input, size_t inputLen,
   uint8_t *output, size_t *outputLen);

error_t sshDecodeOpenSshPrivateKeyFile(const char_t *input, size_t inputLen,
   uint8_t *output, size_t *outputLen);

int_t sshSearchMarker(const char_t *s, size_t sLen, const char_t *marker,
   size_t markerLen);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
