/**
 * @file ssh_key_parse.h
 * @brief SSH key parsing
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_KEY_PARSE_H
#define _SSH_KEY_PARSE_H

//Dependencies
#include "ssh_types.h"

//Magic identifier size
#define SSH_AUTH_MAGIC_SIZE 15


//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief RSA host key
 **/

typedef struct
{
   SshString keyFormatId;
   SshBinaryString e;
   SshBinaryString n;
} SshRsaHostKey;


/**
 * @brief DSA host key
 **/

typedef struct
{
   SshString keyFormatId;
   SshBinaryString p;
   SshBinaryString q;
   SshBinaryString g;
   SshBinaryString y;
} SshDsaHostKey;


/**
 * @brief ECDSA host key
 **/

typedef struct
{
   SshString keyFormatId;
   SshString curveName;
   SshBinaryString q;
} SshEcdsaHostKey;


/**
 * @brief EdDSA host key
 **/

typedef struct
{
   SshString keyFormatId;
   SshBinaryString q;
} SshEddsaHostKey;


/**
 * @brief Private key header (OpenSSH format)
 **/

typedef struct
{
   SshString cipherName;
   SshString kdfName;
   SshBinaryString kdfOptions;
   uint32_t numKeys;
   SshBinaryString publicKey;
   SshBinaryString encrypted;
} SshPrivateKeyHeader;


/**
 * @brief RSA private key (OpenSSH format)
 **/

typedef struct
{
   uint32_t checkInt1;
   uint32_t checkInt2;
   SshString keyFormatId;
   SshBinaryString n;
   SshBinaryString e;
   SshBinaryString d;
   SshBinaryString qinv;
   SshBinaryString p;
   SshBinaryString q;
   SshString comment;
} SshRsaPrivateKey;


/**
 * @brief DSA private key (OpenSSH format)
 **/

typedef struct
{
   uint32_t checkInt1;
   uint32_t checkInt2;
   SshString keyFormatId;
   SshBinaryString p;
   SshBinaryString q;
   SshBinaryString g;
   SshBinaryString y;
   SshBinaryString x;
   SshString comment;
} SshDsaPrivateKey;


/**
 * @brief ECDSA private key (OpenSSH format)
 **/

typedef struct
{
   uint32_t checkInt1;
   uint32_t checkInt2;
   SshString keyFormatId;
   SshString curveName;
   SshBinaryString q;
   SshBinaryString d;
   SshString comment;
} SshEcdsaPrivateKey;


/**
 * @brief EdDSA private key (OpenSSH format)
 **/

typedef struct
{
   uint32_t checkInt1;
   uint32_t checkInt2;
   SshString keyFormatId;
   SshBinaryString q;
   SshBinaryString d;
   SshString comment;
} SshEddsaPrivateKey;


//SSH key parsing functions
error_t sshParseHostKey(const uint8_t *data, size_t length,
   SshString *keyFormatId);

error_t sshParseRsaHostKey(const uint8_t *data, size_t length,
   SshRsaHostKey *hostKey);

error_t sshParseDsaHostKey(const uint8_t *data, size_t length,
   SshDsaHostKey *hostKey);

error_t sshParseEcdsaHostKey(const uint8_t *data, size_t length,
   SshEcdsaHostKey *hostKey);

error_t sshParseEd25519HostKey(const uint8_t *data, size_t length,
   SshEddsaHostKey *hostKey);

error_t sshParseEd448HostKey(const uint8_t *data, size_t length,
   SshEddsaHostKey *hostKey);

error_t sshParseOpenSshPrivateKeyHeader(const uint8_t *data, size_t length,
   SshPrivateKeyHeader *privateKeyHeader);

error_t sshParseOpenSshRsaPrivateKey(const uint8_t *data, size_t length,
   SshRsaPrivateKey *privateKey);

error_t sshParseOpenSshDsaPrivateKey(const uint8_t *data, size_t length,
   SshDsaPrivateKey *privateKey);

error_t sshParseOpenSshEcdsaPrivateKey(const uint8_t *data, size_t length,
   SshEcdsaPrivateKey *privateKey);

error_t sshParseOpenSshEd25519PrivateKey(const uint8_t *data, size_t length,
   SshEddsaPrivateKey *privateKey);

error_t sshParseOpenSshEd448PrivateKey(const uint8_t *data, size_t length,
   SshEddsaPrivateKey *privateKey);

error_t sshCheckPrivateKeyPadding(const uint8_t *pad, size_t length);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
