/**
 * @file ssh_misc.h
 * @brief SSH helper functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneSSH Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _SSH_MISC_H
#define _SSH_MISC_H

//Dependencies
#include "ssh/ssh.h"
#include "mpi/mpi.h"

//Maximum port number
#define SSH_MAX_PORT_NUM 65535

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif

//SSH related functions
SshConnection *sshOpenConnection(SshContext *context, Socket *socket);
void sshCloseConnection(SshConnection *connection);

void sshRegisterConnectionEvents(SshContext *context,
   SshConnection *connection, SocketEventDesc *eventDesc);

error_t sshProcessConnectionEvents(SshContext *context,
   SshConnection *connection);

void sshRegisterUserEvents(SshChannel *channel, OsEvent *event,
   uint_t eventMask);

void sshUnregisterUserEvents(SshChannel *channel);
uint_t sshGetUserEvents(SshChannel *channel);

void sshNotifyEvent(SshContext *context);

SshHostKey *sshGetHostKey(SshConnection *connection);
int_t sshSelectHostKey(SshContext *context, const char_t *hostKeyAlgo);
int_t sshSelectNextHostKey(SshConnection *connection);

error_t sshFormatHostKey(SshConnection *connection, uint8_t *p,
   size_t *written);

const EcCurve *sshGetCurve(const SshString *keyFormatId,
   const SshString *curveName);

error_t sshParseString(const uint8_t *p, size_t length, SshString *string);

error_t sshParseBinaryString(const uint8_t *p, size_t length,
   SshBinaryString *string);

error_t sshParseNameList(const uint8_t *p, size_t length,
   SshNameList *nameList);

int_t sshFindName(const SshNameList *nameList, const char_t *name);
bool_t sshGetName(const SshNameList *nameList, uint_t index, SshString *name);

error_t sshFormatString(const char_t *value, uint8_t *p, size_t *written);

error_t sshFormatBinaryString(const void *value, size_t valueLen, uint8_t *p,
   size_t *written);

error_t sshFormatNameList(const char_t *const nameList[], uint_t nameListLen,
   uint8_t *p, size_t *written);

error_t sshFormatMpint(const Mpi *value, uint8_t *p, size_t *written);

error_t sshConvertScalarToMpint(const uint32_t *value, uint_t length,
   uint8_t *p, size_t *written);

error_t sshConvertArrayToMpint(const uint8_t *value, size_t length, uint8_t *p,
   size_t *written);

bool_t sshCompareString(const SshString *string, const char_t *value);
bool_t sshCompareStrings(const SshString *string1, const SshString *string2);
bool_t sshCompareAlgo(const char_t *name1, const char_t *name2);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
