/**
 * @file coap_server.c
 * @brief CoAP server
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL COAP_TRACE_LEVEL

//Dependencies
#include <stdlib.h>
#include "coap/coap_server.h"
#include "coap/coap_server_transport.h"
#include "coap/coap_server_misc.h"
#include "coap/coap_debug.h"
#include "debug.h"

//Check TCP/IP stack configuration
#if (COAP_SERVER_SUPPORT == ENABLED)


/**
 * @brief Initialize settings with default values
 * @param[out] settings Structure that contains CoAP server settings
 **/

void coapServerGetDefaultSettings(CoapServerSettings *settings)
{
   //Default task parameters
   settings->task = OS_TASK_DEFAULT_PARAMS;
   settings->task.stackSize = COAP_SERVER_STACK_SIZE;
   settings->task.priority = COAP_SERVER_PRIORITY;

   //TCP/IP stack context
   settings->netContext = NULL;
   //The CoAP server is not bound to any interface
   settings->interface = NULL;

   //CoAP port number
   settings->port = COAP_PORT;

   //UDP initialization callback
   settings->udpInitCallback = NULL;

#if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
   //DTLS initialization callback function
   settings->dtlsInitCallback = NULL;
#endif

   //CoAP request callback function
   settings->requestCallback = NULL;
}


/**
 * @brief CoAP server initialization
 * @param[in] context Pointer to the CoAP server context
 * @param[in] settings CoAP server specific settings
 * @return Error code
 **/

error_t coapServerInit(CoapServerContext *context,
   const CoapServerSettings *settings)
{
   error_t error;

   //Debug message
   TRACE_INFO("Initializing CoAP server...\r\n");

   //Ensure the parameters are valid
   if(context == NULL || settings == NULL)
      return ERROR_INVALID_PARAMETER;

   //Initialize status code
   error = NO_ERROR;

   //Clear the CoAP server context
   osMemset(context, 0, sizeof(CoapServerContext));

   //Initialize task parameters
   context->taskParams = settings->task;
   context->taskId = OS_INVALID_TASK_ID;

   //Attach TCP/IP stack context
   if(settings->netContext != NULL)
   {
      context->netContext = settings->netContext;
   }
   else if(settings->interface != NULL)
   {
      context->netContext = settings->interface->netContext;
   }
   else
   {
      context->netContext = netGetDefaultContext();
   }

   //Save user settings
   context->interface = settings->interface;
   context->port = settings->port;
   context->udpInitCallback = settings->udpInitCallback;
#if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
   context->dtlsInitCallback = settings->dtlsInitCallback;
#endif
   context->requestCallback = settings->requestCallback;

   //Create an event object to poll the state of the UDP socket
   if(!osCreateEvent(&context->event))
   {
      //Failed to create event
      error = ERROR_OUT_OF_RESOURCES;
   }

   //Check status code
   if(error)
   {
      //Clean up side effects
      coapServerDeinit(context);
   }

   //Return status code
   return error;
}


/**
 * @brief Set cookie secret
 *
 * This function specifies the cookie secret used while generating and
 * verifying a cookie during the DTLS handshake
 *
 * @param[in] context Pointer to the CoAP server context
 * @param[in] cookieSecret Pointer to the secret key
 * @param[in] cookieSecretLen Length of the secret key, in bytes
 * @return Error code
 **/

error_t coapServerSetCookieSecret(CoapServerContext *context,
   const uint8_t *cookieSecret, size_t cookieSecretLen)
{
#if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
   //Ensure the parameters are valid
   if(context == NULL || cookieSecret == NULL)
      return ERROR_INVALID_PARAMETER;
   if(cookieSecretLen > COAP_SERVER_MAX_COOKIE_SECRET_SIZE)
      return ERROR_INVALID_PARAMETER;

   //Save the secret key
   osMemcpy(context->cookieSecret, cookieSecret, cookieSecretLen);
   //Save the length of the secret key
   context->cookieSecretLen = cookieSecretLen;

   //Successful processing
   return NO_ERROR;
#else
   //Not implemented
   return ERROR_NOT_IMPLEMENTED;
#endif
}


/**
 * @brief Start CoAP server
 * @param[in] context Pointer to the CoAP server context
 * @return Error code
 **/

error_t coapServerStart(CoapServerContext *context)
{
   error_t error;

   //Make sure the CoAP server context is valid
   if(context == NULL)
      return ERROR_INVALID_PARAMETER;

   //Debug message
   TRACE_INFO("Starting CoAP server...\r\n");

   //Make sure the CoAP server is not already running
   if(context->running)
      return ERROR_ALREADY_RUNNING;

   //Start of exception handling block
   do
   {
      //Open a UDP socket
      context->socket = socketOpenEx(context->netContext, SOCKET_TYPE_DGRAM,
         SOCKET_IP_PROTO_UDP);
      //Failed to open socket?
      if(context->socket == NULL)
      {
         //Report an error
         error = ERROR_OPEN_FAILED;
         break;
      }

      //Force the socket to operate in non-blocking mode
      error = socketSetTimeout(context->socket, 0);
      //Any error to report?
      if(error)
         break;

      //Associate the socket with the relevant interface
      error = socketBindToInterface(context->socket, context->interface);
      //Unable to bind the socket to the desired interface?
      if(error)
         break;

      //The CoAP server listens for datagrams on port 5683
      error = socketBind(context->socket, &IP_ADDR_ANY, context->port);
      //Unable to bind the socket to the desired port?
      if(error)
         break;

      //Any registered callback?
      if(context->udpInitCallback != NULL)
      {
         //Invoke user callback function
         error = context->udpInitCallback(context, context->socket);
         //Any error to report?
         if(error)
            break;
      }

      //Start the CoAP server
      context->stop = FALSE;
      context->running = TRUE;

      //Create a task
      context->taskId = osCreateTask("CoAP Server", (OsTaskCode) coapServerTask,
         context, &context->taskParams);

      //Failed to create task?
      if(context->taskId == OS_INVALID_TASK_ID)
      {
         //Report an error
         error = ERROR_OUT_OF_RESOURCES;
         break;
      }

      //End of exception handling block
   } while(0);

   //Any error to report?
   if(error)
   {
      //Clean up side effects
      context->running = FALSE;

      //Close the UDP socket
      socketClose(context->socket);
      context->socket = NULL;
   }

   //Return status code
   return error;
}


/**
 * @brief Stop CoAP server
 * @param[in] context Pointer to the CoAP server context
 * @return Error code
 **/

error_t coapServerStop(CoapServerContext *context)
{
#if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
   uint_t i;
#endif

   //Make sure the CoAP server context is valid
   if(context == NULL)
      return ERROR_INVALID_PARAMETER;

   //Debug message
   TRACE_INFO("Stopping CoAP server...\r\n");

   //Check whether the CoAP server is running
   if(context->running)
   {
#if (NET_RTOS_SUPPORT == ENABLED)
      //Stop the CoAP server
      context->stop = TRUE;
      //Send a signal to the task to abort any blocking operation
      osSetEvent(&context->event);

      //Wait for the task to terminate
      while(context->running)
      {
         osDelayTask(1);
      }
#endif

#if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
      //Loop through DTLS sessions
      for(i = 0; i < COAP_SERVER_MAX_SESSIONS; i++)
      {
         //Release DTLS session
         coapServerDeleteSession(&context->session[i]);
      }
#endif

      //Close the UDP socket
      socketClose(context->socket);
      context->socket = NULL;
   }

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief CoAP server task
 * @param[in] context Pointer to the CoAP server context
 **/

void coapServerTask(CoapServerContext *context)
{
   error_t error;
   SocketEventDesc eventDesc;

#if (NET_RTOS_SUPPORT == ENABLED)
   //Task prologue
   osEnterTask();

   //Process events
   while(1)
   {
#endif
      //Specify the events the application is interested in
      eventDesc.socket = context->socket;
      eventDesc.eventMask = SOCKET_EVENT_RX_READY;
      eventDesc.eventFlags = 0;

      //Wait for an event
      socketPoll(&eventDesc, 1, &context->event, COAP_SERVER_TICK_INTERVAL);

      //Stop request?
      if(context->stop)
      {
         //Stop CoAP server operation
         context->running = FALSE;
         //Task epilogue
         osExitTask();
         //Kill ourselves
         osDeleteTask(OS_SELF_TASK_ID);
      }

      //Any datagram received?
      if(eventDesc.eventFlags != 0)
      {
         //Receive incoming datagram
         error = socketReceiveEx(context->socket, &context->clientIpAddr,
            &context->clientPort, &context->serverIpAddr, context->buffer,
            COAP_SERVER_BUFFER_SIZE, &context->bufferLen, 0);

         //Check status code
         if(!error)
         {
            //An endpoint must be prepared to receive multicast messages but may
            //ignore them if multicast service discovery is not desired
            if(!ipIsMulticastAddr(&context->serverIpAddr))
            {
   #if (COAP_SERVER_DTLS_SUPPORT == ENABLED)
               //DTLS-secured communication?
               if(context->dtlsInitCallback != NULL)
               {
                  //Demultiplexing of incoming datagrams into separate DTLS sessions
                  error = coapServerDemultiplexSession(context);
               }
               else
   #endif
               {
                  //Process the received CoAP message
                  error = coapServerProcessRequest(context, context->buffer,
                     context->bufferLen);
               }
            }
         }
      }

      //Handle periodic operations
      coapServerTick(context);
#if (NET_RTOS_SUPPORT == ENABLED)
   }
#endif
}


/**
 * @brief Release CoAP server context
 * @param[in] context Pointer to the CoAP server context
 **/

void coapServerDeinit(CoapServerContext *context)
{
   //Make sure the CoAP server context is valid
   if(context != NULL)
   {
      //Free previously allocated resources
      osDeleteEvent(&context->event);

      //Clear CoAP server context
      osMemset(context, 0, sizeof(CoapServerContext));
   }
}

#endif
