/**
 * @file udp.c
 * @brief UDP (User Datagram Protocol)
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL UDP_TRACE_LEVEL

//Dependencies
#include "core/net.h"
#include "core/ip.h"
#include "core/udp.h"
#include "core/socket.h"
#include "core/socket_misc.h"
#include "ipv4/ipv4.h"
#include "ipv4/ipv4_misc.h"
#include "ipv6/ipv6.h"
#include "ipv6/ipv6_misc.h"
#include "mibs/mib2_module.h"
#include "mibs/udp_mib_module.h"
#include "debug.h"

//Check TCP/IP stack configuration
#if (UDP_SUPPORT == ENABLED)

//Table that holds the registered user callbacks
UdpRxCallbackEntry udpCallbackTable[UDP_CALLBACK_TABLE_SIZE];


/**
 * @brief UDP related initialization
 * @param[in] context Pointer to the TCP/IP stack context
 * @return Error code
 **/

error_t udpInit(NetContext *context)
{
   //Reset ephemeral port number
   context->udpDynamicPort = 0;

   //Initialize callback table
   osMemset(udpCallbackTable, 0, sizeof(udpCallbackTable));

   //Successful initialization
   return NO_ERROR;
}


/**
 * @brief Get an ephemeral port number
 * @param[in] context Pointer to the TCP/IP stack context
 * @return Ephemeral port
 **/

uint16_t udpGetDynamicPort(NetContext *context)
{
   uint_t port;

   //Retrieve current port number
   port = context->udpDynamicPort;

   //Invalid port number?
   if(port < SOCKET_EPHEMERAL_PORT_MIN || port > SOCKET_EPHEMERAL_PORT_MAX)
   {
      //Generate a random port number
      port = netGenerateRandRange(context, SOCKET_EPHEMERAL_PORT_MIN,
         SOCKET_EPHEMERAL_PORT_MAX);
   }

   //Next dynamic port to use
   if(port < SOCKET_EPHEMERAL_PORT_MAX)
   {
      //Increment port number
      context->udpDynamicPort = port + 1;
   }
   else
   {
      //Wrap around if necessary
      context->udpDynamicPort = SOCKET_EPHEMERAL_PORT_MIN;
   }

   //Return an ephemeral port number
   return port;
}


/**
 * @brief Incoming UDP datagram processing
 * @param[in] interface Underlying network interface
 * @param[in] pseudoHeader UDP pseudo header
 * @param[in] buffer Multi-part buffer containing the incoming UDP datagram
 * @param[in] offset Offset to the first byte of the UDP header
 * @param[in] ancillary Additional options passed to the stack along with
 *   the packet
 * @return Error code
 **/

error_t udpProcessDatagram(NetInterface *interface,
   const IpPseudoHeader *pseudoHeader, const NetBuffer *buffer, size_t offset,
   const NetRxAncillary *ancillary)
{
   error_t error;
   uint_t i;
   size_t length;
   UdpHeader *header;
   Socket *socket;
   SocketQueueItem *queueItem;
   NetBuffer *p;

   //Retrieve the length of the UDP datagram
   length = netBufferGetLength(buffer) - offset;

   //Ensure the UDP header is valid
   if(length < sizeof(UdpHeader))
   {
      //Number of received UDP datagrams that could not be delivered for
      //reasons other than the lack of an application at the destination port
      MIB2_UDP_INC_COUNTER32(udpInErrors, 1);
      UDP_MIB_INC_COUNTER32(udpInErrors, 1);

      //Report an error
      return ERROR_INVALID_HEADER;
   }

   //Point to the UDP header
   header = netBufferAt(buffer, offset, sizeof(UdpHeader));
   //Sanity check
   if(header == NULL)
      return ERROR_FAILURE;

   //Debug message
   TRACE_INFO("UDP datagram received (%" PRIuSIZE " bytes)...\r\n", length);
   //Dump UDP header contents for debugging purpose
   udpDumpHeader(header);

   //Make sure the length field is correct
   if(ntohs(header->length) < sizeof(UdpHeader) ||
      ntohs(header->length) > length)
   {
      //Number of received UDP datagrams that could not be delivered for
      //reasons other than the lack of an application at the destination port
      MIB2_UDP_INC_COUNTER32(udpInErrors, 1);
      UDP_MIB_INC_COUNTER32(udpInErrors, 1);

      //Report an error
      return ERROR_INVALID_HEADER;
   }

   //Convert the length field from network byte order
   length = ntohs(header->length);

   //When UDP runs over IPv6, the checksum is mandatory
   if(header->checksum != 0x0000 ||
      pseudoHeader->length == sizeof(Ipv6PseudoHeader))
   {
      //Verify UDP checksum
      if(ipCalcUpperLayerChecksumEx(pseudoHeader->data,
         pseudoHeader->length, buffer, offset, length) != 0x0000)
      {
         //Debug message
         TRACE_WARNING("Wrong UDP header checksum!\r\n");

         //Number of received UDP datagrams that could not be delivered for
         //reasons other than the lack of an application at the destination port
         MIB2_UDP_INC_COUNTER32(udpInErrors, 1);
         UDP_MIB_INC_COUNTER32(udpInErrors, 1);

         //Report an error
         return ERROR_WRONG_CHECKSUM;
      }
   }

   //Loop through opened sockets
   for(i = 0; i < SOCKET_MAX_COUNT; i++)
   {
      //Point to the current socket
      socket = &socketTable[i];

      //UDP socket found?
      if(socket->type != SOCKET_TYPE_DGRAM)
         continue;

      //Check whether the socket is bound to a particular interface
      if(socket->interface != NULL && socket->interface != interface)
         continue;

      //Check destination port number
      if(socket->localPort == 0 || socket->localPort != ntohs(header->destPort))
         continue;

      //Source port number filtering
      if(socket->remotePort != 0 && socket->remotePort != ntohs(header->srcPort))
         continue;

#if (IPV4_SUPPORT == ENABLED)
      //IPv4 packet received?
      if(pseudoHeader->length == sizeof(Ipv4PseudoHeader))
      {
         //Check whether the socket is restricted to IPv6 communications only
         if((socket->options & SOCKET_OPTION_IPV6_ONLY) != 0)
            continue;

         //Check whether the destination address is a unicast, broadcast or
         //multicast address
         if(ipv4IsBroadcastAddr(interface, pseudoHeader->ipv4Data.destAddr))
         {
            //Check whether broadcast datagrams are accepted or not
            if((socket->options & SOCKET_OPTION_BROADCAST) == 0)
               continue;
         }
         else if(ipv4IsMulticastAddr(pseudoHeader->ipv4Data.destAddr))
         {
            IpAddr srcAddr;
            IpAddr destAddr;

            //Get source IPv4 address
            srcAddr.length = sizeof(Ipv4Addr);
            srcAddr.ipv4Addr = pseudoHeader->ipv4Data.srcAddr;

            //Get destination IPv4 address
            destAddr.length = sizeof(Ipv4Addr);
            destAddr.ipv4Addr = pseudoHeader->ipv4Data.destAddr;

            //Multicast address filtering
            if(!socketMulticastFilter(socket, &destAddr, &srcAddr))
            {
               continue;
            }
         }
         else
         {
            //Destination IP address filtering
            if(socket->localIpAddr.length != 0)
            {
               //An IPv4 address is expected
               if(socket->localIpAddr.length != sizeof(Ipv4Addr))
                  continue;

               //Filter out non-matching addresses
               if(socket->localIpAddr.ipv4Addr != IPV4_UNSPECIFIED_ADDR &&
                  socket->localIpAddr.ipv4Addr != pseudoHeader->ipv4Data.destAddr)
               {
                  continue;
               }
            }
         }

         //Source IP address filtering
         if(socket->remoteIpAddr.length != 0)
         {
            //An IPv4 address is expected
            if(socket->remoteIpAddr.length != sizeof(Ipv4Addr))
               continue;

            //Filter out non-matching addresses
            if(socket->remoteIpAddr.ipv4Addr != IPV4_UNSPECIFIED_ADDR &&
               socket->remoteIpAddr.ipv4Addr != pseudoHeader->ipv4Data.srcAddr)
            {
               continue;
            }
         }
      }
      else
#endif
#if (IPV6_SUPPORT == ENABLED)
      //IPv6 packet received?
      if(pseudoHeader->length == sizeof(Ipv6PseudoHeader))
      {
         //Check whether the destination address is a unicast or multicast
         //address
         if(ipv6IsMulticastAddr(&pseudoHeader->ipv6Data.destAddr))
         {
            IpAddr srcAddr;
            IpAddr destAddr;

            //Get source IPv6 address
            srcAddr.length = sizeof(Ipv6Addr);
            srcAddr.ipv6Addr = pseudoHeader->ipv6Data.srcAddr;

            //Get destination IPv6 address
            destAddr.length = sizeof(Ipv6Addr);
            destAddr.ipv6Addr = pseudoHeader->ipv6Data.destAddr;

            //Multicast address filtering
            if(!socketMulticastFilter(socket, &destAddr, &srcAddr))
            {
               continue;
            }
         }
         else
         {
            //Destination IP address filtering
            if(socket->localIpAddr.length != 0)
            {
               //An IPv6 address is expected
               if(socket->localIpAddr.length != sizeof(Ipv6Addr))
                  continue;

               //Filter out non-matching addresses
               if(!ipv6CompAddr(&socket->localIpAddr.ipv6Addr,
                  &IPV6_UNSPECIFIED_ADDR) &&
                  !ipv6CompAddr(&socket->localIpAddr.ipv6Addr,
                  &pseudoHeader->ipv6Data.destAddr))
               {
                  continue;
               }
            }
         }

         //Source IP address filtering
         if(socket->remoteIpAddr.length != 0)
         {
            //An IPv6 address is expected
            if(socket->remoteIpAddr.length != sizeof(Ipv6Addr))
               continue;

            //Filter out non-matching addresses
            if(!ipv6CompAddr(&socket->remoteIpAddr.ipv6Addr,
               &IPV6_UNSPECIFIED_ADDR) &&
               !ipv6CompAddr(&socket->remoteIpAddr.ipv6Addr,
               &pseudoHeader->ipv6Data.srcAddr))
            {
               continue;
            }
         }
      }
      else
#endif
      //Invalid packet received?
      {
         //This should never occur...
         continue;
      }

      //The current socket meets all the criteria
      break;
   }

   //Point to the payload
   offset += sizeof(UdpHeader);
   length -= sizeof(UdpHeader);

   //No matching socket found?
   if(i >= SOCKET_MAX_COUNT)
   {
      //Invoke user callback, if any
      error = udpInvokeRxCallback(interface, pseudoHeader, header, buffer,
         offset, ancillary);
      //Return status code
      return error;
   }

   //Empty receive queue?
   if(socket->receiveQueue == NULL)
   {
      //Allocate a memory buffer to hold the data and the associated descriptor
      p = netBufferAlloc(sizeof(SocketQueueItem) + length);

      //Successful memory allocation?
      if(p != NULL)
      {
         //Point to the newly created item
         queueItem = netBufferAt(p, 0, 0);
         queueItem->buffer = p;
         //Add the newly created item to the queue
         socket->receiveQueue = queueItem;
      }
      else
      {
         //Memory allocation failed
         queueItem = NULL;
      }
   }
   else
   {
      //Point to the very first item
      queueItem = socket->receiveQueue;

      //Reach the last item in the receive queue
      for(i = 1; queueItem->next; i++)
      {
         queueItem = queueItem->next;
      }

      //Check whether the receive queue is full
      if(i >= UDP_RX_QUEUE_SIZE)
      {
         //Number of inbound packets which were chosen to be discarded even
         //though no errors had been detected
         NET_IF_STATS_INC_COUNTER32(inDiscards, 1);

         //Report an error
         return ERROR_RECEIVE_QUEUE_FULL;
      }

      //Allocate a memory buffer to hold the data and the associated descriptor
      p = netBufferAlloc(sizeof(SocketQueueItem) + length);

      //Successful memory allocation?
      if(p != NULL)
      {
         //Add the newly created item to the queue
         queueItem->next = netBufferAt(p, 0, 0);
         //Point to the newly created item
         queueItem = queueItem->next;
         queueItem->buffer = p;
      }
      else
      {
         //Memory allocation failed
         queueItem = NULL;
      }
   }

   //Not enough resources to properly handle the packet?
   if(queueItem == NULL)
   {
      //Number of inbound packets which were chosen to be discarded even
      //though no errors had been detected
      NET_IF_STATS_INC_COUNTER32(inDiscards, 1);

      //Report an error
      return ERROR_OUT_OF_MEMORY;
   }

   //Initialize next field
   queueItem->next = NULL;
   //Network interface where the packet was received
   queueItem->interface = interface;
   //Record the source port number
   queueItem->srcPort = ntohs(header->srcPort);

#if (IPV4_SUPPORT == ENABLED)
   //IPv4 remote address?
   if(pseudoHeader->length == sizeof(Ipv4PseudoHeader))
   {
      //Save the source IPv4 address
      queueItem->srcIpAddr.length = sizeof(Ipv4Addr);
      queueItem->srcIpAddr.ipv4Addr = pseudoHeader->ipv4Data.srcAddr;

      //Save the destination IPv4 address
      queueItem->destIpAddr.length = sizeof(Ipv4Addr);
      queueItem->destIpAddr.ipv4Addr = pseudoHeader->ipv4Data.destAddr;
   }
#endif
#if (IPV6_SUPPORT == ENABLED)
   //IPv6 remote address?
   if(pseudoHeader->length == sizeof(Ipv6PseudoHeader))
   {
      //Save the source IPv6 address
      queueItem->srcIpAddr.length = sizeof(Ipv6Addr);
      queueItem->srcIpAddr.ipv6Addr = pseudoHeader->ipv6Data.srcAddr;

      //Save the destination IPv6 address
      queueItem->destIpAddr.length = sizeof(Ipv6Addr);
      queueItem->destIpAddr.ipv6Addr = pseudoHeader->ipv6Data.destAddr;
   }
#endif

   //Offset to the payload
   queueItem->offset = sizeof(SocketQueueItem);
   //Copy the payload
   netBufferCopy(queueItem->buffer, queueItem->offset, buffer, offset, length);

   //Additional options can be passed to the stack along with the packet
   queueItem->ancillary = *ancillary;

   //Notify user that data is available
   udpUpdateEvents(socket);

   //Total number of UDP datagrams delivered to UDP users
   MIB2_UDP_INC_COUNTER32(udpInDatagrams, 1);
   UDP_MIB_INC_COUNTER32(udpInDatagrams, 1);
   UDP_MIB_INC_COUNTER64(udpHCInDatagrams, 1);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Send a UDP datagram
 * @param[in] socket Handle referencing the socket
 * @param[in] message Pointer to the structure describing the datagram
 * @param[in] flags Set of flags that influences the behavior of this function
 * @return Error code
 **/

error_t udpSendDatagram(Socket *socket, const SocketMsg *message, uint_t flags)
{
   error_t error;
   size_t offset;
   NetBuffer *buffer;
   NetInterface *interface;
   NetTxAncillary ancillary;

   //Select the relevant network interface
   if(message->interface != NULL)
   {
      interface = message->interface;
   }
   else
   {
      interface = socket->interface;
   }

   //Allocate a memory buffer to hold the UDP datagram
   buffer = udpAllocBuffer(0, &offset);
   //Failed to allocate buffer?
   if(buffer == NULL)
      return ERROR_OUT_OF_MEMORY;

   //Copy data payload
   error = netBufferAppend(buffer, message->data, message->length);

   //Successful processing?
   if(!error)
   {
      //Additional options can be passed to the stack along with the packet
      ancillary = NET_DEFAULT_TX_ANCILLARY;

      //This option allows UDP checksum generation to be bypassed
      if((socket->options & SOCKET_OPTION_UDP_NO_CHECKSUM) != 0)
      {
         ancillary.noChecksum = TRUE;
      }

      //Set the TTL value to be used
      if(message->ttl != 0)
      {
         ancillary.ttl = message->ttl;
      }
      else if(ipIsMulticastAddr(&message->destIpAddr))
      {
         ancillary.ttl = socket->multicastTtl;
      }
      else
      {
         ancillary.ttl = socket->ttl;
      }

      //Set ToS field
      if(message->tos != 0)
      {
         ancillary.tos = message->tos;
      }
      else
      {
         ancillary.tos = socket->tos;
      }

      //This flag can be used to send IP packets without fragmentation
      if(message->destIpAddr.length == sizeof(Ipv4Addr) &&
         (socket->options & SOCKET_OPTION_IPV4_DONT_FRAG) != 0)
      {
         ancillary.dontFrag = TRUE;
      }
      else if(message->destIpAddr.length == sizeof(Ipv6Addr) &&
         (socket->options & SOCKET_OPTION_IPV6_DONT_FRAG) != 0)
      {
         ancillary.dontFrag = TRUE;
      }
      else
      {
         ancillary.dontFrag = message->dontFrag;
      }

      //This flag tells the stack that the destination is on a locally attached
      //network and not to perform a lookup of the routing table
      if((flags & SOCKET_FLAG_DONT_ROUTE) != 0)
      {
         ancillary.dontRoute = TRUE;
      }

#if (ETH_SUPPORT == ENABLED)
      //Set source and destination MAC addresses
      ancillary.srcMacAddr = message->srcMacAddr;
      ancillary.destMacAddr = message->destMacAddr;
#endif

#if (ETH_VLAN_SUPPORT == ENABLED)
      //Set VLAN PCP and DEI fields
      ancillary.vlanPcp = socket->vlanPcp;
      ancillary.vlanDei = socket->vlanDei;
#endif

#if (ETH_VMAN_SUPPORT == ENABLED)
      //Set VMAN PCP and DEI fields
      ancillary.vmanPcp = socket->vmanPcp;
      ancillary.vmanDei = socket->vmanDei;
#endif

#if (ETH_PORT_TAGGING_SUPPORT == ENABLED)
      //Set switch port identifier
      ancillary.port = message->switchPort;
#endif

#if (ETH_TIMESTAMP_SUPPORT == ENABLED)
      //Unique identifier for hardware time stamping
      ancillary.timestampId = message->timestampId;
#endif

      //Send UDP datagram
      error = udpSendBuffer(socket->netContext, interface, &message->srcIpAddr,
         socket->localPort, &message->destIpAddr, message->destPort, buffer,
         offset, &ancillary);
   }

   //Free previously allocated memory
   netBufferFree(buffer);

   //Return status code
   return error;
}


/**
 * @brief Send a UDP datagram
 * @param[in] context Pointer to the TCP/IP stack context
 * @param[in] interface Underlying network interface
 * @param[in] srcIpAddr Source IP address (optional parameter)
 * @param[in] srcPort Source port
 * @param[in] destIpAddr IP address of the target host
 * @param[in] destPort Target port number
 * @param[in] buffer Multi-part buffer containing the payload
 * @param[in] offset Offset to the first payload byte
 * @param[in] ancillary Additional options passed to the stack along with
 *   the packet
 * @return Error code
 **/

error_t udpSendBuffer(NetContext *context, NetInterface *interface,
   const IpAddr *srcIpAddr, uint16_t srcPort, const IpAddr *destIpAddr,
   uint16_t destPort, NetBuffer *buffer, size_t offset,
   NetTxAncillary *ancillary)
{
   error_t error;
   size_t length;
   UdpHeader *header;
   IpPseudoHeader pseudoHeader;

   //Make room for the UDP header
   offset -= sizeof(UdpHeader);
   //Retrieve the length of the datagram
   length = netBufferGetLength(buffer) - offset;

   //Check the length of the payload
   if(length > UINT16_MAX)
      return ERROR_INVALID_LENGTH;

   //Point to the UDP header
   header = netBufferAt(buffer, offset, sizeof(UdpHeader));
   //Sanity check
   if(header == NULL)
      return ERROR_FAILURE;

   //Format UDP header
   header->srcPort = htons(srcPort);
   header->destPort = htons(destPort);
   header->length = htons(length);
   header->checksum = 0;

#if (IPV4_SUPPORT == ENABLED)
   //Destination address is an IPv4 address?
   if(destIpAddr->length == sizeof(Ipv4Addr))
   {
      //Valid source IP address?
      if(srcIpAddr != NULL && srcIpAddr->length == sizeof(Ipv4Addr))
      {
         //Use default network interface?
         if(interface == NULL)
         {
            interface = netGetDefaultInterface(context);
         }

         //Copy the source IP address
         pseudoHeader.ipv4Data.srcAddr = srcIpAddr->ipv4Addr;
      }
      else
      {
         Ipv4Addr ipAddr;

         //Select the source IPv4 address and the relevant network interface
         //to use when sending data to the specified destination host
         error = ipv4SelectSourceAddr(context, &interface, destIpAddr->ipv4Addr,
            &ipAddr);

         //Check status code
         if(!error)
         {
            //Copy the resulting source IP address
            pseudoHeader.ipv4Data.srcAddr = ipAddr;
         }
         else
         {
            //Handle the special case where the destination address is the
            //broadcast address
            if(destIpAddr->ipv4Addr == IPV4_BROADCAST_ADDR && interface != NULL)
            {
               //Use the unspecified address as source address
               pseudoHeader.ipv4Data.srcAddr = IPV4_UNSPECIFIED_ADDR;
            }
            else
            {
               //Source address selection failed
               return error;
            }
         }
      }

      //Format IPv4 pseudo header
      pseudoHeader.length = sizeof(Ipv4PseudoHeader);
      pseudoHeader.ipv4Data.destAddr = destIpAddr->ipv4Addr;
      pseudoHeader.ipv4Data.reserved = 0;
      pseudoHeader.ipv4Data.protocol = IPV4_PROTOCOL_UDP;
      pseudoHeader.ipv4Data.length = htons(length);

      //UDP checksum is optional for IPv4
      if(!ancillary->noChecksum)
      {
         //Calculate UDP header checksum
         header->checksum = ipCalcUpperLayerChecksumEx(&pseudoHeader.ipv4Data,
            sizeof(Ipv4PseudoHeader), buffer, offset, length);

         //If the computed checksum is zero, it is transmitted as all ones.
         //An all zero transmitted checksum value means that the transmitter
         //generated no checksum (refer to RFC 768)
         if(header->checksum == 0)
         {
            header->checksum = 0xFFFF;
         }
      }
   }
   else
#endif
#if (IPV6_SUPPORT == ENABLED)
   //Destination address is an IPv6 address?
   if(destIpAddr->length == sizeof(Ipv6Addr))
   {
      //Valid source IP address?
      if(srcIpAddr != NULL && srcIpAddr->length == sizeof(Ipv6Addr))
      {
         //Use default network interface?
         if(interface == NULL)
         {
            interface = netGetDefaultInterface(context);
         }

         //Copy the source IP address
         pseudoHeader.ipv6Data.srcAddr = srcIpAddr->ipv6Addr;
      }
      else
      {
         //Select the source IPv6 address and the relevant network interface
         //to use when sending data to the specified destination host
         error = ipv6SelectSourceAddr(context, &interface, &destIpAddr->ipv6Addr,
            &pseudoHeader.ipv6Data.srcAddr);
         //Any error to report?
         if(error)
            return error;
      }

      //Format IPv6 pseudo header
      pseudoHeader.length = sizeof(Ipv6PseudoHeader);
      pseudoHeader.ipv6Data.destAddr = destIpAddr->ipv6Addr;
      pseudoHeader.ipv6Data.length = htonl(length);
      pseudoHeader.ipv6Data.reserved[0] = 0;
      pseudoHeader.ipv6Data.reserved[1] = 0;
      pseudoHeader.ipv6Data.reserved[2] = 0;
      pseudoHeader.ipv6Data.nextHeader = IPV6_UDP_HEADER;

      //Unlike IPv4, when UDP packets are originated by an IPv6 node, the UDP
      //checksum is not optional (refer to RFC 2460, section 8.1)
      header->checksum = ipCalcUpperLayerChecksumEx(&pseudoHeader.ipv6Data,
         sizeof(Ipv6PseudoHeader), buffer, offset, length);

      //If that computation yields a result of zero, it must be changed to hex
      //FFFF for placement in the UDP header
      if(header->checksum == 0)
      {
         header->checksum = 0xFFFF;
      }
   }
   else
#endif
   //Invalid destination address?
   {
      //An internal error has occurred
      return ERROR_FAILURE;
   }

   //Total number of UDP datagrams sent from this entity
   MIB2_UDP_INC_COUNTER32(udpOutDatagrams, 1);
   UDP_MIB_INC_COUNTER32(udpOutDatagrams, 1);
   UDP_MIB_INC_COUNTER64(udpHCOutDatagrams, 1);

   //Debug message
   TRACE_INFO("Sending UDP datagram (%" PRIuSIZE " bytes)\r\n", length);
   //Dump UDP header contents for debugging purpose
   udpDumpHeader(header);

   //Send UDP datagram
   error = ipSendDatagram(interface, &pseudoHeader, buffer, offset, ancillary);

   //Return status code
   return error;
}


/**
 * @brief Receive data from a UDP socket
 * @param[in] socket Handle referencing the socket
 * @param[out] message Received UDP datagram and ancillary data
 * @param[in] flags Set of flags that influences the behavior of this function
 * @return Error code
 **/

error_t udpReceiveDatagram(Socket *socket, SocketMsg *message, uint_t flags)
{
   error_t error;
   SocketQueueItem *queueItem;

   //The SOCKET_FLAG_DONT_WAIT enables non-blocking operation
   if((flags & SOCKET_FLAG_DONT_WAIT) == 0)
   {
      //Check whether the receive queue is empty
      if(socket->receiveQueue == NULL)
      {
         //Set the events the application is interested in
         socket->eventMask = SOCKET_EVENT_RX_READY;

         //Reset the event object
         osResetEvent(&socket->event);

         //Release exclusive access
         netUnlock(socket->netContext);
         //Wait until an event is triggered
         osWaitForEvent(&socket->event, socket->timeout);
         //Get exclusive access
         netLock(socket->netContext);
      }
   }

   //Any datagram received?
   if(socket->receiveQueue != NULL)
   {
      //Point to the first item in the receive queue
      queueItem = socket->receiveQueue;

      //Copy data to user buffer
      message->length = netBufferRead(message->data, queueItem->buffer,
         queueItem->offset, message->size);

      //Network interface where the packet was received
      message->interface = queueItem->interface;
      //Save the source IP address
      message->srcIpAddr = queueItem->srcIpAddr;
      //Save the source port number
      message->srcPort = queueItem->srcPort;
      //Save the destination IP address
      message->destIpAddr = queueItem->destIpAddr;

      //Save TTL value
      message->ttl = queueItem->ancillary.ttl;
      //Save ToS field
      message->tos = queueItem->ancillary.tos;

#if (ETH_SUPPORT == ENABLED)
      //Save source and destination MAC addresses
      message->srcMacAddr = queueItem->ancillary.srcMacAddr;
      message->destMacAddr = queueItem->ancillary.destMacAddr;
#endif

#if (ETH_PORT_TAGGING_SUPPORT == ENABLED)
      //Save switch port identifier
      message->switchPort = queueItem->ancillary.port;
#endif

#if (ETH_TIMESTAMP_SUPPORT == ENABLED)
      //Save captured time stamp
      message->timestamp = queueItem->ancillary.timestamp;
#endif

      //If the SOCKET_FLAG_PEEK flag is set, the data is copied into the
      //buffer but is not removed from the input queue
      if((flags & SOCKET_FLAG_PEEK) == 0)
      {
         //Remove the item from the receive queue
         socket->receiveQueue = queueItem->next;

         //Deallocate memory buffer
         netBufferFree(queueItem->buffer);
      }

      //Update the state of events
      udpUpdateEvents(socket);

      //Successful read operation
      error = NO_ERROR;
   }
   else
   {
      //Total number of data that have been received
      message->length = 0;

      //Report a timeout error
      error = ERROR_TIMEOUT;
   }

   //Return status code
   return error;
}


/**
 * @brief Allocate a buffer to hold a UDP packet
 * @param[in] length Desired payload length
 * @param[out] offset Offset to the first byte of the payload
 * @return The function returns a pointer to the newly allocated
 *   buffer. If the system is out of resources, NULL is returned
 **/

NetBuffer *udpAllocBuffer(size_t length, size_t *offset)
{
   NetBuffer *buffer;

   //Allocate a buffer to hold the UDP header and the payload
   buffer = ipAllocBuffer(length + sizeof(UdpHeader), offset);
   //Failed to allocate buffer?
   if(buffer == NULL)
      return NULL;

   //Offset to the first byte of the payload
   *offset += sizeof(UdpHeader);

   //Return a pointer to the freshly allocated buffer
   return buffer;
}


/**
 * @brief Update UDP related events
 * @param[in] socket Handle referencing the socket
 **/

void udpUpdateEvents(Socket *socket)
{
   //Clear event flags
   socket->eventFlags = 0;

   //The socket is marked as readable if a datagram is pending in the queue
   if(socket->receiveQueue)
      socket->eventFlags |= SOCKET_EVENT_RX_READY;

   //Check whether the socket is bound to a particular network interface
   if(socket->interface != NULL)
   {
      //Handle link up and link down events
      if(socket->interface->linkState)
      {
         socket->eventFlags |= SOCKET_EVENT_LINK_UP;
      }
      else
      {
         socket->eventFlags |= SOCKET_EVENT_LINK_DOWN;
      }
   }

   //Mask unused events
   socket->eventFlags &= socket->eventMask;

   //Any event to signal?
   if(socket->eventFlags)
   {
      //Unblock I/O operations currently in waiting state
      osSetEvent(&socket->event);

      //Set user event to signaled state if necessary
      if(socket->userEvent != NULL)
      {
         osSetEvent(socket->userEvent);
      }
   }
}


/**
 * @brief Register user callback
 * @param[in] interface Underlying network interface
 * @param[in] port UDP port number
 * @param[in] callback Callback function to be called when a datagram is received
 * @param[in] param Callback function parameter (optional)
 * @return Error code
 **/

error_t udpRegisterRxCallback(NetInterface *interface, uint16_t port,
   UdpRxCallback callback, void *param)
{
   uint_t i;
   UdpRxCallbackEntry *entry;

   //Loop through the table
   for(i = 0; i < UDP_CALLBACK_TABLE_SIZE; i++)
   {
      //Point to the current entry
      entry = &udpCallbackTable[i];

      //Check whether the entry is currently in use
      if(entry->callback == NULL)
      {
         //Create a new entry
         entry->interface = interface;
         entry->port = port;
         entry->callback = callback;
         entry->param = param;
         //We are done
         break;
      }
   }

   //Failed to attach the specified user callback?
   if(i >= UDP_CALLBACK_TABLE_SIZE)
      return ERROR_OUT_OF_RESOURCES;

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Unregister user callback
 * @param[in] interface Underlying network interface
 * @param[in] port UDP port number
 * @return Error code
 **/

error_t udpUnregisterRxCallback(NetInterface *interface, uint16_t port)
{
   error_t error;
   uint_t i;
   UdpRxCallbackEntry *entry;

   //Initialize status code
   error = ERROR_FAILURE;

   //Loop through the table
   for(i = 0; i < UDP_CALLBACK_TABLE_SIZE; i++)
   {
      //Point to the current entry
      entry = &udpCallbackTable[i];

      //Check whether the entry is currently in use
      if(entry->callback != NULL)
      {
         //Does the specified port number match the current entry?
         if(entry->port == port && entry->interface == interface)
         {
            //Unregister user callback
            entry->callback = NULL;
            //A matching entry has been found
            error = NO_ERROR;
         }
      }
   }

   //Return status code
   return error;
}


/**
 * @brief Invoke user callback
 * @param[in] interface Underlying network interface
 * @param[in] pseudoHeader UDP pseudo header
 * @param[in] header UDP header
 * @param[in] buffer Multi-part buffer containing the payload
 * @param[in] offset Offset to the first byte of the payload
 * @param[in] ancillary Additional options passed to the stack along with
 *   the packet
 * @return Error code
 **/

error_t udpInvokeRxCallback(NetInterface *interface,
   const IpPseudoHeader *pseudoHeader, const UdpHeader *header,
   const NetBuffer *buffer, size_t offset, const NetRxAncillary *ancillary)
{
   error_t error;
   uint_t i;
   UdpRxCallbackEntry *entry;

   //Initialize status code
   error = ERROR_PORT_UNREACHABLE;

   //Loop through the table
   for(i = 0; i < UDP_CALLBACK_TABLE_SIZE; i++)
   {
      //Point to the current entry
      entry = &udpCallbackTable[i];

      //Check whether the entry is currently in use
      if(entry->callback != NULL)
      {
         //Bound to a particular interface?
         if(entry->interface == NULL || entry->interface == interface)
         {
            //Does the specified port number match the current entry?
            if(entry->port == ntohs(header->destPort))
            {
               //Invoke user callback function
               entry->callback(interface, pseudoHeader, header, buffer, offset,
                  ancillary, entry->param);

               //A matching entry has been found
               error = NO_ERROR;
            }
         }
      }
   }

   //Check status code
   if(error)
   {
      //Total number of received UDP datagrams for which there was
      //no application at the destination port
      MIB2_UDP_INC_COUNTER32(udpNoPorts, 1);
      UDP_MIB_INC_COUNTER32(udpNoPorts, 1);
   }
   else
   {
      //Total number of UDP datagrams delivered to UDP users
      MIB2_UDP_INC_COUNTER32(udpInDatagrams, 1);
      UDP_MIB_INC_COUNTER32(udpInDatagrams, 1);
      UDP_MIB_INC_COUNTER64(udpHCInDatagrams, 1);
   }

   //Return status code
   return error;
}


/**
 * @brief Dump UDP header for debugging purpose
 * @param[in] datagram Pointer to the UDP header
 **/

void udpDumpHeader(const UdpHeader *datagram)
{
   //Dump UDP header contents
   TRACE_DEBUG("  Source Port = %" PRIu16 "\r\n", ntohs(datagram->srcPort));
   TRACE_DEBUG("  Destination Port = %" PRIu16 "\r\n", ntohs(datagram->destPort));
   TRACE_DEBUG("  Length = %" PRIu16 "\r\n", ntohs(datagram->length));
   TRACE_DEBUG("  Checksum = 0x%04" PRIX16 "\r\n", ntohs(datagram->checksum));
}

#endif
