/**
 * @file dp83td510_driver.c
 * @brief DP83TD510 10Base-T1L Ethernet PHY driver
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL NIC_TRACE_LEVEL

//Dependencies
#include "core/net.h"
#include "drivers/phy/dp83td510_driver.h"
#include "debug.h"


/**
 * @brief DP83TD510 Ethernet PHY driver
 **/

const PhyDriver dp83td510PhyDriver =
{
   dp83td510Init,
   dp83td510Tick,
   dp83td510EnableIrq,
   dp83td510DisableIrq,
   dp83td510EventHandler
};


/**
 * @brief DP83TD510 PHY transceiver initialization
 * @param[in] interface Underlying network interface
 * @return Error code
 **/

error_t dp83td510Init(NetInterface *interface)
{
   //Debug message
   TRACE_INFO("Initializing DP83TD510...\r\n");

   //Undefined PHY address?
   if(interface->phyAddr >= 32)
   {
      //Use the default address
      interface->phyAddr = DP83TD510_PHY_ADDR;
   }

   //Initialize serial management interface
   if(interface->smiDriver != NULL)
   {
      interface->smiDriver->init();
   }

   //Initialize external interrupt line driver
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->init();
   }

   //Reset PHY transceiver
   dp83td510WritePhyReg(interface, DP83TD510_MII_REG_0,
      DP83TD510_MII_REG_0_MII_RESET);

   //Wait for the reset to complete
   while(dp83td510ReadPhyReg(interface, DP83TD510_MII_REG_0) &
      DP83TD510_MII_REG_0_MII_RESET)
   {
   }

   //Dump PHY registers for debugging purpose
   dp83td510DumpPhyReg(interface);

   //Perform custom configuration
   dp83td510InitHook(interface);

   //Force the TCP/IP stack to poll the link state at startup
   interface->phyEvent = TRUE;
   //Notify the TCP/IP stack of the event
   osSetEvent(&interface->netContext->event);

   //Successful initialization
   return NO_ERROR;
}


/**
 * @brief DP83TD510 custom configuration
 * @param[in] interface Underlying network interface
 **/

__weak_func void dp83td510InitHook(NetInterface *interface)
{
}


/**
 * @brief DP83TD510 timer handler
 * @param[in] interface Underlying network interface
 **/

void dp83td510Tick(NetInterface *interface)
{
   uint16_t value;
   bool_t linkState;

   //No external interrupt line driver?
   if(interface->extIntDriver == NULL)
   {
      //Read PHY status register
      value = dp83td510ReadPhyReg(interface, DP83TD510_PHY_STS);
      //Retrieve current link state
      linkState = (value & DP83TD510_PHY_STS_LINK_STATUS) ? TRUE : FALSE;

      //Link up event?
      if(linkState && !interface->linkState)
      {
         //Set event flag
         interface->phyEvent = TRUE;
         //Notify the TCP/IP stack of the event
         osSetEvent(&interface->netContext->event);
      }
      //Link down event?
      else if(!linkState && interface->linkState)
      {
         //Set event flag
         interface->phyEvent = TRUE;
         //Notify the TCP/IP stack of the event
         osSetEvent(&interface->netContext->event);
      }
   }
}


/**
 * @brief Enable interrupts
 * @param[in] interface Underlying network interface
 **/

void dp83td510EnableIrq(NetInterface *interface)
{
   //Enable PHY transceiver interrupts
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->enableIrq();
   }
}


/**
 * @brief Disable interrupts
 * @param[in] interface Underlying network interface
 **/

void dp83td510DisableIrq(NetInterface *interface)
{
   //Disable PHY transceiver interrupts
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->disableIrq();
   }
}


/**
 * @brief DP83TD510 event handler
 * @param[in] interface Underlying network interface
 **/

void dp83td510EventHandler(NetInterface *interface)
{
   uint16_t value;

   //Read PHY status register
   value = dp83td510ReadPhyReg(interface, DP83TD510_PHY_STS);

   //Check link state
   if((value & DP83TD510_PHY_STS_LINK_STATUS) != 0)
   {
      //The PHY is only able to operate in 10 Mbps mode
      interface->linkSpeed = NIC_LINK_SPEED_10MBPS;
      interface->duplexMode = NIC_FULL_DUPLEX_MODE;

      //Adjust MAC configuration parameters for proper operation
      interface->nicDriver->updateMacConfig(interface);

      //Update link state
      interface->linkState = TRUE;
   }
   else
   {
      //Update link state
      interface->linkState = FALSE;
   }

   //Process link state change event
   nicNotifyLinkChange(interface);
}


/**
 * @brief Write PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address PHY register address
 * @param[in] data Register value
 **/

void dp83td510WritePhyReg(NetInterface *interface, uint8_t address,
   uint16_t data)
{
   //Write the specified PHY register
   if(interface->smiDriver != NULL)
   {
      interface->smiDriver->writePhyReg(SMI_OPCODE_WRITE,
         interface->phyAddr, address, data);
   }
   else
   {
      interface->nicDriver->writePhyReg(SMI_OPCODE_WRITE,
         interface->phyAddr, address, data);
   }
}


/**
 * @brief Read PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address PHY register address
 * @return Register value
 **/

uint16_t dp83td510ReadPhyReg(NetInterface *interface, uint8_t address)
{
   uint16_t data;

   //Read the specified PHY register
   if(interface->smiDriver != NULL)
   {
      data = interface->smiDriver->readPhyReg(SMI_OPCODE_READ,
         interface->phyAddr, address);
   }
   else
   {
      data = interface->nicDriver->readPhyReg(SMI_OPCODE_READ,
         interface->phyAddr, address);
   }

   //Return the value of the PHY register
   return data;
}


/**
 * @brief Dump PHY registers for debugging purpose
 * @param[in] interface Underlying network interface
 **/

void dp83td510DumpPhyReg(NetInterface *interface)
{
   uint8_t i;

   //Loop through PHY registers
   for(i = 0; i < 32; i++)
   {
      //Display current PHY register
      TRACE_DEBUG("%02" PRIu8 ": 0x%04" PRIX16 "\r\n", i,
         dp83td510ReadPhyReg(interface, i));
   }

   //Terminate with a line feed
   TRACE_DEBUG("\r\n");
}


/**
 * @brief Write MMD register
 * @param[in] interface Underlying network interface
 * @param[in] devAddr Device address
 * @param[in] regAddr Register address
 * @param[in] data MMD register value
 **/

void dp83td510WriteMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr, uint16_t data)
{
   //Select register operation
   dp83td510WritePhyReg(interface, DP83TD510_REGCR,
      DP83TD510_REGCR_CMD_ADDR | (devAddr & DP83TD510_REGCR_DEVAD));

   //Write MMD register address
   dp83td510WritePhyReg(interface, DP83TD510_ADDAR, regAddr);

   //Select data operation
   dp83td510WritePhyReg(interface, DP83TD510_REGCR,
      DP83TD510_REGCR_CMD_DATA_NO_POST_INC | (devAddr & DP83TD510_REGCR_DEVAD));

   //Write the content of the MMD register
   dp83td510WritePhyReg(interface, DP83TD510_ADDAR, data);
}


/**
 * @brief Read MMD register
 * @param[in] interface Underlying network interface
 * @param[in] devAddr Device address
 * @param[in] regAddr Register address
 * @return MMD register value
 **/

uint16_t dp83td510ReadMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr)
{
   //Select register operation
   dp83td510WritePhyReg(interface, DP83TD510_REGCR,
      DP83TD510_REGCR_CMD_ADDR | (devAddr & DP83TD510_REGCR_DEVAD));

   //Write MMD register address
   dp83td510WritePhyReg(interface, DP83TD510_ADDAR, regAddr);

   //Select data operation
   dp83td510WritePhyReg(interface, DP83TD510_REGCR,
      DP83TD510_REGCR_CMD_DATA_NO_POST_INC | (devAddr & DP83TD510_REGCR_DEVAD));

   //Read the content of the MMD register
   return dp83td510ReadPhyReg(interface, DP83TD510_ADDAR);
}
