/**
 * @file igmp_host.h
 * @brief IGMP host
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _IGMP_HOST_H
#define _IGMP_HOST_H

//Dependencies
#include "core/net.h"
#include "igmp/igmp_common.h"

//IGMP host support
#ifndef IGMP_HOST_SUPPORT
   #define IGMP_HOST_SUPPORT ENABLED
#elif (IGMP_HOST_SUPPORT != ENABLED && IGMP_HOST_SUPPORT != DISABLED)
   #error IGMP_HOST_SUPPORT parameter is not valid
#endif

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif


/**
 * @brief Multicast group states
 **/

typedef enum
{
   IGMP_HOST_GROUP_STATE_NON_MEMBER      = 0,
   IGMP_HOST_GROUP_STATE_INIT_MEMBER     = 1,
   IGMP_HOST_GROUP_STATE_DELAYING_MEMBER = 2,
   IGMP_HOST_GROUP_STATE_IDLE_MEMBER     = 3
} IgmpHostGroupState;


/**
 * @brief Source address
 **/

typedef struct
{
   Ipv4Addr addr;          ///<Source address
   uint_t retransmitCount; ///<Retransmission counter
} IgmpHostSrcAddr;


/**
 * @brief Source address list
 **/

typedef struct
{
   uint_t numSources;                                   ///<Number of source address
#if (IPV4_MAX_MULTICAST_SOURCES > 0)
   IgmpHostSrcAddr sources[IPV4_MAX_MULTICAST_SOURCES]; ///<List of source addresses
#endif
} IgmpHostSrcAddrList;


/**
 * @brief Multicast group
 **/

typedef struct
{
   IgmpHostGroupState state;       ///<Multicast group state
   Ipv4Addr addr;                  ///<Multicast group address
   bool_t flag;                    ///<We are the last host to send a report for this group
   uint_t retransmitCount;         ///<Filter mode retransmission counter
   NetTimer timer;                 ///<Report delay timer
   IpFilterMode filterMode;        ///<Filter mode
   Ipv4SrcAddrList filter;         ///<Current-state record
#if (IPV4_MAX_MULTICAST_SOURCES > 0)
   IgmpHostSrcAddrList allow;      ///<ALLOW group record
   IgmpHostSrcAddrList block;      ///<BLOCK group record
   Ipv4SrcAddrList queriedSources; ///<List of sources to be reported
#endif
} IgmpHostGroup;


/**
 * @brief IGMP host context
 **/

typedef struct
{
   NetContext *netContext;                           ///<TCP/IP stack context
   NetInterface *interface;                          ///<Underlying network interface
   IgmpVersion compatibilityMode;                    ///<Host compatibility mode
   NetTimer igmpv1QuerierPresentTimer;               ///<IGMPv1 querier present timer
   NetTimer igmpv2QuerierPresentTimer;               ///<IGMPv2 querier present timer
   NetTimer generalQueryTimer;                       ///<Timer for scheduling responses to general queries
   NetTimer stateChangeReportTimer;                  ///<Retransmission timer for state-change reports
   IgmpHostGroup groups[IPV4_MULTICAST_FILTER_SIZE]; ///<Multicast groups
} IgmpHostContext;


//IGMP host related functions
error_t igmpHostInit(NetInterface *interface);
void igmpHostTick(IgmpHostContext *context);

void igmpHostStateChangeEvent(IgmpHostContext *context, Ipv4Addr groupAddr,
   IpFilterMode newFilterMode, const Ipv4SrcAddrList *newFilter);

void igmpHostLinkChangeEvent(IgmpHostContext *context);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
