/**
 * @file ntp_common.c
 * @brief Definitions common to NTP client and server
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Eval
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL NTP_TRACE_LEVEL

//Dependencies
#include "core/net.h"
#include "ntp/ntp_common.h"
#include "debug.h"

//Check TCP/IP stack configuration
#if (SNTP_CLIENT_SUPPORT == ENABLED || NTS_CLIENT_SUPPORT == ENABLED)


/**
 * @brief Search a NTP packet for a given extension
 * @param[in] extensions Pointer to the NTP extensions
 * @param[in] length Length of the NTP extensions, in bytes
 * @param[in] type Extension type
 * @param[in] index Extension occurrence index
 * @return If the specified extension is found, a pointer to the corresponding
 *   extension is returned. Otherwise NULL pointer is returned
 **/

const NtpExtension *ntpGetExtension(const uint8_t *extensions, size_t length,
   uint16_t type, uint_t index)
{
   uint_t k;
   size_t i;
   size_t n;
   const NtpExtension *extension;

   //Initialize occurrence index
   k = 0;

   //Parse extension fields
   for(i = 0; i < length; i += n)
   {
      //Malformed extension?
      if(length < sizeof(NtpExtension))
         break;

      //Point to the current extension
      extension = (NtpExtension *) (extensions + i);
      //Retrieve the length of the extension
      n = ntohs(extension->length);

      //Malformed extension?
      if(n < sizeof(NtpExtension) || (i + n) > length)
         break;

      //Matching extension type?
      if(ntohs(extension->fieldType) == type)
      {
         //Matching occurrence found?
         if(k++ == index)
         {
            return extension;
         }
      }
   }

   //The specified extension type was not found
   return NULL;
}

#endif
