/**
 * @file header.c
 * @brief Generate the header section of an update image
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#include <stdio.h>
#include "core/crypto.h"
#include "hash/sha256.h"
#include "crc32.h"
#include "header.h"
#include "utils.h"
#include "ImageBuilderConfig.h"
#include "cli.h"
#include "debug.h"

// Global variables
char *input_binary = NULL;
size_t input_binary_size = 0;
char *input_binary_and_signature = NULL;
size_t input_binary_and_signature_size = 0;
char *blockified_padding_and_input_binary = NULL;
size_t blockified_padding_and_input_binary_size = 0;

char *padding_and_input_binary = NULL;
uint32_t padding_and_input_binary_size = 0;

/**
 * @brief Make the image header
 * @param[in] header Pointer to the image header
 * @param cli_options Pointer to CLI parameters
 * @param cipher_info Pointer to Crypto stuff
 * @return Status code
 **/

int headerMake(ImageHeader *header, struct builder_cli_configuration *cli_options, CipherInfo cipher_info) {
   uint32_t headerDataSize;
   uint32_t padding_required;
   char *imgIdx_char;
   char *_firmware_version;

   ImageVersion image_version;

   CheckDataInfo check_data_info = {0};

   char *binary_signature_data;

   // Generate the header version
   uint32_t headerVersion = IMAGE_HEADER_VERSION;
   // Choose header section integrity algorithm
   HashAlgo const *crc32_algo = (HashAlgo *) CRC32_HASH_ALGO;

   // Read the binary file from the disk
   int status = read_file(cli_options->input, &input_binary, &input_binary_size);

   if(status)
   {
      printf("headerMake: failed to open input binary file.\n");
      return EXIT_FAILURE;
   }

   header->binarySize = input_binary_size;

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
   // If runtime signature check is enabled
   if(cli_options->runtime_signature_algo != NULL && cli_options->runtime_signature_key != NULL)
   {

      // Sign the binary
      check_data_info.sign_algo = cli_options->runtime_signature_algo;
      check_data_info.signKey = cli_options->runtime_signature_key;
      check_data_info.signHashAlgo = SHA256_HASH_ALGO;

      sign(&cipher_info, &check_data_info, input_binary, input_binary_size,
         &binary_signature_data, &check_data_info.signKeySize);

      //Append the signature to input binary
      input_binary_and_signature = malloc(input_binary_size + check_data_info.signKeySize);
      memcpy(input_binary_and_signature, input_binary, input_binary_size);
      memcpy(input_binary_and_signature + input_binary_size, binary_signature_data,
         check_data_info.signKeySize);

      input_binary_and_signature_size = input_binary_size + check_data_info.signKeySize;

   }
#endif

   //Computing padding according given VTOR align value
   if(cli_options->vtor_align)
   {
      char *pad;
      padding_required = strtol(cli_options->vtor_align, &pad, 10);
   }
   else
   {
      padding_required = 0;
   }

   if((padding_required == 0) || (padding_required == sizeof(ImageHeader)))
   {
      //No padding
      header->dataPadding = 0;
   }
   else
   {
      header->dataPadding = padding_required - (sizeof(ImageHeader) % padding_required);
      //Debug message
      TRACE_DEBUG("Applying %d bytes padding between header and binary...\r\n", header->dataPadding);
   }

   if(cli_options->runtime_signature_algo != NULL && cli_options->runtime_signature_key != NULL)
   {
      padding_and_input_binary_size = input_binary_size + header->dataPadding + check_data_info.signKeySize;
   }
   else
   {
      padding_and_input_binary_size = input_binary_size + header->dataPadding;
   }
   // Make a buffer big enough to keep the padding (if supplied, otherwise 0) and the binary file (and appended signature)
   padding_and_input_binary = malloc(padding_and_input_binary_size);
   // Set the buffer to zero and copy the padding and the input binary (respectively) to the buffer
   memset(padding_and_input_binary, 0, padding_and_input_binary_size);

   if(cli_options->runtime_signature_algo != NULL && cli_options->runtime_signature_key != NULL)
   {
      memcpy(padding_and_input_binary + header->dataPadding, input_binary_and_signature,
         input_binary_and_signature_size);

      // Calculate the size of the data (padding + binary size)
      headerDataSize = padding_and_input_binary_size;

   }
   else
   {
      memcpy(padding_and_input_binary + header->dataPadding, input_binary, input_binary_size);

      // Calculate the size of the data (padding + binary size)
      headerDataSize = input_binary_size + header->dataPadding;
   }


   // If the image should be encrypted, it must be further divided into block of 16-bytes each
   // this is the size of data an algorithm like AES-CBC expects to work on.
   if(cli_options->encryption_algo != NULL && cli_options->encryption_key != NULL)
   {
      status = blockify(16, padding_and_input_binary, padding_and_input_binary_size,
         &blockified_padding_and_input_binary, &blockified_padding_and_input_binary_size);

      if(status)
      {
         printf("headerMake: failed to blockify input binary file.\n");
         return EXIT_FAILURE;
      }

      headerDataSize = blockified_padding_and_input_binary_size;
   }

   // Fill-in the rest of the fields of header
   header->dataSize = headerDataSize;
   header->headVers = headerVersion;

   if(!cli_options->firmware_index)
      cli_options->firmware_index = "0";   // cannot be NULL

   header->imgIndex = strtol(cli_options->firmware_index, &imgIdx_char, 10);

   //Parse received firmware "string" version
   image_version.major = (uint8_t)strtol(cli_options->firmware_version, &_firmware_version, 10);
   image_version.minor = (uint8_t)strtol(_firmware_version + 1, &_firmware_version, 10);
   image_version.revision = (uint16_t)strtol(_firmware_version + 1, &_firmware_version, 10);
   image_version.buildNum = (uint32_t)strtoul(_firmware_version + 1, &_firmware_version, 16);

   TRACE_DEBUG("Major: %u\n", image_version.major);
   TRACE_DEBUG("Minor: %u\n", image_version.minor);
   TRACE_DEBUG("Revision: %u\n", image_version.revision);
   TRACE_DEBUG("BuildNum: 0x%X (%u)\n", image_version.buildNum, image_version.buildNum);

   header->dataVers = image_version;
   header->imgTime = time(NULL);

   memset(header->reserved, 0, sizeof(header->reserved));

   // Calculate the CRC of the header
   crc32_algo->compute(header, sizeof(ImageHeader) - CRC32_DIGEST_SIZE, header->headCrc);

   return EXIT_SUCCESS;
}
