/**
 * @file utils.c
 * @brief Utility functions used throughout the application
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * This software is provided in source form for a short-term evaluation only. The
 * evaluation license expires 90 days after the date you first download the software.
 *
 * If you plan to use this software in a commercial product, you are required to
 * purchase a commercial license from Oryx Embedded SARL.
 *
 * After the 90-day evaluation period, you agree to either purchase a commercial
 * license or delete all copies of this software. If you wish to extend the
 * evaluation period, you must contact sales@oryx-embedded.com.
 *
 * This evaluation software is provided "as is" without warranty of any kind.
 * Technical support is available as an option during the evaluation period.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifdef IS_WINDOWS
#define _CRTDBG_MAP_ALLOC
#define _WINERROR_
#include <Winsock2.h>
#endif

#include <stdio.h>

#ifdef IS_LINUX
#include <unistd.h>
#include <sys/random.h>
#endif

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
#include "ecc/ec.h"
#include "ecc/ecdsa.h"
#include "ecc/ec_curves.h"
#include "pkc/rsa.h"
#include "pkix/pem_import.h"
#include "pkix/pem_key_import.h"
#include "rng/hmac_drbg.h"
#endif

#include "main.h"
#include "header.h"
#include "body.h"
#include "utils.h"
#include "cli.h"
#include "debug.h"

#ifdef IS_WINDOWS
#include "wincrypt.h"
#endif

/**
 * @brief Read a file
 * @param[in] file_path Path to the file to be read
 * @param[in] file_contents Buffer to store content of the file read
 * @param[in] file_size Pointer to the file size
 * @return Status code
 **/
int read_file(const char *file_path, char **file_contents, size_t *file_size)
{
   FILE *fh = NULL;
   size_t fs = 0;

   if(file_path == NULL)
   {
      TRACE_ERROR("read_file: Error. Missing file path.\r\n");
      return EXIT_FAILURE;
   }

   // Open input file
   fh = fopen(file_path, "rb");

   // Failed to open input file?
   if(fh == NULL)
   {
      // User message
      TRACE_ERROR("read_file: Error. Cannot open %s!\r\n", file_path);
      // Report an error
      return EXIT_FAILURE;
   }

   // Retrieve the length of the file
   fseek(fh, 0, SEEK_END);
   fs = ftell(fh);
   fseek(fh, 0, SEEK_SET);

   *file_contents = (char *)malloc(fs);

   if(*file_contents == NULL)
   {
      // User message
      TRACE_ERROR("read_file: Error. Failed to allocate memory for the input file!\r\n");

      // Cleanup side effects
      fclose(fh);

      // Report an error
      return EXIT_FAILURE;
   }

   // Read the contents of the file
   fread(*file_contents, fs, 1, fh);

   // Copy the file size to the input parameter
   *file_size = fs;

   // Close input file
   fclose(fh);

   return EXIT_SUCCESS;
}

/**
 * @brief Slice a given buffer in to a blocks of a given size
 * @param[in] blockSize Size of the blocks
 * @param[in] input Input buffer
 * @param[in] inputSize Size of the input buffer
 * @param[in] output Output buffer
 * @param[in] outputSize Size of the output buffer
 * @return Status code
 **/
int blockify(size_t blockSize, char *input, size_t inputSize, char **output, size_t *outputSize)
{
   if(input == NULL || output == NULL)
   {
      return EXIT_FAILURE;
   }
   // is the binary size a multiple of a given block size?
   if(blockSize == 0 || inputSize % blockSize == 0)
   {
      // The block size is zero or input size is already the multiple of given block size
      // We can use directly the input data buffer as it is
      *outputSize = inputSize;
      *output = input;
   }
   else
   {
      // Binary block size must be a multiple of the given block size
      *outputSize = inputSize + blockSize - (inputSize % blockSize);

      // Allocate memory for the binary
      *output = malloc(*outputSize);

      // Is allocation failed
      if(*output == NULL)
      {
         TRACE_ERROR("Blockify: Memory allocation failed.\n");
         return EXIT_FAILURE;
      }
      // Initialize memory
      memset(*output, 0, *outputSize);
      // Copy binary application
      memcpy(*output, input, inputSize);

      free(input);
   }

   return EXIT_SUCCESS;
}

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
/**
 * @brief Initialize platform specific random contexts for crypto operations
 * @param[in] cipherInfo Crypto related information
 * @return Status code
 **/
int init_crypto(CipherInfo *cipherInfo)
{
   error_t error = 0;
   size_t randSeedSize = 0;

#ifdef IS_WINDOWS
   HCRYPTPROV hProv;
   cipherInfo->prngAlgo = (PrngAlgo *)HMAC_DRBG_PRNG_ALGO;

   if(CryptAcquireContext(&hProv,
      NULL,
      NULL,
      PROV_RSA_FULL,
      CRYPT_VERIFYCONTEXT))
   {
      if(!CryptGenRandom(hProv, SEED_LENGTH, cipherInfo->seed))
      {
         TRACE_ERROR("Error during IV generation (CryptGenRandom Error).\n");
         return ERROR_FAILURE;
      }

      CryptReleaseContext(hProv, 0);

      // PRNG initialization
      error = hmacDrbgInit(cipherInfo->hmacDrbgContext, SHA256_HASH_ALGO);
      // Any error to report?
      if(error)
      {
         // Debug message
         TRACE_ERROR("Error : PRNG initialization failed (%d)\r\n", error);
         // Exit immediately
         return ERROR_FAILURE;
      }

      // Properly seed the PRNG
      error = hmacDrbgSeed(cipherInfo->hmacDrbgContext, cipherInfo->seed, sizeof(cipherInfo->seed));      // Any error to report?
      if(error)
      {
         // Debug message
         TRACE_ERROR("Error : Failed to seed PRNG (%d)\r\n", error);
         // Exit immediately
         return error;
      }
   }
   else
   {
      TRACE_ERROR("Error acquiring CryptContext.\r\n");
      return ERROR_FAILURE;
   }

   return NO_ERROR;

#endif
#ifdef IS_LINUX
   // Generate a CSPRNG seed
   // https://man7.org/linux/man-pages/man2/getrandom.2.html
   // getrandom() was introduced in version 3.17 of the Linux kernel.
   randSeedSize = getrandom(cipherInfo->seed, SEED_LENGTH, GRND_RANDOM);
   if(randSeedSize != SEED_LENGTH)
   {
      // Debug message
      TRACE_ERROR("init_crypto : Error. CSPRNG Seed failed (%d)\r\n", error);
      return ERROR_FAILURE;
   }

   // PRNG initialization
   error = hmacDrbgInit(cipherInfo->hmacDrbgContext, SHA256_HASH_ALGO);
   // Any error to report?
   if(error)
   {
      // Debug message
      TRACE_ERROR("init_crypto : Error. CSPRNG initialization failed (%d)\r\n", error);
      // Exit immediately
      return ERROR_FAILURE;
   }

   // Properly seed the PRNG
   error = hmacDrbgSeed(cipherInfo->hmacDrbgContext, (uint8_t *)cipherInfo->seed, SEED_LENGTH);
   // Any error to report?
   if(error)
   {
      // Debug message
      TRACE_ERROR("init_crypto : Error. Failed to seed CSPRNG (%d)\r\n", error);
      // Exit immediately
      return error;
   }

   return NO_ERROR;
#endif
}
#endif
/**
 * @brief Generate an initialization vector with random data
 * @param[in] buffer Buffer containing initialization vector
 * @param[in] length Buffer length
 * @return Status code
 **/
void seedInitVector(uint8_t *buffer, size_t length) {
#ifdef IS_LINUX
   getrandom(buffer, length, GRND_RANDOM);
#endif
#ifdef IS_WINDOWS
   HCRYPTPROV hProv;
   CryptAcquireContext(&hProv, NULL, NULL, PROV_RSA_FULL, CRYPT_VERIFYCONTEXT);
   CryptGenRandom(hProv, length, buffer);
   CryptReleaseContext(hProv, 0);
#endif
}

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
/**
 * @brief Generic function to encrypt a given data buffer using AES-CBC
 * @param[in] plainData plain-text buffer
 * @param[in] plainDataSize plain-text buffer length
 * @param[in] cipherData cipher-text buffer
 * @param[in] cipherInfo Crypto related information
 * @return Status code
 **/
int encrypt(char *plainData, size_t plainDataSize, char *cipherData, CipherInfo cipherInfo)
{
   error_t status;
   char context[MAX_CIPHER_CONTEXT_SIZE];
   char iv_copy[16];

   if(plainData == NULL || cipherData == NULL)
   {
      TRACE_ERROR("encrypt: input data invalid.\n");
      return EXIT_FAILURE;
   }

   if(cipherInfo.iv == NULL || cipherInfo.cipherKey == NULL)
   {
      TRACE_ERROR("encrypt: invalid cipher info.\n");
      return EXIT_FAILURE;
   }

   // Initialize AES-CBC algorithm context
   status = AES_CIPHER_ALGO->init(context, cipherInfo.cipherKey, cipherInfo.cipherKeySize);

   if(status)
   {
      TRACE_ERROR("encrypt: AES-CBC initialization failed.\n");
      return EXIT_FAILURE;
   }

   // Encrypt
   memcpy(iv_copy, cipherInfo.iv, cipherInfo.ivSize);
   status = cbcEncrypt(AES_CIPHER_ALGO, context, iv_copy, plainData, cipherData, plainDataSize);

   if(status)
   {
      TRACE_ERROR("encrypt: AES-CBC encryption failed.\n");
      return EXIT_FAILURE;
   }

   return EXIT_SUCCESS;
}

/**
 * @brief Generic function to encrypt a given data buffer using AES-CBC
 * @param[in] cipherInfo Crypto related information for encryption operations
 * @param[in] checkDataInfo Crypto related information for image verification operations
 * @param[in] data Data buffer to be verified
 * @param[in] dataLen Verification data buffer length
 * @param[in] signData Signature buffer resulting from a signature operation
 * @param[in] signDataLen Signature buffer length
 * @return Status code
 **/
int sign(CipherInfo *cipherInfo, CheckDataInfo *checkDataInfo, char *data, size_t dataLen, char **signData, size_t *signDataLen)
{
   // TODO: make sure free's are performed even an error is returned
   error_t error;
   EcPrivateKey ecPrivateKey;
   EcdsaSignature ecdsaSignature;
   char ecdsaSignDigest[MAX_HASH_DIGEST_SIZE];
   RsaPrivateKey rsaPrivateKey;
   char rsaSignDigest[MAX_HASH_DIGEST_SIZE];
   uint8_t rawSignature[64];

   char signature[1024];  // TODO: make this a macro. Also, why 1024?
   size_t signatureLen;
   char *privateKey = NULL;
   size_t privateKeySize = 0;

   error = read_file(checkDataInfo->signKey, &privateKey, &privateKeySize);
   if(error)
   {
      TRACE_ERROR("sign: error opening private key.\n");
      return EXIT_FAILURE;
   }

   if(checkDataInfo->sign_algo == NULL || checkDataInfo->signKey == NULL)
   {
      TRACE_ERROR("sign: signature algorithm or signature key not found.\n");
      return EXIT_FAILURE;
   }

   if(strcasecmp(checkDataInfo->sign_algo, "ecdsa-sha256") == 0)
   {
      // Compute Hash value
      checkDataInfo->signHashAlgo->compute(data, dataLen, (uint8_t *)ecdsaSignDigest);
      // Initialize ECDSA signature
      ecdsaInitSignature(&ecdsaSignature);
      // Initialize EC private keys
      ecInitPrivateKey(&ecPrivateKey);
      //Import the EC private key
      error = pemImportEcPrivateKey(&ecPrivateKey, privateKey, privateKeySize, NULL);
      if(error)
      {
         TRACE_ERROR("sign: error importing PEM private key.\n");
         return EXIT_FAILURE;
      }

      // Generate ECDSA signature (R,S)
      error = ecdsaGenerateSignature(cipherInfo->prngAlgo, cipherInfo->hmacDrbgContext, &ecPrivateKey,
         (uint8_t *)ecdsaSignDigest, checkDataInfo->signHashAlgo->digestSize,
         &ecdsaSignature);

      if(error)
      {
         TRACE_ERROR("sign: error generating ECDSA signature.\n");
         return EXIT_FAILURE;
      }

      error = ecdsaExportSignature(&ecdsaSignature, rawSignature, &signatureLen, ECDSA_SIGNATURE_FORMAT_RAW);

      if(error)
      {
         TRACE_ERROR("sign: error exporting ECDSA signature.\n");
         return EXIT_FAILURE;
      }

      *signDataLen = signatureLen;
      *signData = malloc(signatureLen);

      memcpy(*signData, rawSignature, signatureLen);

      // release previously allocated resources
      ecdsaFreeSignature(&ecdsaSignature);
      ecFreePrivateKey(&ecPrivateKey);
   }
   else if(strcasecmp(checkDataInfo->sign_algo, "rsa-sha256") == 0)
   {
      rsaInitPrivateKey(&rsaPrivateKey);
      error = pemImportRsaPrivateKey(&rsaPrivateKey, privateKey, privateKeySize, NULL);
      if(error)
      {
         TRACE_ERROR("sign: failed to import PEM RSA private key.\n");
         return EXIT_FAILURE;
      }

      error = checkDataInfo->signHashAlgo->compute(data, dataLen, (uint8_t *)rsaSignDigest);

      if(error)
      {
         TRACE_ERROR("sign: error computing RSA signature hash.\n");
         return EXIT_FAILURE;
      }

      error = rsassaPkcs1v15Sign(&rsaPrivateKey, checkDataInfo->signHashAlgo,
         (const uint8_t *)rsaSignDigest, (uint8_t *)&signature, &signatureLen);

      if(error)
      {
         TRACE_ERROR("sign: error generating RSA + SHA256 signature.\n");
         return EXIT_FAILURE;
      }

      // allocate memory for the signature
      *signDataLen = signatureLen;
      *signData = malloc(*signDataLen);

      // Save signature
      memcpy(*signData, signature, signatureLen);

      // free RSA private key
      rsaFreePrivateKey(&rsaPrivateKey);
   }
   else
   {
      TRACE_ERROR("sign: Unknown signature algorithm.\n");
      return EXIT_FAILURE;
   }

   return EXIT_SUCCESS;
}
#endif

/**
 * @brief Write the update image to a file
 * @param[in] image Pointer containing the update image to be rewritten to disk
 * @param[in] cipherInfo Crypto related information
 * @param[in] output_file_path Path to write the image
 * @return Status code
 **/
int write_image_to_file(UpdateImage *image, CipherInfo *cipherInfo, const char *output_file_path)
{
   char offset_buffer[1];
   FILE *fh;

   TRACE_INFO("Generating update image...\n");
   fh = fopen(output_file_path, "wb+");
   if(fh == NULL)
   {
      TRACE_ERROR("write_to_image_file: Error. cannot open output file.\n");
      return EXIT_FAILURE;
   }

   fwrite(image->header, 1, sizeof(ImageHeader), fh);

   if(cipherInfo->iv != 0)
   {
      fwrite(cipherInfo->iv, 1, cipherInfo->ivSize, fh);
      fwrite(image->body->binary, 1, image->body->binarySize, fh);
   }
   else
   {
      fwrite(image->body->binary, 1, image->body->binarySize, fh);
   }

   fwrite(image->body->checkData, 1, image->body->checkDataSize, fh);

   fclose(fh);

   TRACE_INFO("Done.\n");

   return EXIT_SUCCESS;
}

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
/**
 * @brief Generate a keypair
 * @param[in] cli_options Structure containing CLI options
 * @param[in] cli_options Crypto options
 * @return Status code
 **/
int generate_key_pair(struct builder_cli_configuration *cli_options, CipherInfo *cipher_info) {

   if(strcasecmp(cli_options->keygen_keytype, "ecdsa") == 0)
   {
      error_t error;
      size_t n;
      EcPrivateKey privateKey;
      EcPublicKey publicKey;

      //Initialize EC key pair
      ecInitPrivateKey(&privateKey);
      ecInitPublicKey(&publicKey);

      char_t generatedPubKey[256];
      char_t generatedPrivKey[256];

      //Debug message
      TRACE_INFO("Generating ECDSA key pair...\r\n");

      //Start of exception handling block
      do
      {
         //Generate an EC key pair
         error = ecGenerateKeyPair(HMAC_DRBG_PRNG_ALGO, cipher_info->hmacDrbgContext,
            SECP256R1_CURVE, &privateKey, &publicKey);
         //Any error to report?
         if(error)
            break;

         //Export the EC private key to PEM format
         error = pemExportEcPrivateKey(&privateKey, generatedPrivKey, &n,
            PEM_PRIVATE_KEY_FORMAT_DEFAULT);
         //Any error to report?
         if(error)
            break;

         //Dump private key (for debugging purpose only)
         generatedPrivKey[n] = '\0';
         TRACE_DEBUG("Private key (%" PRIuSIZE " bytes):\r\n", n);
         TRACE_DEBUG("%s\r\n", generatedPrivKey);

         // Write the private key to disk
         FILE *pk_file = fopen(cli_options->keygen_privatekey, "wb");     // "wb" = write binary
         if(pk_file == NULL)
         {
            perror("fopen");
            return EXIT_FAILURE;
         }

         size_t written = fwrite(generatedPrivKey, 1, n, pk_file);
         if(written != n)
         {
            perror("fwrite");
            fclose(pk_file);
            return EXIT_FAILURE;
         }

         fclose(pk_file);

         //Export the EC public key to PEM format
         // generatedPubKey = NULL for size of pem public key without generating/writing
         error = pemExportEcPublicKey(&publicKey, generatedPubKey, &n,
            PEM_PUBLIC_KEY_FORMAT_DEFAULT);
         //Any error to report?
         if(error)
            break;

         //Dump public key
         generatedPubKey[n] = '\0';
         TRACE_DEBUG("Public key (%" PRIuSIZE " bytes):\r\n", n);
         TRACE_DEBUG("%s\r\n", generatedPubKey);

         // Write the file to disk
         FILE *pub_file = fopen(cli_options->keygen_publickey, "wb");     // "wb" = write binary
         if(pub_file == NULL)
         {
            perror("fopen");
            return EXIT_FAILURE;
         }

         written = fwrite(generatedPubKey, 1, n, pub_file);
         if(written != n)
         {
            perror("fwrite");
            fclose(pub_file);
            return EXIT_FAILURE;
         }

         fclose(pub_file);

         //End of exception handling block
      } while(0);

      //Release EC key pair
      ecFreePrivateKey(&privateKey);
      ecFreePublicKey(&publicKey);

      //Return status code
      return error;
   }

   if(strcasecmp(cli_options->keygen_keytype, "rsa") == 0)
   {
      error_t error;
      size_t n;
      RsaPrivateKey privateKey;
      RsaPublicKey publicKey;

      //Initialize RSA key pair
      rsaInitPrivateKey(&privateKey);
      rsaInitPublicKey(&publicKey);

      char_t generatedPubKey[512];
      char_t generatedPrivKey[2048];

      //Debug message
      TRACE_INFO("Generating RSA key pair...\r\n");

      //Start of exception handling block
      do
      {
         //Generate RSA key pair
         error = rsaGenerateKeyPair(HMAC_DRBG_PRNG_ALGO, cipher_info->hmacDrbgContext, 2048, 65537,
            &privateKey, &publicKey);

         //Any error to report?
         if(error)
            break;

         //Export the RSA private key to PEM format
         error = pemExportRsaPrivateKey(&privateKey, generatedPrivKey, &n,
            PEM_PRIVATE_KEY_FORMAT_DEFAULT);
         //Any error to report?
         if(error)
            break;

         //Dump private key (for debugging purpose only)
         generatedPrivKey[n] = '\0';
         TRACE_DEBUG("PEM private key (%" PRIuSIZE " bytes):\r\n", n);
         TRACE_DEBUG("%s\r\n", generatedPrivKey);

         // Write the private key to disk
         FILE *pk_file = fopen(cli_options->keygen_privatekey, "wb");     // "wb" = write binary
         if(pk_file == NULL)
         {
            perror("fopen");
            return EXIT_FAILURE;
         }

         size_t written = fwrite(generatedPrivKey, 1, n, pk_file);
         if(written != n)
         {
            perror("fwrite");
            fclose(pk_file);
            return EXIT_FAILURE;
         }

         fclose(pk_file);

         //Export the RSA public key to PEM format
         // GeneratedPubKey = NULL => key size
         // Make a test of the size against the buffer size
         // Make sure n < sizeof(generatedPubKey) => if true, buffer is OK
         error = pemExportRsaPublicKey(&publicKey, generatedPubKey, &n,
            PEM_PUBLIC_KEY_FORMAT_DEFAULT);
         //Any error to report?
         if(error)
            break;

         //Dump public key
         generatedPubKey[n] = '\0';
         TRACE_DEBUG("Public key (%" PRIuSIZE " bytes):\r\n", n);
         TRACE_DEBUG("%s\r\n", generatedPubKey);

         // Write the file to disk
         FILE *pub_file = fopen(cli_options->keygen_publickey, "wb");     // "wb" = write binary
         if(pub_file == NULL)
         {
            perror("fopen");
            return EXIT_FAILURE;
         }

         written = fwrite(generatedPubKey, 1, n, pub_file);
         if(written != n)
         {
            perror("fwrite");
            fclose(pub_file);
            return EXIT_FAILURE;
         }

         fclose(pub_file);

         //End of exception handling block
      } while(0);

      //Release RSA key pair
      rsaFreePrivateKey(&privateKey);
      rsaFreePublicKey(&publicKey);

      //Return status code
      return error;
   }

   return ERROR_ABORTED;
}
#endif
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////


void dump_buffer(void *buffer, size_t buffer_size)
{
   for(int i = 0; i < buffer_size; ++i)
   {
      printf("0x%02hhx\t", ((char *)buffer)[i]);
   }

   printf("\n\n");
}


void dump_buffer_2(void *buffer, size_t buffer_size, size_t columSize)
{
   for(int i = 0; i < buffer_size; ++i)
   {
      if(i != 0 & (i % columSize == 0))
         printf("\r\n");
      printf("%02hhx ", ((char *)buffer)[i]);
   }
   printf("\n\n");
}


void dumpHeader(ImageHeader *header)
{
   printf("\r\n");
   printf("Image header:\r\n");
   printf("-------------\r\n");
   printf("- headVers = %d.%d.%d\r\n", (header->headVers >> 16) & 0xFF,
      (header->headVers >> 8) & 0xFF, header->headVers & 0xFF);
   printf("- imgIndex   = %d\r\n", header->imgIndex);
   printf("- type          = %d (0 = APP, 1 = OTHER)\r\n", header->imgType);
   printf("- data offset   = 0x%X -> %d bytes\r\n", header->dataPadding, header->dataPadding);
   printf("- dataSize       = 0x%X -> %d bytes\r\n", header->dataSize, header->dataSize);
   printf("- dataVers    = %d.%d.%d (rev.%d)\r\n", (header->dataVers.major) & 0xFF,
      (header->dataVers.minor) & 0xFF, header->dataVers.revision & 0xFFFF, header->dataVers.buildNum & 0xFFFFFFFF);
   printf("- time          = 0x%X%X\r\n", (uint32_t)((header->imgTime >> 32) & 0xFFFFFFFF),
      (uint32_t)(header->imgTime & 0xFFFFFFFF));
   printf("- reserved (%zu bytes) =\r\n", sizeof(header->reserved));  // set reserved size to get a 256 bytes header
   dump_buffer(header->reserved, sizeof(header->reserved));
   printf("- headCrc   (%zu bytes)   =\r\n", sizeof(header->headCrc));
   dump_buffer(header->headCrc, sizeof(header->headCrc));
   printf("\r\n");
}

void dumpBody(ImageBody *body)
{
   printf("\r\n");
   printf("Image body:\r\n");
   printf("-----------\r\n");
   printf("binary (%zu bytes) =\r\n", body->binarySize);
   dump_buffer(body->binary, body->binarySize);
   printf("body check data size = %zu bytes\n", body->checkDataSize);
   printf("body check data = \r\n");
   dump_buffer(body->checkData, body->checkDataSize);

   printf("\r\n");
}

void dumpFooter(char *check_data, size_t check_data_size)
{
   printf("\r\n");
   printf("Image footer:\r\n");
   printf("-------------\r\n");
   printf("Binary check data =\r\n");
   dump_buffer(check_data, check_data_size);
   printf("\r\n");
}

void dump_cleartext_image(void *buffer, size_t buffer_size) {
   uint8_t *p = (uint8_t *)buffer;
   ImageHeader *header = (ImageHeader *)p;
   printf("\r\n");
   dumpHeader(header);
   printf("\r\n");

   p += sizeof(ImageHeader);

   printf("Padding & Body:\r\n");
   printf("--------\r\n");
   dump_buffer(p, header->dataSize);

   printf("\r\n");

   printf("Check Field: \r\n");
   printf("------------\r\n");

   p += header->dataSize;

   dump_buffer(p, CHECK_DATA_LENGTH);
}

void dump_ciphertext_image(void *buffer, size_t buffer_size) {
   uint8_t *p = buffer;
   ImageHeader *header = (ImageHeader *)p;

   printf("\r\n");
   dumpHeader(header);
   printf("\r\n");

   p += sizeof(ImageHeader);

   printf("Initialization Vector: \r\n");
   dump_buffer(p, INIT_VECTOR_LENGTH);

   p += INIT_VECTOR_LENGTH;

   printf("CRC Magic: \r\n");
   dump_buffer(p, CRC32_DIGEST_SIZE);

   p += CRC32_DIGEST_SIZE;
   p += 12;  // Take into account the block size

   printf("Padding & Body:\r\n");
   printf("--------\r\n");
   dump_buffer(p, header->dataSize);

   printf("\r\n");

   printf("Check Field: \r\n");
   printf("------------\r\n");

   p += header->dataSize;

   dump_buffer(p, CHECK_DATA_LENGTH);
}

int is_hex(const char *str) {
   // Check if the string consists of hexadecimal characters
   while(*str)
   {
      if(!isxdigit(*str))
         return 0;
      str++;
   }
   return 1;
}

int hexCharToDecimal(char c) {
   if(c >= '0' && c <= '9')
   {
      return c - '0';
   }
   else if(c >= 'a' && c <= 'f')
   {
      return 10 + c - 'a';
   }
   else if(c >= 'A' && c <= 'F')
   {
      return 10 + c - 'A';
   }
   else
   {
      return -1;   // Invalid hex character
   }
}

// Function to convert a hex string to a byte array
int hex_string_to_byte_array(const char *hexString, unsigned char **byteArray, size_t *byteArraySize) {
   size_t length = strlen(hexString);

   // Check if the hex string has an even length
   if(length % 2 != 0)
   {
      return -1;   // Invalid hex string length
   }

   *byteArraySize = length / 2;
   *byteArray = (unsigned char *)malloc(*byteArraySize);

   // Check if memory allocation was successful
   if(*byteArray == NULL)
   {
      return -1;   // Memory allocation failed
   }

   for(size_t i = 0; i < *byteArraySize; ++i)
   {
      int highNibble = hexCharToDecimal(hexString[i * 2]);
      int lowNibble = hexCharToDecimal(hexString[i * 2 + 1]);

      // Check for invalid characters
      if(highNibble == -1 || lowNibble == -1)
      {
         free(*byteArray);
         *byteArray = NULL;    // Prevent dangling pointer
         return -1;    // Invalid hex character
      }

      (*byteArray)[i] = (unsigned char)((highNibble << 4) | lowNibble);
   }

   return 0;  // Success
}
