/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2025 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneBOOT Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.5.2
 **/

//Dependencies
#include <stdlib.h>
#include "sam.h"
#include "same54_xplained_pro.h"
#include "bootloader/boot.h"
#include "drivers/flash/internal/sam_ed_5x_flash_driver.h"
#include "core/cboot_error.h"
#include "boot_config.h"
#include "debug.h"

//Global variables
uint8_t ledState = 0;
systime_t ledStart = 0;
systime_t ledTimeout = 0;

BootSettings bootSettings;
BootContext bootContext;

/**
 * @brief I/O initialization
 **/

void ioInit(void)
{
   //Enable PORT bus clock (CLK_PORT_APB)
   MCLK_REGS->MCLK_APBBMASK |= MCLK_APBBMASK_PORT_Msk;

   //Configure LED0
   PORT_REGS->GROUP[LED0_GROUP].PORT_DIRSET = LED0_MASK;
   PORT_REGS->GROUP[LED0_GROUP].PORT_OUTSET = LED0_MASK;

   //Configure SW0 button
   PORT_REGS->GROUP[SW0_GROUP].PORT_DIRCLR = SW0_MASK;
   PORT_REGS->GROUP[SW0_GROUP].PORT_PINCFG[SW0_PIN] |= PORT_PINCFG_INEN_Msk;
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void)
{
   if(systemTicks - ledStart >= ledTimeout)
   {
      if(ledState == 0)
      {
         PORT_REGS->GROUP[LED0_GROUP].PORT_OUTCLR = LED0_MASK;
         ledStart = systemTicks;
         ledTimeout = 100;
         ledState = 1;
      }
      else
      {
         PORT_REGS->GROUP[LED0_GROUP].PORT_OUTSET = LED0_MASK;
         ledStart = systemTicks;
         ledTimeout = 900;
         ledState = 0;
      }
   }
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void)
{
   cboot_error_t cerror;
   
   //Update system core clock
   SystemCoreClockUpdate();

   //Initialize kernel
   osInitKernel();
   //Configure debug UART
   debugInit(115200);

   //Start-up message
   TRACE_INFO("\r\n");
   TRACE_INFO("**********************************************\r\n");
   TRACE_INFO("*** CycloneBOOT IAP Single-Bank Bootloader ***\r\n");
   TRACE_INFO("**********************************************\r\n");
   TRACE_INFO("Copyright: 2010-2024 Oryx Embedded SARL\r\n");
   TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
   TRACE_INFO("Target: SAME54\r\n");
   TRACE_INFO("\r\n");

   //Configure I/Os
   ioInit();

   //Begining of handling exception
   do
   {
      //Get default booloader user settings
      bootGetDefaultSettings(&bootSettings);
      
      //User settings primary memory configuration
      bootSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
      bootSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
      bootSettings.memories[0].driver = &sam_ed_5x_FlashDriver;
      bootSettings.memories[0].nbSlots = 2;
      //User settings primary memory slot 0 configuration
      bootSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
      bootSettings.memories[0].slots[0].cType = SLOT_CONTENT_BINARY;
      bootSettings.memories[0].slots[0].memParent = &bootSettings.memories[0];
      bootSettings.memories[0].slots[0].addr = 0x00020000;
      bootSettings.memories[0].slots[0].size = 0x60000;
      //User settings primary memory slot 1 configuration
      bootSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
      bootSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE;
      bootSettings.memories[0].slots[1].memParent = &bootSettings.memories[0];
      bootSettings.memories[0].slots[1].addr = 0x00080000;
      bootSettings.memories[0].slots[1].size = 0x60000;

      //Initialize bootloader
      cerror = bootInit(&bootContext, &bootSettings);
      //Is any error?
      if(cerror)
         break;
   }while(0);

   //Is any error?
   if(cerror)
   {
      //Debug message
      TRACE_ERROR("Bootloader configuration failed!\r\n");
      while(1);
   }

   while(1)
   {
      ledTask();
      cerror = bootFsm(&bootContext);
      //Is any error?
      if(cerror)
      {
         //Debug message
         TRACE_ERROR("Bootloader failure!\r\n");
         break;
      }
   }

   //This function should never return
   return 0;
}
