/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Copyright (C) 2010-2024 Oryx Embedded SARL. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.4.4
 **/

//Dependencies
#include <stdlib.h>
#include <stdbool.h>
#include "stm32f7xx.h"
#include "stm32f7xx_hal.h"
#include "stm32f769i_eval.h"
#include "stm32f769i_eval_lcd.h"
#include "core/net.h"
#include "drivers/mac/stm32f7xx_eth_driver.h"
#include "drivers/phy/dp83848_driver.h"
#include "dhcp/dhcp_client.h"
#include "ipv6/slaac.h"
#include "mqtt/mqtt_client.h"
#include "hardware/stm32f7xx/stm32f7xx_crypto.h"
#include "rng/trng.h"
#include "rng/yarrow.h"
#include "smi_driver.h"
#include "update/update.h"
#include "security/verify.h"
#include "drivers/flash/internal/stm32f7xx_flash_driver.h"
#include "drivers/flash/external/n25q512a_flash_driver.h"
#include "version.h"
#include "debug.h"

//LCD frame buffers
#define LCD_FRAME_BUFFER_LAYER0 0xC0400000
#define LCD_FRAME_BUFFER_LAYER1 0xC0480000

//Ethernet interface configuration
#define APP_IF_NAME "eth0"
#define APP_HOST_NAME "mqtt-client-demo"
#define APP_MAC_ADDR "00-AB-CD-EF-07-69"

#define APP_USE_DHCP_CLIENT ENABLED
#define APP_IPV4_HOST_ADDR "192.168.0.20"
#define APP_IPV4_SUBNET_MASK "255.255.255.0"
#define APP_IPV4_DEFAULT_GATEWAY "192.168.0.254"
#define APP_IPV4_PRIMARY_DNS "8.8.8.8"
#define APP_IPV4_SECONDARY_DNS "8.8.4.4"

#define APP_USE_SLAAC ENABLED
#define APP_IPV6_LINK_LOCAL_ADDR "fe80::769"
#define APP_IPV6_PREFIX "2001:db8::"
#define APP_IPV6_PREFIX_LENGTH 64
#define APP_IPV6_GLOBAL_ADDR "2001:db8::769"
#define APP_IPV6_ROUTER "fe80::1"
#define APP_IPV6_PRIMARY_DNS "2001:4860:4860::8888"
#define APP_IPV6_SECONDARY_DNS "2001:4860:4860::8844"

//MQTT server name
#define APP_SERVER_NAME "157.90.125.161"

//CycloneBOOT Global variables
UpdateSettings updateSettings;
UpdateContext updateContext;

/**********************************************************************************************************/
/************************************ MQTT FUOTA Related Configuration ************************************/
/**********************************************************************************************************/

uint32_t big_endian_u32_read_from_array(const uint8_t* array);
void big_endian_u32_write_to_array(uint32_t u32_value, uint8_t* array);
void big_to_little_endian_transform(const uint8_t* input_buffer, uint8_t* output_buffer, size_t length);

// Topic from Server to Setup Device:
// Trigger update check, Provide last FW version, FW update start message
#define MQTT_TOPIC_SUB_OTA_SETUP "/" APP_MAC_ADDR "/ota/setup"

// Topic from Server to send Firmware Data messages
#define MQTT_TOPIC_SUB_OTA_DATA "/" APP_MAC_ADDR "/ota/data"

// Set QoS for messages
#define MQTT_FUOTA_QOS MQTT_QOS_LEVEL_2 //receive all messages exactly once

// Topic from Device for OTA Control Requests:
// FW Update check (request last FW version information), Request FW update,
// Notify Update completed
#define MQTT_TOPIC_PUB_OTA_CONTROL "/" APP_MAC_ADDR "/ota/control"
// Topic from Device to Acknowledge of received data messages
#define MQTT_TOPIC_PUB_OTA_ACK "/" APP_MAC_ADDR "/ota/ack"

/* MQTT FUOTA Commands */
// Message to force the device to trigger a FW update check
#define MSG_SETUP_CMD_TRIGGER_FW_UPDATE_CHECK 0x00
#define MSG_SETUP_CMD_TRIGGER_FW_UPDATE_CHECK_LENGTH 1U

// Message to provide last stable FW information (version, size, checksum)
#define MSG_SETUP_CMD_LAST_FW_INFO 0x01U
#define MSG_SETUP_CMD_LAST_FW_INFO_LENGTH 40U

// Message to request the device to start the FUOTA process (listening for FW
// data block messages
#define MSG_SETUP_CMD_FUOTA_START 0x02U
#define MSG_SETUP_CMD_FUOTA_START_LENGTH 1U

// Commands Frame Length
#define CMD_LEN 4U

// Firmware Version Length (XXX.YYY.ZZZ - Major.Minor.Patch)
#define FW_VERSION_LENGTH 3U

// MD5 Hash algorithm string value length
#define MD5_LENGTH 32U

// Firmware Data Block Size
#define FW_DATA_BLOCK_SIZE 512U

/* Constants - Control Message Commands (Device to Server) */

// Device request a FW Update Check to get last FW information from server
#define MSG_CONTROL_CMD_FW_UPDATE_CHECK { 0xafU, 0x12U, 0x34U, 0x56U }

// Device request to launch a FUOTA process to Server
#define MSG_CONTROL_CMD_REQUEST_FW_UPDATE { 0x55U, 0x55U, 0xffU, 0xffU }

// Device ready to start FUOTA process and handle reception of FW data blocks
#define MSG_ACK_FUOTA_START { 0xaaU, 0xaaU, 0xaaU, 0xaaU }

// Device NOT ready to start FUOTA process
#define MSG_NACK_FUOTA_START { 0xbbU, 0xbbU, 0xbbU, 0xbbU }

// FUOTA process completed successfully
#define MSG_CONTROL_CMD_FW_UPDATE_COMPLETED_OK { 0x55U, 0xaaU, 0xffU, 0xffU }

// FUOTA process completed but update on device has fail
#define MSG_CONTROL_CMD_FW_UPDATE_COMPLETED_FAIL { 0x55U, 0xaaU, 0x00U, 0x00U }

// Setup Message Fields buffer index locations
// Last FW Info Frame:
//   [ FW_INFO_CMD(0), FW_INFO_VER_MAJOR(1:3), FW_INFO_SIZE(4:7),
//     FW_INFO_MD5(8:40) ]
enum t_msg_fw_info_field
{
    FW_INFO_CMD = 0,
    FW_INFO_VER_MAJOR = 1,
    FW_INFO_VER_MINOR = 2,
    FW_INFO_VER_PATCH = 3,
    FW_INFO_SIZE = 4,
#if 0 /* Unused (used MD5 instead) */
    FW_INFO_CRC = 8,
#endif
    FW_INFO_MD5 = 8,
};

// Firmware Data Message Fields buffer index locations
// FW Data Block Frame:
//   [ FW_BLOCK_NUM(0:3), FW_BLOCK_DATA(4:1028) ]
enum t_msg_fw_data_block
{
    FW_BLOCK_NUM = 0,
    FW_BLOCK_DATA = 4,
};

typedef struct t_fw_info
{
    uint8_t version[FW_VERSION_LENGTH];
    uint32_t size;
#if 0 /* Unused CRC (used MD5 instead) */
    uint32_t crc;
#endif
    char md5[MD5_LENGTH+1];
} t_fw_info;


/**
 * @brief Flag to be set if the Device accept to update the FW (the FW
 * information received is valid and an update can proceed).
 */
bool valid_update;

/**
 * @brief Flag to identify that a FUOTA process is in progress.
 */
bool fuota_on_progress;

/**
 * @brief Flag to identify that a FUOTA process has been completed.
 */
bool fw_update_completed;

/**
 * @brief Flag to identify that a new FW data block has been received.
 */
bool fw_data_block_received;

/**
 * @brief Counter of Firmware bytes written to memory during a FUOTA
 * process.
 */
uint32_t fw_bytes_written;

/**
 * @brief Last Firmware data block received during the FUOTA process.
 */
uint32_t fw_block_n;

/**
 * @brief Current Device Firmware information (size, version and
 * checksum).
 */
t_fw_info fw_device;

/**
 * @brief Server available Firmware information (size, version and
 * checksum).
 */
t_fw_info fw_server;


uint8_t askForUpdateVersionFromServer = 0;
uint8_t askForFirmwareUpdateFromServer = 0;
uint8_t acknowledgeFirmwareUpdateStart = 0;
uint8_t denyFirmwareUpdateStart = 0;
uint32_t numberOfDataBlocks = 0;
/**********************************************************************************************************/
/**********************************************************************************************************/
/**********************************************************************************************************/

//MQTT server port
#define APP_SERVER_PORT 1883   //MQTT over TCP
//#define APP_SERVER_PORT 8883 //MQTT over TLS
//#define APP_SERVER_PORT 8884 //MQTT over TLS (mutual authentication)
//#define APP_SERVER_PORT 8080 //MQTT over WebSocket
//#define APP_SERVER_PORT 8081 //MQTT over secure WebSocket

//URI (for MQTT over WebSocket only)
#define APP_SERVER_URI "/ws"

//Client's certificate
const char_t clientCert[] =
   "-----BEGIN CERTIFICATE-----"
   "MIICmzCCAYOgAwIBAgIBADANBgkqhkiG9w0BAQsFADCBkDELMAkGA1UEBhMCR0Ix"
   "FzAVBgNVBAgMDlVuaXRlZCBLaW5nZG9tMQ4wDAYDVQQHDAVEZXJieTESMBAGA1UE"
   "CgwJTW9zcXVpdHRvMQswCQYDVQQLDAJDQTEWMBQGA1UEAwwNbW9zcXVpdHRvLm9y"
   "ZzEfMB0GCSqGSIb3DQEJARYQcm9nZXJAYXRjaG9vLm9yZzAeFw0yNDA1MjAxMjIw"
   "MTJaFw0yNDA4MTgxMjIwMTJaMEAxCzAJBgNVBAYTAkZSMRYwFAYDVQQKDA1Pcnl4"
   "IEVtYmVkZGVkMRkwFwYDVQQDDBBtcXR0LWNsaWVudC1kZW1vMFkwEwYHKoZIzj0C"
   "AQYIKoZIzj0DAQcDQgAEWT/enOkLuY+9NzUQPOuNVFARl5Y3bc4lLt3TyVwWG0Ez"
   "IIk8Wll5Ljjrv+buPSKBVQtOwF9VgyW4QuQ1uYSAIaMaMBgwCQYDVR0TBAIwADAL"
   "BgNVHQ8EBAMCBeAwDQYJKoZIhvcNAQELBQADggEBALW6YSU2jqs7TegW2CoydK7W"
   "rEzy/8ecasdeDy+8vIkxC1chtm2VgYsndFUaLhSD31I0paLqE67kGzyD8TKhBAyp"
   "4trhLsgFUjTsKYj8kPj147am2wwP0hbd1dygr+kaNxV3glZqot8IK2+topA3BbFO"
   "34wvdNI81o63ldKn+Q3sp522osxfwiCy8/5DZ+2VLzGBwbsKaYNk2RwhUNbF88eD"
   "IHAi9iY3YYx7hE9UvMA3CbWBOmz06lFWr2Nwyr3lYM6qO87GKKMZaFjdrFtzQlaB"
   "ngH1f6yCuM0iwd7gbWKwPWoilIDdg8DVi38fN/Clv5PbTt+KJslVbNElV6ykbyA="
   "-----END CERTIFICATE-----";

//Client's private key
const char_t clientKey[] =
   "-----BEGIN EC PRIVATE KEY-----"
   "MHcCAQEEICYULY0KQ6nDAXFl5tgK9ljqAZyb14JQmI3iT7tdScDloAoGCCqGSM49"
   "AwEHoUQDQgAEWT/enOkLuY+9NzUQPOuNVFARl5Y3bc4lLt3TyVwWG0EzIIk8Wll5"
   "Ljjrv+buPSKBVQtOwF9VgyW4QuQ1uYSAIQ=="
   "-----END EC PRIVATE KEY-----";

//List of trusted CA certificates
const char_t trustedCaList[] =
   "-----BEGIN CERTIFICATE-----"
   "MIIEAzCCAuugAwIBAgIUBY1hlCGvdj4NhBXkZ/uLUZNILAwwDQYJKoZIhvcNAQEL"
   "BQAwgZAxCzAJBgNVBAYTAkdCMRcwFQYDVQQIDA5Vbml0ZWQgS2luZ2RvbTEOMAwG"
   "A1UEBwwFRGVyYnkxEjAQBgNVBAoMCU1vc3F1aXR0bzELMAkGA1UECwwCQ0ExFjAU"
   "BgNVBAMMDW1vc3F1aXR0by5vcmcxHzAdBgkqhkiG9w0BCQEWEHJvZ2VyQGF0Y2hv"
   "by5vcmcwHhcNMjAwNjA5MTEwNjM5WhcNMzAwNjA3MTEwNjM5WjCBkDELMAkGA1UE"
   "BhMCR0IxFzAVBgNVBAgMDlVuaXRlZCBLaW5nZG9tMQ4wDAYDVQQHDAVEZXJieTES"
   "MBAGA1UECgwJTW9zcXVpdHRvMQswCQYDVQQLDAJDQTEWMBQGA1UEAwwNbW9zcXVp"
   "dHRvLm9yZzEfMB0GCSqGSIb3DQEJARYQcm9nZXJAYXRjaG9vLm9yZzCCASIwDQYJ"
   "KoZIhvcNAQEBBQADggEPADCCAQoCggEBAME0HKmIzfTOwkKLT3THHe+ObdizamPg"
   "UZmD64Tf3zJdNeYGYn4CEXbyP6fy3tWc8S2boW6dzrH8SdFf9uo320GJA9B7U1FW"
   "Te3xda/Lm3JFfaHjkWw7jBwcauQZjpGINHapHRlpiCZsquAthOgxW9SgDgYlGzEA"
   "s06pkEFiMw+qDfLo/sxFKB6vQlFekMeCymjLCbNwPJyqyhFmPWwio/PDMruBTzPH"
   "3cioBnrJWKXc3OjXdLGFJOfj7pP0j/dr2LH72eSvv3PQQFl90CZPFhrCUcRHSSxo"
   "E6yjGOdnz7f6PveLIB574kQORwt8ePn0yidrTC1ictikED3nHYhMUOUCAwEAAaNT"
   "MFEwHQYDVR0OBBYEFPVV6xBUFPiGKDyo5V3+Hbh4N9YSMB8GA1UdIwQYMBaAFPVV"
   "6xBUFPiGKDyo5V3+Hbh4N9YSMA8GA1UdEwEB/wQFMAMBAf8wDQYJKoZIhvcNAQEL"
   "BQADggEBAGa9kS21N70ThM6/Hj9D7mbVxKLBjVWe2TPsGfbl3rEDfZ+OKRZ2j6AC"
   "6r7jb4TZO3dzF2p6dgbrlU71Y/4K0TdzIjRj3cQ3KSm41JvUQ0hZ/c04iGDg/xWf"
   "+pp58nfPAYwuerruPNWmlStWAXf0UTqRtg4hQDWBuUFDJTuWuuBvEXudz74eh/wK"
   "sMwfu1HFvjy5Z0iMDU8PUDepjVolOCue9ashlS4EB5IECdSR2TItnAIiIwimx839"
   "LdUdRudafMu5T5Xma182OC0/u/xRlEm+tvKGGmfFcN0piqVl8OrSPBgIlb+1IKJE"
   "m/XriWr/Cq4h/JfB7NTsezVslgkBaoU="
   "-----END CERTIFICATE-----"
   "-----BEGIN CERTIFICATE-----"
   "MIIFFjCCAv6gAwIBAgIRAJErCErPDBinU/bWLiWnX1owDQYJKoZIhvcNAQELBQAw"
   "TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh"
   "cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwHhcNMjAwOTA0MDAwMDAw"
   "WhcNMjUwOTE1MTYwMDAwWjAyMQswCQYDVQQGEwJVUzEWMBQGA1UEChMNTGV0J3Mg"
   "RW5jcnlwdDELMAkGA1UEAxMCUjMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK"
   "AoIBAQC7AhUozPaglNMPEuyNVZLD+ILxmaZ6QoinXSaqtSu5xUyxr45r+XXIo9cP"
   "R5QUVTVXjJ6oojkZ9YI8QqlObvU7wy7bjcCwXPNZOOftz2nwWgsbvsCUJCWH+jdx"
   "sxPnHKzhm+/b5DtFUkWWqcFTzjTIUu61ru2P3mBw4qVUq7ZtDpelQDRrK9O8Zutm"
   "NHz6a4uPVymZ+DAXXbpyb/uBxa3Shlg9F8fnCbvxK/eG3MHacV3URuPMrSXBiLxg"
   "Z3Vms/EY96Jc5lP/Ooi2R6X/ExjqmAl3P51T+c8B5fWmcBcUr2Ok/5mzk53cU6cG"
   "/kiFHaFpriV1uxPMUgP17VGhi9sVAgMBAAGjggEIMIIBBDAOBgNVHQ8BAf8EBAMC"
   "AYYwHQYDVR0lBBYwFAYIKwYBBQUHAwIGCCsGAQUFBwMBMBIGA1UdEwEB/wQIMAYB"
   "Af8CAQAwHQYDVR0OBBYEFBQusxe3WFbLrlAJQOYfr52LFMLGMB8GA1UdIwQYMBaA"
   "FHm0WeZ7tuXkAXOACIjIGlj26ZtuMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcw"
   "AoYWaHR0cDovL3gxLmkubGVuY3Iub3JnLzAnBgNVHR8EIDAeMBygGqAYhhZodHRw"
   "Oi8veDEuYy5sZW5jci5vcmcvMCIGA1UdIAQbMBkwCAYGZ4EMAQIBMA0GCysGAQQB"
   "gt8TAQEBMA0GCSqGSIb3DQEBCwUAA4ICAQCFyk5HPqP3hUSFvNVneLKYY611TR6W"
   "PTNlclQtgaDqw+34IL9fzLdwALduO/ZelN7kIJ+m74uyA+eitRY8kc607TkC53wl"
   "ikfmZW4/RvTZ8M6UK+5UzhK8jCdLuMGYL6KvzXGRSgi3yLgjewQtCPkIVz6D2QQz"
   "CkcheAmCJ8MqyJu5zlzyZMjAvnnAT45tRAxekrsu94sQ4egdRCnbWSDtY7kh+BIm"
   "lJNXoB1lBMEKIq4QDUOXoRgffuDghje1WrG9ML+Hbisq/yFOGwXD9RiX8F6sw6W4"
   "avAuvDszue5L3sz85K+EC4Y/wFVDNvZo4TYXao6Z0f+lQKc0t8DQYzk1OXVu8rp2"
   "yJMC6alLbBfODALZvYH7n7do1AZls4I9d1P4jnkDrQoxB3UqQ9hVl3LEKQ73xF1O"
   "yK5GhDDX8oVfGKF5u+decIsH4YaTw7mP3GFxJSqv3+0lUFJoi5Lc5da149p90Ids"
   "hCExroL1+7mryIkXPeFM5TgO9r0rvZaBFOvV2z0gp35Z0+L4WPlbuEjN/lxPFin+"
   "HlUjr8gRsI3qfJOQFy/9rKIJR0Y/8Omwt/8oTWgy1mdeHmmjk7j1nYsvC9JSQ6Zv"
   "MldlTTKB3zhThV1+XWYp6rjd5JW1zbVWEkLNxE7GJThEUG3szgBVGP7pSWTUTsqX"
   "nLRbwHOoq7hHwg=="
   "-----END CERTIFICATE-----";

//Global variables
uint_t lcdLine = 0;
uint_t lcdColumn = 0;
uint_t adcValue = 0;

DhcpClientSettings dhcpClientSettings;
DhcpClientContext dhcpClientContext;
SlaacSettings slaacSettings;
SlaacContext slaacContext;
MqttClientContext mqttClientContext;
YarrowContext yarrowContext;
uint8_t seed[32];


/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/

void lcdSetCursor(uint_t line, uint_t column)
{
   lcdLine = MIN(line, 20);
   lcdColumn = MIN(column, 50);
}


/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/

void lcdPutChar(char_t c)
{
   if(c == '\r')
   {
      lcdColumn = 0;
   }
   else if(c == '\n')
   {
      lcdColumn = 0;
      lcdLine++;
   }
   else if(lcdLine < 20 && lcdColumn < 50)
   {
      //Display current character
      BSP_LCD_DisplayChar(lcdColumn * 16, lcdLine * 24, c);

      //Advance the cursor position
      if(++lcdColumn >= 50)
      {
         lcdColumn = 0;
         lcdLine++;
      }
   }
}


/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void)
{
   RCC_OscInitTypeDef RCC_OscInitStruct = {0};
   RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

   //Enable Power Control clock
   __HAL_RCC_PWR_CLK_ENABLE();

   //Enable HSE Oscillator and activate PLL with HSE as source
   RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
   RCC_OscInitStruct.HSEState = RCC_HSE_ON;
   RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
   RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
   RCC_OscInitStruct.PLL.PLLM = 25;
   RCC_OscInitStruct.PLL.PLLN = 400;
   RCC_OscInitStruct.PLL.PLLP = RCC_PLLP_DIV2;
   RCC_OscInitStruct.PLL.PLLQ = 9;
   HAL_RCC_OscConfig(&RCC_OscInitStruct);

   //Enable overdrive mode
   HAL_PWREx_EnableOverDrive();

   //Select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2
   //clocks dividers
   RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
      RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2;
   RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
   RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
   RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV4;
   RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV2;
   HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_7);
}


/**
 * @brief MPU configuration
 **/

void MPU_Config(void)
{
   MPU_Region_InitTypeDef MPU_InitStruct;

   //Disable MPU
   HAL_MPU_Disable();

   //SRAM2 (no cache)
   MPU_InitStruct.Enable = MPU_REGION_ENABLE;
   MPU_InitStruct.Number = MPU_REGION_NUMBER0;
   MPU_InitStruct.BaseAddress = 0x2007C000;
   MPU_InitStruct.Size = MPU_REGION_SIZE_16KB;
   MPU_InitStruct.SubRegionDisable = 0;
   MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL1;
   MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
   MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
   MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
   MPU_InitStruct.IsCacheable = MPU_ACCESS_NOT_CACHEABLE;
   MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
   HAL_MPU_ConfigRegion(&MPU_InitStruct);

   //SDRAM
   MPU_InitStruct.Enable = MPU_REGION_ENABLE;
   MPU_InitStruct.Number = MPU_REGION_NUMBER1;
   MPU_InitStruct.BaseAddress = 0xC0000000;
   MPU_InitStruct.Size = MPU_REGION_SIZE_8MB;
   MPU_InitStruct.SubRegionDisable = 0;
   MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL1;
   MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
   MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
   MPU_InitStruct.IsShareable = MPU_ACCESS_NOT_SHAREABLE;
   MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
   MPU_InitStruct.IsBufferable = MPU_ACCESS_BUFFERABLE;
   HAL_MPU_ConfigRegion(&MPU_InitStruct);

   //LCD frame buffer 
   MPU_InitStruct.Enable = MPU_REGION_ENABLE;
   MPU_InitStruct.Number = MPU_REGION_NUMBER2;
   MPU_InitStruct.BaseAddress = 0xC0400000;
   MPU_InitStruct.Size = MPU_REGION_SIZE_4MB;
   MPU_InitStruct.SubRegionDisable = 0;
   MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
   MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
   MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
   MPU_InitStruct.IsShareable = MPU_ACCESS_NOT_SHAREABLE;
   MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
   MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
   HAL_MPU_ConfigRegion(&MPU_InitStruct);

   //Enable MPU
   HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}


/**
 * @brief Random data generation callback function
 * @param[out] data Buffer where to store the random data
 * @param[in] length Number of bytes that are required
 * @return Error code
 **/

error_t webSocketRngCallback(uint8_t *data, size_t length)
{
   //Generate some random data
   return yarrowRead(&yarrowContext, data, length);
}


/**
 * @brief TLS initialization callback
 * @param[in] context Pointer to the MQTT client context
 * @param[in] tlsContext Pointer to the TLS context
 * @return Error code
 **/

error_t mqttTestTlsInitCallback(MqttClientContext *context,
   TlsContext *tlsContext)
{
   error_t error;

   //Debug message
   TRACE_INFO("MQTT: TLS initialization callback\r\n");

   //Set the PRNG algorithm to be used
   error = tlsSetPrng(tlsContext, YARROW_PRNG_ALGO, &yarrowContext);
   //Any error to report?
   if(error)
      return error;

   //Set the fully qualified domain name of the server
   error = tlsSetServerName(tlsContext, APP_SERVER_NAME);
   //Any error to report?
   if(error)
      return error;

#if (APP_SERVER_PORT == 8884)
   //Load client's certificate
   error = tlsLoadCertificate(tlsContext, 0, clientCert, strlen(clientCert),
      clientKey, strlen(clientKey), NULL);
   //Any error to report?
   if(error)
      return error;
#endif

   //Import trusted CA certificates
   error = tlsSetTrustedCaList(tlsContext, trustedCaList, strlen(trustedCaList));
   //Any error to report?
   if(error)
      return error;

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Publish callback function
 * @param[in] context Pointer to the MQTT client context
 * @param[in] topic Topic name
 * @param[in] message Message payload
 * @param[in] length Length of the message payload
 * @param[in] dup Duplicate delivery of the PUBLISH packet
 * @param[in] qos QoS level used to publish the message
 * @param[in] retain This flag specifies if the message is to be retained
 * @param[in] packetId Packet identifier
 **/

void mqttTestPublishCallback(MqttClientContext *context,
   const char_t *topic, const uint8_t *message, size_t length,
   bool_t dup, MqttQosLevel qos, bool_t retain, uint16_t packetId)
{

   //Debug message
//   TRACE_INFO("PUBLISH packet received...\r\n");
//   TRACE_INFO("  Dup: %u\r\n", dup);
//   TRACE_INFO("  QoS: %u\r\n", qos);
//   TRACE_INFO("  Retain: %u\r\n", retain);
//   TRACE_INFO("  Packet Identifier: %u\r\n", packetId);
//   TRACE_INFO("  Topic: %s\r\n", topic);
//   TRACE_INFO("  Message (%" PRIuSIZE " bytes):\r\n", length);
//   TRACE_INFO_ARRAY("    ", message, length);

   if(!strcmp(topic, MQTT_TOPIC_SUB_OTA_SETUP))
   {
	   if(length == MSG_SETUP_CMD_TRIGGER_FW_UPDATE_CHECK_LENGTH && (*message == MSG_SETUP_CMD_TRIGGER_FW_UPDATE_CHECK)) {
		   TRACE_INFO("Server requesting device to check for FW update\r\n");
		   askForUpdateVersionFromServer = 1;
	   }
	   if(length == MSG_SETUP_CMD_LAST_FW_INFO_LENGTH && (*message == MSG_SETUP_CMD_LAST_FW_INFO)) {
		   TRACE_INFO("Server sent current latest available firmware version: ");
		   // Get FW version
		   fw_server.version[0] = message[FW_INFO_VER_MAJOR];
		   fw_server.version[1] = message[FW_INFO_VER_MINOR];
		   fw_server.version[2] = message[FW_INFO_VER_PATCH];

		   // Get FW size
		   fw_server.size =
				   big_endian_u32_read_from_array(&(message[FW_INFO_SIZE]));

		   numberOfDataBlocks = (fw_server.size / FW_DATA_BLOCK_SIZE);

		   // Get and convert MD5 bytes to string of chars
		   memcpy((void*)(fw_server.md5),
				   (const void*)(&(message[FW_INFO_MD5])),
				   (size_t)(MD5_LENGTH));
		   fw_server.md5[MD5_LENGTH] = '\0';


		   TRACE_INFO(" %d.%d.%d\r\n", fw_server.version[0],fw_server.version[1],fw_server.version[2]);
		   TRACE_INFO("Firmware size: %" PRIu32  " bytes\r\n", fw_server.size);
		   TRACE_INFO("Expecting %" PRIu32  " blocks of 512 bytes\r\n", numberOfDataBlocks);

		   askForUpdateVersionFromServer = 0;
		   askForFirmwareUpdateFromServer = 1;
	   }
	   if(length == MSG_SETUP_CMD_FUOTA_START_LENGTH && (*message == MSG_SETUP_CMD_FUOTA_START)) {

		   cboot_error_t cerror = CBOOT_NO_ERROR;

#if (UPDATE_ANTI_ROLLBACK_SUPPORT == ENABLED)
		   updateSettings.appVersion = APP_VERSION;
#endif
		   //User update settings security configuration
		   updateSettings.imageInCrypto.verifySettings.verifyMethod    = VERIFY_METHOD_INTEGRITY;
		   updateSettings.imageInCrypto.verifySettings.integrityAlgo   = SHA256_HASH_ALGO;

		   //User update settings primary memory configuration
		   updateSettings.memories[0].memoryRole 			   = MEMORY_ROLE_PRIMARY;
		   updateSettings.memories[0].memoryType 			   = MEMORY_TYPE_FLASH;
		   updateSettings.memories[0].driver     			   = &stm32f7xxFlashDriver;
		   updateSettings.memories[0].nbSlots    			   = 1;
		   //User update settings primary memory slot 0 configuration
		   updateSettings.memories[0].slots[0].type 		   = SLOT_TYPE_DIRECT;
		   updateSettings.memories[0].slots[0].cType 		= SLOT_CONTENT_APP;
		   updateSettings.memories[0].slots[0].memParent 	= &updateSettings.memories[0];
		   updateSettings.memories[0].slots[0].addr 		   = 0x08020000;
		   updateSettings.memories[0].slots[0].size 		   = 0x1E0000;

		   //User update settings secondary memory configuration
		   updateSettings.memories[1].memoryRole 			   = MEMORY_ROLE_SECONDARY;
		   updateSettings.memories[1].memoryType 			   = MEMORY_TYPE_FLASH;
		   updateSettings.memories[1].driver 				   = &n25q512aFlashDriver;
		   updateSettings.memories[1].nbSlots 				   = 1;

		   //User update settings secondary memory slot 0 configuration
		   updateSettings.memories[1].slots[0].type 		   = SLOT_TYPE_DIRECT;
		   updateSettings.memories[1].slots[0].cType 		= SLOT_CONTENT_APP | SLOT_CONTENT_BACKUP;
		   updateSettings.memories[1].slots[0].memParent 	= &updateSettings.memories[1];
		   updateSettings.memories[1].slots[0].addr 		   = 0x00000000;
		   updateSettings.memories[1].slots[0].size 		   = 0x1E0000;

		   //Initialize IAP update context
		   cerror = updateInit(&updateContext, &updateSettings);
		   //Is any error?
		   if(cerror)
		   {
			   //Debug message
			   TRACE_ERROR("Failed to initialize IAP!\r\n");
			   denyFirmwareUpdateStart = 1;
		   } else {
			   TRACE_INFO("Acknowledging firmware update\r\n");
			   acknowledgeFirmwareUpdateStart = 1;
		   }
	   }
   }

   if(!strcmp(topic, MQTT_TOPIC_SUB_OTA_DATA))
   {
	   cboot_error_t cerror = CBOOT_NO_ERROR;
	   uint8_t buffer[FW_DATA_BLOCK_SIZE];

	   // Parse payload and get FW block number and data
	   fw_block_n = big_endian_u32_read_from_array(&(message[FW_BLOCK_NUM]));
	   const uint8_t* fw_data = &(message[FW_BLOCK_DATA]);
	   uint32_t fw_data_length = length - (FW_BLOCK_DATA - FW_BLOCK_NUM);

	   // Limit bytes to write if there is coming more than expected
	   if (fw_bytes_written + fw_data_length > fw_server.size)
	   {   fw_data_length = fw_server.size - fw_bytes_written;   }

	   // Write FW data block into memory
	   big_to_little_endian_transform(fw_data, buffer, fw_data_length);

	   //Write received bytes in flash
	   TRACE_INFO("Received %ld bytes\r\n",fw_data_length);
	   cerror = updateProcess(&updateContext, (uint8_t*) buffer, fw_data_length);

	   //Is any error?
	   if(cerror != CBOOT_NO_ERROR)
	   {
		   //Debug message
		   TRACE_ERROR("Failed to update firmware!\r\n");
		   denyFirmwareUpdateStart = 1;
	   } else
	   {
		   fw_bytes_written += fw_data_length;
		   TRACE_INFO("So far written %ld bytes\r\n",fw_bytes_written);
	   }

	   // Check if FW update has been completed
	   if (fw_bytes_written == fw_server.size)
	   {
		   //Check application firmware validity
		   cerror = updateFinalize(&updateContext);
		   //Is application is invalid?
		   if(cerror != CBOOT_NO_ERROR)
		   {
			   //Debug message
			   TRACE_ERROR("Failed to finalize firmware update!\r\n");
			   fw_update_completed = false;

		   } else
		   {
			   fw_update_completed = true;
		   }

	   }

	   // Set flag of new FW data block received
	   fw_data_block_received = true;
   }


}


/**
 * @brief Establish MQTT connection
 **/

error_t mqttTestConnect(void)
{
   error_t error;
   IpAddr ipAddr;

   //Debug message
   TRACE_INFO("\r\n\r\nResolving server name...\r\n");

   //Resolve MQTT server name
   error = getHostByName(NULL, APP_SERVER_NAME, &ipAddr, 0);
   //Any error to report?
   if(error)
      return error;

#if (APP_SERVER_PORT == 8080 || APP_SERVER_PORT == 8081)
   //Register RNG callback
   webSocketRegisterRandCallback(webSocketRngCallback);
#endif

   //Set the MQTT version to be used
   mqttClientSetVersion(&mqttClientContext, MQTT_VERSION_3_1_1);

#if (APP_SERVER_PORT == 1883)
   //MQTT over TCP
   mqttClientSetTransportProtocol(&mqttClientContext, MQTT_TRANSPORT_PROTOCOL_TCP);
#elif (APP_SERVER_PORT == 8883 || APP_SERVER_PORT == 8884)
   //MQTT over TLS
   mqttClientSetTransportProtocol(&mqttClientContext, MQTT_TRANSPORT_PROTOCOL_TLS);
   //Register TLS initialization callback
   mqttClientRegisterTlsInitCallback(&mqttClientContext, mqttTestTlsInitCallback);
#elif (APP_SERVER_PORT == 8080)
   //MQTT over WebSocket
   mqttClientSetTransportProtocol(&mqttClientContext, MQTT_TRANSPORT_PROTOCOL_WS);
#elif (APP_SERVER_PORT == 8081)
   //MQTT over secure WebSocket
   mqttClientSetTransportProtocol(&mqttClientContext, MQTT_TRANSPORT_PROTOCOL_WSS);
   //Register TLS initialization callback
   mqttClientRegisterTlsInitCallback(&mqttClientContext, mqttTestTlsInitCallback);
#endif

   //Register publish callback function
   mqttClientRegisterPublishCallback(&mqttClientContext, mqttTestPublishCallback);

   //Set communication timeout
   mqttClientSetTimeout(&mqttClientContext, 20000);
   //Set keep-alive value
   mqttClientSetKeepAlive(&mqttClientContext, 30);

#if (APP_SERVER_PORT == 8080 || APP_SERVER_PORT == 8081)
   //Set the hostname of the resource being requested
   mqttClientSetHost(&mqttClientContext, APP_SERVER_NAME);
   //Set the name of the resource being requested
   mqttClientSetUri(&mqttClientContext, APP_SERVER_URI);
#endif

   //Set client identifier
   //mqttClientSetIdentifier(&mqttClientContext, "client12345678");

   //Set user name and password
   //mqttClientSetAuthInfo(&mqttClientContext, "username", "password");

   //Set Will message
   mqttClientSetWillMessage(&mqttClientContext, "board/status",
      "offline", 7, MQTT_QOS_LEVEL_0, FALSE);

   //Debug message
   TRACE_INFO("Connecting to MQTT server %s...\r\n", ipAddrToString(&ipAddr, NULL));

   //Start of exception handling block
   do
   {
      //Establish connection with the MQTT server
      error = mqttClientConnect(&mqttClientContext,
         &ipAddr, APP_SERVER_PORT, TRUE);
      //Any error to report?
      if(error)
         break;

      //Subscribe to FUOTA Setup Topic
      error = mqttClientSubscribe(&mqttClientContext,
    		  MQTT_TOPIC_SUB_OTA_SETUP, MQTT_FUOTA_QOS, NULL);
      //Any error to report?
      if(error)
    	  break;

      //Subscribe to FUOTA Data Topic
      error = mqttClientSubscribe(&mqttClientContext,
    		  MQTT_TOPIC_SUB_OTA_DATA, MQTT_FUOTA_QOS, NULL);
      //Any error to report?
      if(error)
    	  break;


      //Send PUBLISH packet
      error = mqttClientPublish(&mqttClientContext, "board/status",
         "online", 6, MQTT_FUOTA_QOS, TRUE, NULL);
      //Any error to report?
      if(error)
         break;

      //End of exception handling block
   } while(0);

   //Check status code
   if(error)
   {
      //Close connection
      mqttClientClose(&mqttClientContext);
   }

   //Return status code
   return error;
}


/**
 * @brief MQTT test task
 **/

void mqttTestTask(void *param)
{
   error_t error;
   bool_t connectionState;

   //Initialize variables
   connectionState = FALSE;

   //Initialize MQTT client context
   mqttClientInit(&mqttClientContext);

   //Endless loop
   while(1)
   {
      //Check connection state
      if(!connectionState)
      {
         //Make sure the link is up
         if(netGetLinkState(&netInterface[0]))
         {
            //Try to connect to the MQTT server
            error = mqttTestConnect();

            //Successful connection?
            if(!error)
            {
               //The MQTT client is connected to the server
               connectionState = TRUE;
            }
            else
            {
               //Delay between subsequent connection attempts
               osDelayTask(2000);
            }
         }
         else
         {
            //The link is down
            osDelayTask(1000);
         }
      }
      else
      {
         //Initialize status code
         error = NO_ERROR;

         //Publish to FUOTA Control Topic
         if(askForUpdateVersionFromServer)
         {
        	 //Ask the server to send firmware version of the update
        	 static const char updateCheckCommand[] = MSG_CONTROL_CMD_FW_UPDATE_CHECK;
        	 size_t updateCheckCommandLength = sizeof(updateCheckCommand) / sizeof(updateCheckCommand[0]);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_CONTROL,
        			 updateCheckCommand, updateCheckCommandLength, MQTT_FUOTA_QOS, FALSE, NULL);

        	 askForUpdateVersionFromServer = 0;
         }

         //Publish to FUOTA Control Topic
         if(askForFirmwareUpdateFromServer)
         {
        	 //Ask the server to send firmware version of the update
        	 static const char updateRequestCommand[] = MSG_CONTROL_CMD_REQUEST_FW_UPDATE;
        	 size_t updateRequestCommandLength = sizeof(updateRequestCommand) / sizeof(updateRequestCommand[0]);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_CONTROL,
        			 updateRequestCommand, updateRequestCommandLength, MQTT_FUOTA_QOS, FALSE, NULL);

        	 askForFirmwareUpdateFromServer = 0;
         }

         //Publish to FUOTA Control Topic
         if(acknowledgeFirmwareUpdateStart)
         {
        	 //Acknowledge to the server that client is ready for update
        	 static const char updateAcknowledgeCommand[] = MSG_ACK_FUOTA_START;
        	 size_t updateAcknowledgeCommandLength = sizeof(updateAcknowledgeCommand) / sizeof(updateAcknowledgeCommand[0]);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_CONTROL,
        			 updateAcknowledgeCommand, updateAcknowledgeCommandLength, MQTT_FUOTA_QOS, FALSE, NULL);

        	 acknowledgeFirmwareUpdateStart = 0;
         }

         //Publish to FUOTA Control Topic
         if(denyFirmwareUpdateStart)
         {
        	 static const char updateRejectCommand[] = MSG_NACK_FUOTA_START;
        	 size_t updateRejectCommandLength = sizeof(updateRejectCommand) / sizeof(updateRejectCommand[0]);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_CONTROL,
        			 updateRejectCommand, updateRejectCommandLength, MQTT_FUOTA_QOS, FALSE, NULL);

        	 denyFirmwareUpdateStart = 0;
         }

         //Publish to FUOTA ACK Topic
         if(fw_data_block_received)
         {
        	 uint8_t payload[CMD_LEN];
        	 //Acknowledge to the server that a data block has completely been received
        	 //Server expects Big Endian payload
        	 big_endian_u32_write_to_array(fw_block_n, payload);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_ACK,
        			 payload, CMD_LEN, MQTT_FUOTA_QOS, FALSE, NULL);

        	 fw_data_block_received = false;
         }

         if(fw_update_completed)
         {
        	 cboot_error_t cerror = CBOOT_NO_ERROR;
        	 //Acknowledge to the server that all data blocks has completely been received
        	 static const char updateCompleteCommand[] = MSG_CONTROL_CMD_FW_UPDATE_COMPLETED_OK;
        	 size_t updateCompleteCommandLength = sizeof(updateCompleteCommand) / sizeof(updateCompleteCommand[0]);

        	 error = mqttClientPublish(&mqttClientContext, MQTT_TOPIC_PUB_OTA_CONTROL,
        			 updateCompleteCommand, updateCompleteCommandLength, MQTT_FUOTA_QOS, FALSE, NULL);

        	 //TODO: implement protocol logic to notify the update is complete and device is requesting reboot
        	 //TODO: implement a callback to perform actual reboot (probably after cleanup) instead of doing it here
        	 cerror = updateReboot(&updateContext);
        	 //Is any error?
        	 if(cerror != CBOOT_NO_ERROR)
        	 {
        		 //Debug message
        		 TRACE_ERROR("Failed to reboot!\r\n");
        	 }

        	 fw_update_completed = false;
         }

         //Check status code
         if(!error)
         {
            //Process events
            error = mqttClientTask(&mqttClientContext, 100);
         }

         //Connection to MQTT server lost?
         if(error)
         {
            //Close connection
            mqttClientClose(&mqttClientContext);
            //Update connection state
            connectionState = FALSE;
            //Recovery delay
            osDelayTask(2000);
         }
      }
   }
}


/**
 * @brief User task
 * @param[in] param Unused parameter
 **/

void userTask(void *param)
{
   char_t buffer[40];
#if (IPV4_SUPPORT == ENABLED)
   Ipv4Addr ipv4Addr;
#endif
#if (IPV6_SUPPORT == ENABLED)
   Ipv6Addr ipv6Addr;
#endif

   //Point to the network interface
   NetInterface *interface = &netInterface[0];

   //Initialize LCD display
   lcdSetCursor(2, 0);
   printf("IPv4 Addr\r\n");
   lcdSetCursor(5, 0);
   printf("IPv6 Link-Local Addr\r\n");
   lcdSetCursor(8, 0);
   printf("IPv6 Global Addr\r\n");

   //Endless loop
   while(1)
   {
#if (IPV4_SUPPORT == ENABLED)
      //Display IPv4 host address
      lcdSetCursor(3, 0);
      ipv4GetHostAddr(interface, &ipv4Addr);
      printf("%-16s\r\n", ipv4AddrToString(ipv4Addr, buffer));
#endif

#if (IPV6_SUPPORT == ENABLED)
      //Display IPv6 link-local address
      lcdSetCursor(6, 0);
      ipv6GetLinkLocalAddr(interface, &ipv6Addr);
      printf("%-40s\r\n", ipv6AddrToString(&ipv6Addr, buffer));

      //Display IPv6 global address
      lcdSetCursor(9, 0);
      ipv6GetGlobalAddr(interface, 0, &ipv6Addr);
      printf("%-40s\r\n", ipv6AddrToString(&ipv6Addr, buffer));
#endif

      //Perform A/D conversion
      adcValue = BSP_POTENTIOMETER_GetLevel();

      //Loop delay
      osDelayTask(100);
   }
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param)
{
   //Endless loop
   while(1)
   {
      BSP_LED_On(LED1);
      osDelayTask(100);
      BSP_LED_Off(LED1);
      osDelayTask(900);
   }
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void)
{
   error_t error;
   OsTaskId taskId;
   OsTaskParameters taskParams;
   NetInterface *interface;
   MacAddr macAddr;
#if (APP_USE_DHCP_CLIENT == DISABLED)
   Ipv4Addr ipv4Addr;
#endif
#if (APP_USE_SLAAC == DISABLED)
   Ipv6Addr ipv6Addr;
#endif

   //MPU configuration
   MPU_Config();
   //HAL library initialization
   HAL_Init();
   //Configure the system clock
   SystemClock_Config();

   //Enable I-cache and D-cache
   SCB_EnableICache();
   SCB_EnableDCache();

   //Initialize kernel
   osInitKernel();
   //Configure debug UART
   debugInit(115200);

   //Start-up message
   TRACE_INFO("\r\n");
   TRACE_INFO("***********************************\r\n");
   TRACE_INFO("*** CycloneTCP MQTT Client Demo ***\r\n");
   TRACE_INFO("***********************************\r\n");
   TRACE_INFO("Copyright: 2010-2024 Oryx Embedded SARL\r\n");
   TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
   TRACE_INFO("Target: STM32F769\r\n");
   TRACE_INFO("Version: %s\r\n",APP_VERSION_STRING);
   TRACE_INFO("\r\n");

   //LED configuration
   BSP_LED_Init(LED1);
   BSP_LED_Init(LED2);
   BSP_LED_Init(LED3);
   BSP_LED_Init(LED4);

   //Clear LEDs
   BSP_LED_Off(LED1);
   BSP_LED_Off(LED2);
   BSP_LED_Off(LED3);
   BSP_LED_Off(LED4);

   //Initialize user button
   BSP_PB_Init(BUTTON_KEY, BUTTON_MODE_GPIO);

   //Initialize LCD display
   BSP_LCD_Init();
   BSP_LCD_LayerDefaultInit(0, LCD_FRAME_BUFFER_LAYER0);
   BSP_LCD_SelectLayer(0);
   BSP_LCD_SetBackColor(LCD_COLOR_BLUE);
   BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
   BSP_LCD_SetFont(&Font24);
   BSP_LCD_DisplayOn();

   //Clear LCD display
   BSP_LCD_Clear(LCD_COLOR_BLUE);

   //Welcome message
   lcdSetCursor(0, 0);
   printf("MQTT Client Demo\r\n");

   //Initialize potentiometer
   BSP_POTENTIOMETER_Init();

   //Initialize hardware cryptographic accelerator
   error = stm32f7xxCryptoInit();
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize hardware crypto accelerator!\r\n");
   }

   //Generate a random seed
   error = trngGetRandomData(seed, sizeof(seed));
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to generate random data!\r\n");
   }

   //PRNG initialization
   error = yarrowInit(&yarrowContext);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize PRNG!\r\n");
   }

   //Properly seed the PRNG
   error = yarrowSeed(&yarrowContext, seed, sizeof(seed));
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to seed PRNG!\r\n");
   }

   //TCP/IP stack initialization
   error = netInit();
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize TCP/IP stack!\r\n");
   }

   //Configure the first Ethernet interface
   interface = &netInterface[0];

   //Set interface name
   netSetInterfaceName(interface, APP_IF_NAME);
   //Set host name
   netSetHostname(interface, APP_HOST_NAME);
   //Set host MAC address
   macStringToAddr(APP_MAC_ADDR, &macAddr);
   netSetMacAddr(interface, &macAddr);
   //Select the relevant network adapter
   netSetDriver(interface, &stm32f7xxEthDriver);
   netSetPhyDriver(interface, &dp83848PhyDriver);
   //MDIO and MDC are managed by software (default jumper configuration)
   netSetSmiDriver(interface, &smiDriver);

   //Initialize network interface
   error = netConfigInterface(interface);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to configure interface %s!\r\n", interface->name);
   }

#if (IPV4_SUPPORT == ENABLED)
#if (APP_USE_DHCP_CLIENT == ENABLED)
   //Get default settings
   dhcpClientGetDefaultSettings(&dhcpClientSettings);
   //Set the network interface to be configured by DHCP
   dhcpClientSettings.interface = interface;
   //Disable rapid commit option
   dhcpClientSettings.rapidCommit = FALSE;

   //DHCP client initialization
   error = dhcpClientInit(&dhcpClientContext, &dhcpClientSettings);
   //Failed to initialize DHCP client?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize DHCP client!\r\n");
   }

   //Start DHCP client
   error = dhcpClientStart(&dhcpClientContext);
   //Failed to start DHCP client?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to start DHCP client!\r\n");
   }
#else
   //Set IPv4 host address
   ipv4StringToAddr(APP_IPV4_HOST_ADDR, &ipv4Addr);
   ipv4SetHostAddr(interface, ipv4Addr);

   //Set subnet mask
   ipv4StringToAddr(APP_IPV4_SUBNET_MASK, &ipv4Addr);
   ipv4SetSubnetMask(interface, ipv4Addr);

   //Set default gateway
   ipv4StringToAddr(APP_IPV4_DEFAULT_GATEWAY, &ipv4Addr);
   ipv4SetDefaultGateway(interface, ipv4Addr);

   //Set primary and secondary DNS servers
   ipv4StringToAddr(APP_IPV4_PRIMARY_DNS, &ipv4Addr);
   ipv4SetDnsServer(interface, 0, ipv4Addr);
   ipv4StringToAddr(APP_IPV4_SECONDARY_DNS, &ipv4Addr);
   ipv4SetDnsServer(interface, 1, ipv4Addr);
#endif
#endif

#if (IPV6_SUPPORT == ENABLED)
#if (APP_USE_SLAAC == ENABLED)
   //Get default settings
   slaacGetDefaultSettings(&slaacSettings);
   //Set the network interface to be configured
   slaacSettings.interface = interface;

   //SLAAC initialization
   error = slaacInit(&slaacContext, &slaacSettings);
   //Failed to initialize SLAAC?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize SLAAC!\r\n");
   }

   //Start IPv6 address autoconfiguration process
   error = slaacStart(&slaacContext);
   //Failed to start SLAAC process?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to start SLAAC!\r\n");
   }
#else
   //Set link-local address
   ipv6StringToAddr(APP_IPV6_LINK_LOCAL_ADDR, &ipv6Addr);
   ipv6SetLinkLocalAddr(interface, &ipv6Addr);

   //Set IPv6 prefix
   ipv6StringToAddr(APP_IPV6_PREFIX, &ipv6Addr);
   ipv6SetPrefix(interface, 0, &ipv6Addr, APP_IPV6_PREFIX_LENGTH);

   //Set global address
   ipv6StringToAddr(APP_IPV6_GLOBAL_ADDR, &ipv6Addr);
   ipv6SetGlobalAddr(interface, 0, &ipv6Addr);

   //Set default router
   ipv6StringToAddr(APP_IPV6_ROUTER, &ipv6Addr);
   ipv6SetDefaultRouter(interface, 0, &ipv6Addr);

   //Set primary and secondary DNS servers
   ipv6StringToAddr(APP_IPV6_PRIMARY_DNS, &ipv6Addr);
   ipv6SetDnsServer(interface, 0, &ipv6Addr);
   ipv6StringToAddr(APP_IPV6_SECONDARY_DNS, &ipv6Addr);
   ipv6SetDnsServer(interface, 1, &ipv6Addr);
#endif
#endif

   //Set task parameters
   taskParams = OS_TASK_DEFAULT_PARAMS;
   taskParams.stackSize = 500;
   taskParams.priority = OS_TASK_PRIORITY_NORMAL;

   //Create user task
   taskId = osCreateTask("User", userTask, NULL, &taskParams);
   //Failed to create the task?
   if(taskId == OS_INVALID_TASK_ID)
   {
      //Debug message
      TRACE_ERROR("Failed to create task!\r\n");
   }

   //Set task parameters
   taskParams = OS_TASK_DEFAULT_PARAMS;
   taskParams.stackSize = 750;
   taskParams.priority = OS_TASK_PRIORITY_NORMAL;

   //Create MQTT test task
   taskId = osCreateTask("MQTT", mqttTestTask, NULL, &taskParams);
   //Failed to create the task?
   if(taskId == OS_INVALID_TASK_ID)
   {
      //Debug message
      TRACE_ERROR("Failed to create task!\r\n");
   }

   //Set task parameters
   taskParams = OS_TASK_DEFAULT_PARAMS;
   taskParams.stackSize = 200;
   taskParams.priority = OS_TASK_PRIORITY_NORMAL;

   //Create a task to blink the LED
   taskId = osCreateTask("LED", ledTask, NULL, &taskParams);
   //Failed to create the task?
   if(taskId == OS_INVALID_TASK_ID)
   {
      //Debug message
      TRACE_ERROR("Failed to create task!\r\n");
   }

   //Start the execution of tasks
   osStartKernel();

   //This function should never return
   return 0;
}

/**
 * @details This function get and return an uint32_t value from 4 bytes of an
 * array in Big Endian order.
 */
uint32_t big_endian_u32_read_from_array(const uint8_t* array)
{
	return (
        (uint32_t)(array[0] << 24U)
	  | (uint32_t)(array[1] << 16U)
	  | (uint32_t)(array[2] << 8U)
	  | (uint32_t)(array[3])
    );
}

/**
 * @details This function adds the provided uint32_t value into an array in
 * Big Endian order.
 */
void big_endian_u32_write_to_array(uint32_t u32_value, uint8_t* array) {
    array[0] = (uint8_t)(u32_value >> 24U);
    array[1] = (uint8_t)(u32_value >> 16U);
    array[2] = (uint8_t)(u32_value >> 8U);
    array[3] = (uint8_t)u32_value;
}

/**
 * @details This function converts a buffer of data from big endian to little endian.
 * It swaps the byte order in chunks of 4 bytes (32-bit words).
 *
 * @param input_buffer Pointer to the input buffer in big endian.
 * @param output_buffer Pointer to the output buffer to store little endian data.
 * @param length The length of the input buffer in bytes. Must be a multiple of 4.
 */
void big_to_little_endian_transform(const uint8_t* input_buffer, uint8_t* output_buffer, size_t length)
{
    // Ensure the buffer length is a multiple of 4
    if (length % 4 != 0) {
        // Handle error: Invalid length
        return;
    }

    for (size_t i = 0; i < length; i += 4) {
        output_buffer[i]     = input_buffer[i + 3]; // Swap the first and last bytes
        output_buffer[i + 1] = input_buffer[i + 2]; // Swap the second and third bytes
        output_buffer[i + 2] = input_buffer[i + 1];
        output_buffer[i + 3] = input_buffer[i];
    }
}
