/**
 * @file boot.c
 * @brief CycloneBOOT Bootloader management
 *
 * @section License
 *
 * Copyright (C) 2021-2025 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneBOOT Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.5.2
 **/

// Switch to the appropriate trace level
#define TRACE_LEVEL TRACE_LEVEL_INFO

// Dependencies
#include "second_stage/boot.h"

#include "core/flash.h"
#include "image/image.h"
#include "second_stage/boot_common.h"
#include "second_stage/boot_fallback.h"
#if ((BOOT_FALLBACK_SUPPORT == DISABLED) && (BOOT_EXT_MEM_ENCRYPTION_SUPPORT == ENABLED) ||        \
   (BOOT_COUNTER_SUPPORT == ENABLED))
#include "core/mailbox.h"
#endif
#include "core/cboot_error.h"
#include "core/mailbox.h"
#include "debug.h"

// Bootloader private-related functions
cboot_error_t bootGetCipherKey(BootContext *context);
cboot_error_t bootJumpToApp(BootContext *context);

void bootHandleFallbackTriggers(BootContext *context);

/**
 * @brief Initialize bootloader settings with default values
 * @param[in,out] settings Structure that contains Bootloader settings
 **/

void bootGetDefaultSettings(BootSettings *settings)
{
   // Clear bootloader user settings structure
   memset(settings, 0x00, sizeof(BootSettings));

#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_EXT_MEM_ENCRYPTION_SUPPORT == ENABLED)
   // Secondary flash cipher key settings
   settings->psk = NULL;
   settings->pskSize = 0;
#endif
#endif
}

/**
 * @brief Initialize bootloader context
 * @param[in,out] context Bootloader context
 * @param[in] settings Bootloader user settings
 * @return Status code
 **/

cboot_error_t bootInit(BootContext *context, BootSettings *settings)
{
   cboot_error_t cerror;

   // Check parameter validity
   if(context == NULL || settings == NULL)
      return CBOOT_ERROR_INVALID_PARAMETERS;

   // Set context fields to zero
   memset(context, 0, sizeof(BootContext));

   // Save bootloader user settings
   memcpy(&context->settings, settings, sizeof(BootSettings));

   // User defined initialization hook
   bootInitHook();

   // Initialize a primary flash driver and slots
   cerror = bootInitPrimaryMem(context, settings);
   // Is any error?
   if(cerror)
      return cerror;

#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_FALLBACK_AUTO_MODE == ENABLED)
   mailBoxInit();
#endif
#endif

#if EXTERNAL_MEMORY_SUPPORT == ENABLED
   // Initialize a secondary (external) flash driver and slots
   cerror = bootInitSecondaryMem(context, settings);
   // Is any error?
   if(cerror)
      return cerror;
#endif

#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_EXT_MEM_ENCRYPTION_SUPPORT == ENABLED)
   // Check the cipher key used to decode data in secondary flash (external
   // memory)
   if(settings->psk == NULL || (settings->pskSize > sizeof(context->psk)))
   {
      return CBOOT_ERROR_INVALID_PARAMETERS;
   }
   else
   {
      // Store the cipher key used to decode data in secondary flash (external
      // memory)
      memcpy(context->psk, settings->psk, settings->pskSize);
      context->pskSize = settings->pskSize;
   }
#endif

#ifdef BOOT_FALLBACK_MANUAL_MODE
   // Initialize fallback trigger
   cerror = fallbackTriggerInit();
   // Is any error?
   if(cerror)
      return cerror;
#endif

#endif

   // Initialize bootloader state
   context->state = BOOT_STATE_IDLE;

   // Successful process
   return CBOOT_NO_ERROR;
}

/**
 * @brief Bootloader Task routine
 * @param[in] context Pointer to Bootloader context
 * @return None
 **/

cboot_error_t bootFsm(BootContext *context)
{
   cboot_error_t cerror = CBOOT_NO_ERROR;
   uint32_t appStartAddr;

#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_FALLBACK_MANUAL_MODE == ENABLED)
   TriggerStatus trigStatus;
   // Manual trigger, implemented in user code
   cerror = fallbackTriggerGetStatus(&trigStatus);
   if(cerror)
   {
      TRACE_ERROR("bootFsm: failed to get fallback trigger status.\r\n");
      bootHandleFallbackError();
   }
   else
   {
      if(trigStatus == TRIGGER_STATUS_RAISED)
      {
         TRACE_INFO("bootFsm: initiating fallback...\r\n");
         bootChangeState(context, BOOT_STATE_FALLBACK_APP);
      }
      else if(mailBoxIsFallbackRequested())
      {
         TRACE_INFO("bootFsm: initiating fallback...\r\n");
         mailBoxSetFallbackRequested(FALSE);    // Avoid fallback loops
         bootChangeState(context, BOOT_STATE_FALLBACK_APP);
      }
      else
      {
         // Just for sanity
      }
   }
#endif
#if (BOOT_FALLBACK_AUTO_MODE == ENABLED)
#if (BOOT_COUNTER_SUPPORT == ENABLED)
   if(mailBoxGetBootCounter() >= BOOT_COUNTER_MAX_ATTEMPTS)
   {
      TRACE_INFO("bootFsm: max boot count reached. Initiating fallback...\r\n");
      mailBoxClearBootCounter();
      bootChangeState(context, BOOT_STATE_FALLBACK_APP);
   }
#endif
#if (BOOT_FLAG_SUPPORT == ENABLED)
   if((!mailBoxIsUpdateConfirmed() && mailBoxGetBootCounter() != 0) ||
      mailBoxIsFallbackRequested())
   {
      TRACE_INFO("bootFsm: fallback requested or update is not confirmed. "
         "Initiating fallback...\r\n");
      mailBoxClearBootCounter();
      bootChangeState(context, BOOT_STATE_FALLBACK_APP);
   }
#endif
#endif
#endif

   do
   {
      context->busy = FALSE;

      switch(context->state)
      {
      case BOOT_STATE_IDLE:
         bootIdleStateHook();
         // Select update image slot
         cerror = bootSelectUpdateImageSlot(context, &context->selectedSlot);
         // Is any error?
         if (cerror == CBOOT_ERROR_FIRMWARE_CORRUPTED) {
#if BOOT_FALLBACK_SUPPORT == ENABLED
            //Current firmware corrupted. Initiate fallback.
            bootChangeState(context, BOOT_STATE_FALLBACK_APP);
#endif
         }
         else if(cerror || context->selectedSlot.memParent == NULL)
         {
            TRACE_ERROR("bootFsm: No valid update image found.\r\n");
            bootNoValidUpdatesHook();
         }
         else
         {
            TRACE_DEBUG("Selected slot:\r\n");
            TRACE_DEBUG("- address:  0x%08lX\r\n", (unsigned long)context->selectedSlot.addr);
            TRACE_DEBUG("- size:     0x%08X\r\n", context->selectedSlot.size);

            // Is selected slot different from slot containing current application?
            // In other words, is selected image (in selected slot) more recent than
            // current application image?
            uint8_t result;
            memoryCompareSlot(&context->selectedSlot, &context->memories[0].slots[0], &result);
            if(result)
            {
               bootChangeState(context, BOOT_STATE_UPDATE_APP);
            }
            else
            {
               bootChangeState(context, BOOT_STATE_RUN_APP);
            }
         }
         break;
      case BOOT_STATE_RUN_APP:
         // Make sure to select first primary memory slot (contains current
         // application)
         context->selectedSlot = context->memories[0].slots[0];

         // Debug message
         TRACE_INFO("bootFsm: No update available...\r\n");
         TRACE_INFO("bootFsm: Checking current application image...\r\n");

         // Check current application image inside first primary memory slot
         cerror = bootCheckImage(context, &context->selectedSlot);
         // Is any error?
         if(cerror)
         {
#if BOOT_FALLBACK_SUPPORT == ENABLED
            // Change bootloader state
            if (cerror == CBOOT_ERROR_FIRMWARE_CORRUPTED)
               bootChangeState(context, BOOT_STATE_FALLBACK_APP);
#else
            // Change bootloader state
            bootChangeState(context, BOOT_STATE_ERROR);
#endif

         }
         else
         {
            // Check reset vector of the current application
            cerror = bootCheckSlotAppResetVector(&context->selectedSlot);
            // Is reset vector valid?
            if(!cerror)
            {
               // Debug message
               TRACE_INFO("bootFsm: Current application image is valid.\r\n");
               TRACE_INFO("bootFsm: Incrementing boot counter...\r\n");
               TRACE_INFO("bootFsm: Booting to the application...\r\n");

#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_FALLBACK_AUTO_MODE == ENABLED)
               // Update boot counter
               mailBoxIncrementBootCounter();
#endif
#endif

               // Compute application start address
               appStartAddr = context->selectedSlot.addr + mcuGetVtorOffset();

               // Jump to current application inside primary memory slot
               mcuJumpToApplication(appStartAddr);
            }
            else
            {
               // Change bootloader state
               bootChangeState(context, BOOT_STATE_ERROR);
            }
         }
         break;
      case BOOT_STATE_UPDATE_APP:
         TRACE_INFO("bootFsm: Checking update application image...\r\n");

#if (BOOT_FALLBACK_SUPPORT == DISABLED && BOOT_EXT_MEM_ENCRYPTION_SUPPORT == ENABLED)
         // Call bootGetCipherKey here
         cerror = bootGetCipherKey(context);

         // Is any error?
         if(cerror)
         {
            // Debug message
            TRACE_ERROR("Failed to retrieve cipher key!\r\n");
            // Discard error
            cerror = CBOOT_NO_ERROR;
            // Change bootloader state
            bootChangeState(context, BOOT_STATE_RUN_APP);
         }
         else
#else
         if(1)
#endif
            // Check current application image inside first primary memory slot
            cerror = bootCheckImage(context, &context->selectedSlot);

         // Is any error?
         if(cerror)
         {
            // Discard error
            cerror = CBOOT_NO_ERROR;
            // Change bootloader state
            bootChangeState(context, BOOT_STATE_RUN_APP);
         }
         else
         {
            // Debug message
            TRACE_INFO("bootFsm: starting update procedure...\r\n");

            // Start update procedure (could be a new application or because of a
            // previous fallback procedure)
            cerror = bootUpdateApp(context, &context->selectedSlot);

            // Is any error?
            if(cerror)
            {
               // Change bootloader state
               bootChangeState(context, BOOT_STATE_ERROR);
            }
            else
            {
#if (BOOT_FALLBACK_SUPPORT == ENABLED)
#if (BOOT_FALLBACK_AUTO_MODE == ENABLED)
               mailBoxSetUpdateConfirmed(FALSE);
               mailBoxClearBootCounter();
#endif
#endif

               // Debug message
               TRACE_INFO("bootFsm: update procedure finished\r\n");
               TRACE_INFO("Rebooting...\r\n");

               // Reset system
               mcuSystemReset();
            }
         }
         break;
#if (BOOT_FALLBACK_SUPPORT == ENABLED)
      case BOOT_STATE_FALLBACK_APP:
         // Call fallback routine here
         cerror = fallbackTask(context, context->memories);

         // Is any error.
         if(cerror)
         {
            // Debug message
            TRACE_INFO("Fallback procedure failed!\r\n");

            // Call error handler
            bootHandleFallbackError();

            // Change bootloader state
            bootChangeState(context, BOOT_STATE_RUN_APP);
         }
         else
         {
            // Debug message
            TRACE_INFO("bootFsm: fallback procedure finished\r\n");
            TRACE_INFO("Rebooting...\r\n");
            // Reset device
            mcuSystemReset();
         }
         break;
#endif
      case BOOT_STATE_ERROR:
         // Delegate the error handling to a user defined function
         bootHandleGenericError();
         break;
      default:
         cerror = CBOOT_ERROR_INVALID_VALUE;
         break;
      }

   } while(context->busy);

   return cerror;
}

/**
 * @brief Get PSK cipher key used to encrypt output image in external flash
 * memory.
 * @param[in,out] context Pointer to the bootloader context
 **/

cboot_error_t bootGetCipherKey(BootContext *context)
{
#if ((BOOT_FALLBACK_SUPPORT == DISABLED) && (BOOT_EXT_MEM_ENCRYPTION_SUPPORT == ENABLED))
   cboot_error_t cerror;
   uint8_t psk[BOOT_MBX_PSK_MAX_SIZE];
   size_t pskSize = 0;

   // Initialize status code
   cerror = CBOOT_NO_ERROR;

   // Debug message
   TRACE_INFO("Retrieving cipher key...\r\n");

   // Begin of handling block
   do
   {
      if(context->settings.psk == NULL && context->settings.pskSize == 0)
      {
         // Get message from shared SRAM (contains PSK key)
         mailBoxGetPsk(psk, &pskSize);

         // Check cipher key used to decode data in secondary flash (external
         // memory)
         if(pskSize > sizeof(context->psk) || pskSize == 0)
         {
            TRACE_ERROR("Retrieving cipher key failed!\r\n");
            break;     // TODO: in addition to breaking, we should not attempt to
                       // decrypt with an invalid key
         }

         // Store cipher key used to decode data in secondary flash (external
         // memory)
         memcpy(context->psk, psk, pskSize);
         context->pskSize = pskSize;
      }
      else
      {
         // Get PSK from the settings
         memcpy(context->psk, context->settings.psk, context->settings.pskSize);
         context->pskSize = context->settings.pskSize;
      }

   } while(0);

   // Make sure to reset message from shared RAM memory
   mailBoxClearPsk();

   // Return status code
   return cerror;
#else
   // Return error code
   return CBOOT_ERROR_NOT_IMPLEMENTED;
#endif
}

/**
 * @brief Jump to the application binary inside the current image in internal
 * flash.
 * @input[in] context Pointer to the bootloader context
 * @return Status code
 **/

cboot_error_t bootJumpToApp(BootContext *context)
{
   error_t error;
   const FlashInfo *info;
   uint32_t mcuVtorOffset;
   FlashDriver *driver;

   // Check parameter validity
   if(context == NULL)
      return CBOOT_ERROR_INVALID_PARAMETERS;

   // Get primary flash memory information
   driver = (FlashDriver *)context->memories[0].driver;
   error = driver->getInfo(&info);
   // Is any error?
   if(error)
      return CBOOT_ERROR_FAILURE;

   // Get MCU VTOR offset
   mcuVtorOffset = mcuGetVtorOffset();

   // Jump to application at given address
   mcuJumpToApplication(info->flashAddr + BOOT_OFFSET + mcuVtorOffset);

   // Successful process
   return CBOOT_NO_ERROR;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

#if ((defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)) || defined(__GNUC__) ||            \
   defined(__CC_ARM) || defined(__IAR_SYSTEMS_ICC__) || defined(__TASKING__) ||                  \
   defined(__CWCC__) || defined(__TI_ARM__))
__weak_func void bootInitHook(void)
{
   /*
    * This function will be called when the bootloader is initialized.
    * It can be redefined in user code.
    */
}

__weak_func void bootIdleStateHook(void)
{
   /*
    * This function will be called when the bootloader enter to IDLE state.
    * It can be redefined in user code. Note: this function can be called more
    * than once.
    */
}
__weak_func void bootNoValidUpdatesHook(void)
{
   /*
    * This function will be called when the bootFsm determines there are not
    * valid images to be installed.. It can be redefined in user code. Note: this
    * function can be called more than once, at each start up.
    */
}
__weak_func void bootHandleFallbackError(void)
{
   /*
    * This function will be called when the bootloader cannot complete the
    * fallback process. It can be redefined in user code.
    */
}

__weak_func void bootHandleGenericError(void)
{
   /*
    * This function will be called when the bootloader encounters a generic error
    * state. It can be redefined in user code.
    */
}
#endif
