/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h5xx_nucleo.h"
#include "os_port.h"
#include "boot_config.h"
#include "drivers/flash/internal/stm32h5xx_flash_driver.h"
#include "image/image.h"
#include "second_stage/boot.h"
#include "memory/memory.h"
#include "second_stage/boot_fallback.h"
#include "core/mailbox.h"
#include "debug.h"

uint8_t ledState = 0;
systime_t ledDelay = 0;
systime_t ledTimestamp = 0;

uint32_t get_edata_sector(uint32_t address);
uint32_t get_edata_bank(uint32_t address);

/*
 * For example, if 48 Kbytes of data are needed in Bank 1, set EDATA1_EN to 1, and EDATA1_STRT to 7.
 * If no data are needed in Bank2, set EDATA2_EN to 0. In this case the data are accessible from
 * address 0x0900_0000 to 0x0900_BFFF for Bank 1
 *
 * For greater efficiency, it is recommended to use sector on the other bank for flash high-cycle data,
 * so that the application benefits from RWW capability of the dual-bank arrangement.
 * When SWAP_BANK feature is enabled, the banks are swapped: the flash high-cycle data in Bank 2 are accessible
 * from 0x0900_000 to 0x0900_BFFF, and the data in Bank 1 are accessible from 0x0900_C000 to 0x0901_7FFF
 */

//Bank 1 addresses
#define ADDR_EDATA1_STRT_0     (0x0900A800U) /* Base @ of last sector of Bank1 reserved to EDATA (EDATA1_STRT = 0), 6 Kbytes    */
#define ADDR_EDATA1_STRT_1     (0x09009000U) /* Base @ of last 2 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 1), 6 Kbytes */
#define ADDR_EDATA1_STRT_2     (0x09007800U) /* Base @ of last 3 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 2), 6 Kbytes */
#define ADDR_EDATA1_STRT_3     (0x09006000U) /* Base @ of last 4 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 3), 6 Kbytes */
#define ADDR_EDATA1_STRT_4     (0x09004800U) /* Base @ of last 5 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 4), 6 Kbytes */
#define ADDR_EDATA1_STRT_5     (0x09003000U) /* Base @ of last 6 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 5), 6 Kbytes */
#define ADDR_EDATA1_STRT_6     (0x09001800U) /* Base @ of last 7 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 6), 6 Kbytes */
#define ADDR_EDATA1_STRT_7     (0x09000000U) /* Base @ of last 8 sectors of Bank1 reserved to EDATA (EDATA1_STRT = 7), 6 Kbytes */
#define ADDR_EDATA1_END        (0x0900BFFFU)

//Bank 2 addresses
#define ADDR_EDATA2_STRT_0     (0x09016800U) /* Base @ of last sector of Bank2 reserved to EDATA (EDATA2_STRT = 0), 6 Kbytes    */
#define ADDR_EDATA2_STRT_1     (0x09015000U) /* Base @ of last 2 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 1), 6 Kbytes */
#define ADDR_EDATA2_STRT_2     (0x09013800U) /* Base @ of last 3 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 2), 6 Kbytes */
#define ADDR_EDATA2_STRT_3     (0x09012000U) /* Base @ of last 4 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 3), 6 Kbytes */
#define ADDR_EDATA2_STRT_4     (0x09000800U) /* Base @ of last 5 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 4), 6 Kbytes */
#define ADDR_EDATA2_STRT_5     (0x0900F000U) /* Base @ of last 6 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 5), 6 Kbytes */
#define ADDR_EDATA2_STRT_6     (0x0900D800U) /* Base @ of last 7 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 6), 6 Kbytes */
#define ADDR_EDATA2_STRT_7     (0x0900C000U) /* Base @ of last 8 sectors of Bank2 reserved to EDATA (EDATA2_STRT = 7), 6 Kbytes */
#define ADDR_EDATA2_END        (0x09017FFFU)

#define BOOT_MBX_NVM_START_ADDR ADDR_EDATA2_STRT_0
#define BOOT_MBX_NVM_END_ADDR   ADDR_EDATA2_END

#if 1
/**
 * @brief MPU configuration
 */
void MPU_Config(void)
{
  HAL_MPU_Disable();

  // configure the MPU to prohibit access to after the .bss
  // this will trigger an exception if the stack grows too large
  MPU_Region_InitTypeDef MPU_InitStruct;
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER0;
  MPU_InitStruct.BaseAddress = 0x08fff800;
  MPU_InitStruct.LimitAddress = 0x08fff900;
  MPU_InitStruct.AttributesIndex = MPU_ATTRIBUTES_NUMBER0;
  MPU_InitStruct.AccessPermission = MPU_REGION_ALL_RO;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_OUTER_SHAREABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}
#else
/**
  * @brief  Configure the MPU attributes as non-cacheable for Flash high-cycle data area
  * @note   The Base Address is Flash high-cycle data area
  * @param  None
  * @retval None
  */
void MPU_Config(void)
{

  /*
   * Rationale:
   * https://community.st.com/t5/stm32-mcus-products/stm32h5-high-cycle-data-read-more-than-16bit-at-once/td-p/584258
   */
  MPU_Attributes_InitTypeDef   attr;
  MPU_Region_InitTypeDef       region;

  /* Disable MPU before perloading and config update */
  HAL_MPU_Disable();

  /* Define cacheable memory via MPU */
  attr.Number             = MPU_ATTRIBUTES_NUMBER0;
  attr.Attributes         = 0 ;
  HAL_MPU_ConfigMemoryAttributes(&attr);

  /* BaseAddress-LimitAddress configuration */
  region.Enable           = MPU_REGION_ENABLE;
  region.Number           = MPU_REGION_NUMBER0;
  region.AttributesIndex  = MPU_ATTRIBUTES_NUMBER0;
  region.BaseAddress      = BOOT_MBX_NVM_START_ADDR;
  region.LimitAddress     = BOOT_MBX_NVM_END_ADDR;
  region.AccessPermission = MPU_REGION_ALL_RW;
  region.DisableExec      = MPU_INSTRUCTION_ACCESS_DISABLE;
  region.IsShareable      = MPU_ACCESS_OUTER_SHAREABLE;
  HAL_MPU_ConfigRegion(&region);

  /* Enable the MPU */
  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}
#endif
/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void)
{
   RCC_OscInitTypeDef RCC_OscInitStruct = {0};
   RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
   RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};

   //Configure the main internal regulator output voltage
   __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

   //Wait for the voltage scaling ready flag to be set
   while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY))
   {
   }

   //Configure PLL with HSI as source
   RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_HSI48;
   RCC_OscInitStruct.HSIState = RCC_HSI_ON;
   RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;
   RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
   RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
   RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
   RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSI;
   RCC_OscInitStruct.PLL.PLLM = 8;
   RCC_OscInitStruct.PLL.PLLN = 60;
   RCC_OscInitStruct.PLL.PLLP = 2;
   RCC_OscInitStruct.PLL.PLLQ = 2;
   RCC_OscInitStruct.PLL.PLLR = 2;
   RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_3;
   RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
   RCC_OscInitStruct.PLL.PLLFRACN = 0;
   HAL_RCC_OscConfig(&RCC_OscInitStruct);

   //Select PLL as system clock source and configure bus clocks dividers
   RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
      RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
   RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
   RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
   RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
   HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

   //Select clock source for RNG peripheral
   RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
   RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
   HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask()
{
	if((uint32_t) (systemTicks - ledTimestamp) >= ledDelay)
	{
		//Toggle LED state
		if(ledState == 0)
		{
			BSP_LED_On(LED1);
			ledState = 1;
			ledDelay = 100;
		}
		else
		{
			BSP_LED_Off(LED1);
			ledState = 0;
			ledDelay = 900;
		}

		ledTimestamp = systemTicks;
	}
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void)
{

	BootSettings bootSettings;
	BootContext bootContext;

   cboot_error_t cerror;

   //HAL library initialization
   HAL_Init();
   //Configure the system clock
   SystemClock_Config();
   // Configure MPU
   MPU_Config();

   //Enable instruction cache
   //HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
   //HAL_ICACHE_Enable();

   //Initialize kernel
   osInitKernel();
   //Configure debug UART
   debugInit(115200);

   //Start-up message
   TRACE_INFO("\r\n");
   TRACE_INFO("******************************************\r\n");
   TRACE_INFO("*** CycloneBOOT Single Bank Bootloader ***\r\n");
   TRACE_INFO("******************************************\r\n");
   TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
   TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
   TRACE_INFO("Target: STM32H563\r\n");
   TRACE_INFO("\r\n");

   //LED configuration
   BSP_LED_Init(LED1);
   BSP_LED_Init(LED2);
   BSP_LED_Init(LED3);

   //Clear LEDs
   BSP_LED_Off(LED1);
   BSP_LED_Off(LED2);
   BSP_LED_Off(LED3);

   //Set up High-Cycle Flash Data zone
   uint32_t sector_start = 0, sector_end = 0, nb_sectors = 0;
   FLASH_OBProgramInitTypeDef FLASH_OBInitStruct;

#if 0
   //Set up the zone
   HAL_FLASH_Unlock();
   HAL_FLASH_OB_Unlock();

   sector_start = get_edata_sector(BOOT_MBX_NVM_START_ADDR);
   sector_end = get_edata_sector(BOOT_MBX_NVM_END_ADDR);
   nb_sectors = sector_end - sector_start + 1;

   //Set up 8 sectors for the purpose (we certainly will need less)
   FLASH_OBInitStruct.OptionType = OPTIONBYTE_EDATA;
   FLASH_OBInitStruct.Banks = get_edata_bank(BOOT_MBX_NVM_START_ADDR);
   FLASH_OBInitStruct.EDATASize = nb_sectors;
   if(HAL_FLASHEx_OBProgram(&FLASH_OBInitStruct) != HAL_OK)
   {
	   TRACE_ERROR("Failed to set up zone...\r\n");
   }

   HAL_FLASH_OB_Launch();
   HAL_FLASH_OB_Lock();
#endif

   //Begining of handling exception
   do
   {
	   cerror = CBOOT_NO_ERROR;

	   //Get default booloader user settings
	   bootGetDefaultSettings(&bootSettings);

	   //User settings primary memory configuration
	   bootSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
	   bootSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
	   bootSettings.memories[0].driver = &stm32h5xxFlashDriver;
	   bootSettings.memories[0].nbSlots = 3;
	   //User settings primary memory slot 0 configuration
	   bootSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
	   bootSettings.memories[0].slots[0].cType = SLOT_CONTENT_BINARY;
	   bootSettings.memories[0].slots[0].memParent = &bootSettings.memories[0];
	   bootSettings.memories[0].slots[0].addr = 0x08020000;
	   bootSettings.memories[0].slots[0].size = 0x80000;//0x9E000;//0xA0000;
	   //User settings primary memory slot 1 configuration
	   bootSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
	   bootSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
	   bootSettings.memories[0].slots[1].memParent = &bootSettings.memories[0];
	   bootSettings.memories[0].slots[1].addr = 0x080C0000;
	   bootSettings.memories[0].slots[1].size = 0x80000;//0x9E000;//0xA0000;
	   //User settings primary memory slot 2 configuration
	   bootSettings.memories[0].slots[2].type = SLOT_TYPE_DIRECT;
	   bootSettings.memories[0].slots[2].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
	   bootSettings.memories[0].slots[2].memParent = &bootSettings.memories[0];
	   bootSettings.memories[0].slots[2].addr = 0x08160000;
	   bootSettings.memories[0].slots[2].size = 0x80000;//0x9E000;//0xA0000;

	   //Initialize bootloader
	   cerror = bootInit(&bootContext, &bootSettings);
	   //Is any error?
	   if(cerror)
		   break;
   }while(0);

   //Is any error?
   if(cerror)
   {
	   //Debug message
	   TRACE_ERROR("Bootloader configuration failed!\r\n");
	   while(1);
   }

   while(1)
   {
	   ledTask();
	   cerror = bootFsm(&bootContext);
	   //Is any error?
	   if(cerror)
	   {
		   //Debug message
		   TRACE_ERROR("Bootloader failure!\r\n");
	   }
   }

   //This function should never return
   return 0;
}

cboot_error_t mbx_set_nvm_flag(uint32_t flag, bool_t value) {
	FLASH_EraseInitTypeDef EraseInitStruct = {0};
	uint32_t sector_error;

	uint32_t sector_start = get_edata_sector(BOOT_MBX_NVM_START_ADDR);
	uint32_t sector_end = get_edata_sector(BOOT_MBX_NVM_END_ADDR);
	uint32_t nb_sectors = sector_end - sector_start + 1;

	//Erase the zone to clear any existing data
	EraseInitStruct.TypeErase = FLASH_TYPEERASE_SECTORS;
	EraseInitStruct.Banks = get_edata_bank(BOOT_MBX_NVM_START_ADDR);
	EraseInitStruct.Sector = sector_start;
	EraseInitStruct.NbSectors = nb_sectors;

	HAL_FLASH_Unlock();

	if (HAL_FLASHEx_Erase(&EraseInitStruct, &sector_error) != HAL_OK)
	{
		/*
		      Error occurred while sector erase.
		      User can add here some code to deal with this error.
		      SectorError will contain the faulty sector and then to know the code error on this sector,
		      user can call function 'HAL_FLASH_GetError()'
		 */
		TRACE_ERROR("Failed to erase sectors...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	uint16_t boot_magic[1] = {BOOT_MBX_NVM_MAGIC};

	if(HAL_FLASH_Program(FLASH_TYPEPROGRAM_HALFWORD_EDATA, BOOT_MBX_NVM_START_ADDR, (uint32_t)&boot_magic[0]) != HAL_OK)
	{
		TRACE_ERROR("Failed to write magic value...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	// Write the flag value
	uint32_t flagValue = value ? flag : 0xFFFFFFFF;
	if(HAL_FLASH_Program(FLASH_TYPEPROGRAM_HALFWORD_EDATA, BOOT_MBX_NVM_START_ADDR + 4, (uint32_t)&flagValue) != HAL_OK)
	{
		TRACE_ERROR("Failed to write flag value in NVM...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	HAL_FLASH_Lock();

	return CBOOT_NO_ERROR;
}

bool_t mbx_get_nvm_flag(uint32_t flag) {
	uint16_t storedMagic = *(volatile uint16_t*) BOOT_MBX_NVM_START_ADDR;
	uint16_t storedFlag  = *(volatile uint16_t*) (BOOT_MBX_NVM_START_ADDR + 4);
	uint16_t nvmFlag = BOOT_MBX_NVM_FLAG_UPDATE_CONFIRMED;

	// Validate storage
	if (storedMagic != BOOT_MBX_NVM_MAGIC) {
		return FALSE;
	}

	if(flag == UPDATE_CONFIRMED) {
		flag = nvmFlag;
	}

	return (storedFlag == flag) ? TRUE : FALSE;
}

/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
/**
  * @brief  Gets the sector of a given address
  * @param  Addr: Address of the FLASH Memory
  * @retval The sector of a given address
  */
uint32_t get_edata_sector(uint32_t address)
{
  uint32_t sector = 0;
  uint32_t edataSectorSize = 0;

  /*
    (FLASH_EDATA_SIZE/2) is the size of high-cycle area of flash BANK1.
    Flash high-cycle area have 8 sectors in each Bank.
  */
  edataSectorSize = (FLASH_EDATA_SIZE / 2) / 8;
  /* Check if the address is located in the FLASH high-cycle data area of BANK1 */
  if((address >= FLASH_EDATA_BASE) && (address < FLASH_EDATA_BASE + (FLASH_EDATA_SIZE / 2)))
  {
    sector = (address & ~FLASH_EDATA_BASE) / edataSectorSize;
    sector += 120;
  }
  /* Check if the address is located in the FLASH high-cycle data area of BANK2 */
  else if ((address >= FLASH_EDATA_BASE + (FLASH_EDATA_SIZE / 2)) && (address < FLASH_EDATA_BASE + FLASH_EDATA_SIZE))
  {
    sector = ((address & ~FLASH_EDATA_BASE) - (FLASH_EDATA_SIZE / 2)) / edataSectorSize;
    sector += 120;
  }
  else
  {
    sector = 0xFFFFFFFF; /* Address out of range */
  }

  return sector;
}

/**
  * @brief  Gets the bank of a given address in EDATA area
  * @param  Addr: Address of A given address in EDATA area
  * @retval The bank of a given address in EDATA area
  */
uint32_t get_edata_bank(uint32_t address)
{
  uint32_t bank = 0;

  /* (FLASH_EDATA_SIZE/2) is the size of high-cycle area of flash BANK1 */
  if((address >= FLASH_EDATA_BASE) && (address < FLASH_EDATA_BASE + (FLASH_EDATA_SIZE/2)))
  {
    bank = FLASH_BANK_1;
  }
  else if ((address >= FLASH_EDATA_BASE + (FLASH_EDATA_SIZE/2)) && (address < FLASH_EDATA_BASE + FLASH_EDATA_SIZE))
  {
    bank = FLASH_BANK_2;
  }
  else
  {
    bank = 0xFFFFFFFF; /* Address out of range */
  }
  return bank;
}
