/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

/**
 *
 * Flashing the firmware :
 *
 * - 2nd stage bootloader address - 0x08000000
 * - Initial firmware address - 0x08020000
 *
 * - Use ST-Link or equivalent to flash the bootloader (bootloader.bin) and initial firmware application (iap_demo_bootable.bin) to the MCU internal flash :
 *   E.g.,
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\2nd_stage_bl\bootloader.bin 0x08000000 -halt -rst
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\iap_demo_bootable.bin 0x08020000 -halt -rst

 * Uploading the firmware update :
 *
 * HTTP
 *
 *  - The trace outputs are redirected to USART. Exact configuration is found in debug.c file
 *  - Connect Ethernet port to the board
 *  - Once the application is running, from USART trace find the IP address allocated to the device (reset the MCU if necessary after flashing the firmware)
 *  - Using a web browser navigate to the IP address. E.g., http://192.168.1.42
 *  - Upload the iap_demo_2_0_0.img file using the HTML UI
 *  - After a short while, the MCU should reboot and the HTML UI should reflect the updated version
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h5xx_nucleo.h"
#include "core/net.h"
#include "drivers/mac/stm32h5xx_eth_driver.h"
#include "drivers/phy/lan8742_driver.h"
#include "dhcp/dhcp_client.h"
#include "ipv6/slaac.h"
#include "http/http_server.h"
#include "http/mime.h"
#include "hardware/stm32h5xx/stm32h5xx_crypto.h"
#include "rng/trng.h"
#include "rng/yarrow.h"
#include "path.h"
#include "date_time.h"
#include "resource_manager.h"
#include "boot_config.h"
#include "http_server_callbacks.h"
#include "os_port.h"
#include "version.h"
#include "update/update_fallback.h"
#include "core/mailbox.h"
#include "debug.h"

//Ethernet interface configuration
#define APP_IF_NAME "eth0"
#define APP_HOST_NAME "http-server-demo"
#define APP_MAC_ADDR "00-AB-CD-EF-07-77"

#define APP_USE_DHCP_CLIENT ENABLED
#define APP_IPV4_HOST_ADDR "192.168.0.20"
#define APP_IPV4_SUBNET_MASK "255.255.255.0"
#define APP_IPV4_DEFAULT_GATEWAY "192.168.0.254"
#define APP_IPV4_PRIMARY_DNS "8.8.8.8"
#define APP_IPV4_SECONDARY_DNS "8.8.4.4"

#define APP_USE_SLAAC ENABLED
#define APP_IPV6_LINK_LOCAL_ADDR "fe80::769"
#define APP_IPV6_PREFIX "2001:db8::"
#define APP_IPV6_PREFIX_LENGTH 64
#define APP_IPV6_GLOBAL_ADDR "2001:db8::769"
#define APP_IPV6_ROUTER "fe80::1"
#define APP_IPV6_PRIMARY_DNS "2001:4860:4860::8888"
#define APP_IPV6_SECONDARY_DNS "2001:4860:4860::8844"

//Application configuration
#define APP_HTTP_MAX_CONNECTIONS 8

//Update signature public key resource path
#define UPDT_PUBLIC_SIGN_KEY "keys/rsa_public_key.pem"
//#define UPDT_PUBLIC_SIGN_KEY "keys/ecdsa_public_key_prime256v1.pem"

#define ADDR_EDATA2_STRT_0     (0x09016800U) /* Base @ of last sector of Bank2 reserved to EDATA (EDATA2_STRT = 0), 6 Kbytes    */
#define ADDR_EDATA2_END        (0x09017FFFU)

#define BOOT_MBX_NVM_START_ADDR ADDR_EDATA2_STRT_0
#define BOOT_MBX_NVM_END_ADDR   ADDR_EDATA2_END

DhcpClientSettings dhcpClientSettings;
DhcpClientContext dhcpClientContext;
SlaacSettings slaacSettings;
SlaacContext slaacContext;
HttpServerSettings httpServerSettings;
HttpServerContext httpServerContext;
HttpConnection httpConnections[APP_HTTP_MAX_CONNECTIONS];
NetContext netContext;
NetInterface netInterfaces[1];

//Update signature public key declaration
const uint8_t* pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;

//Update mutex handler
OsMutex updateMutex;

uint32_t get_edata_sector(uint32_t address);
uint32_t get_edata_bank(uint32_t address);

#if 1
/**
 * @brief MPU configuration
 */
void MPU_Config(void)
{
  HAL_MPU_Disable();

  // configure the MPU to prohibit access to after the .bss
  // this will trigger an exception if the stack grows to large
  MPU_Region_InitTypeDef MPU_InitStruct;
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER0;
  MPU_InitStruct.BaseAddress = 0x08fff800;
  MPU_InitStruct.LimitAddress = 0x08fff900;
  MPU_InitStruct.AttributesIndex = MPU_ATTRIBUTES_NUMBER0;
  MPU_InitStruct.AccessPermission = MPU_REGION_ALL_RO;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_OUTER_SHAREABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}
#else
/**
  * @brief  Configure the MPU attributes as non-cacheable for Flash high-cycle data area
  * @note   The Base Address is Flash high-cycle data area
  * @param  None
  * @retval None
  */
void MPU_Config(void)
{

  /*
   * Rationale:
   * https://community.st.com/t5/stm32-mcus-products/stm32h5-high-cycle-data-read-more-than-16bit-at-once/td-p/584258
   */
  MPU_Attributes_InitTypeDef   attr;
  MPU_Region_InitTypeDef       region;

  /* Disable MPU before perloading and config update */
  HAL_MPU_Disable();

  /* Define cacheable memory via MPU */
  attr.Number             = MPU_ATTRIBUTES_NUMBER0;
  attr.Attributes         = 0 ;
  HAL_MPU_ConfigMemoryAttributes(&attr);

  /* BaseAddress-LimitAddress configuration */
  region.Enable           = MPU_REGION_ENABLE;
  region.Number           = MPU_REGION_NUMBER0;
  region.AttributesIndex  = MPU_ATTRIBUTES_NUMBER0;
  region.BaseAddress      = BOOT_MBX_NVM_START_ADDR;
  region.LimitAddress     = BOOT_MBX_NVM_END_ADDR;
  region.AccessPermission = MPU_REGION_ALL_RW;
  region.DisableExec      = MPU_INSTRUCTION_ACCESS_DISABLE;
  region.IsShareable      = MPU_ACCESS_OUTER_SHAREABLE;
  HAL_MPU_ConfigRegion(&region);

  /* Enable the MPU */
  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

#endif
/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void)
{
   RCC_OscInitTypeDef RCC_OscInitStruct = {0};
   RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
   RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};

   //Configure the main internal regulator output voltage
   __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

   //Wait for the voltage scaling ready flag to be set
   while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY))
   {
   }

   //Configure PLL with HSI as source
   RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_HSI48;
   RCC_OscInitStruct.HSIState = RCC_HSI_ON;
   RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;
   RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
   RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
   RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
   RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSI;
   RCC_OscInitStruct.PLL.PLLM = 8;
   RCC_OscInitStruct.PLL.PLLN = 60;
   RCC_OscInitStruct.PLL.PLLP = 2;
   RCC_OscInitStruct.PLL.PLLQ = 2;
   RCC_OscInitStruct.PLL.PLLR = 2;
   RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_3;
   RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
   RCC_OscInitStruct.PLL.PLLFRACN = 0;
   HAL_RCC_OscConfig(&RCC_OscInitStruct);

   //Select PLL as system clock source and configure bus clocks dividers
   RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
      RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
   RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
   RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
   RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
   HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

   //Select clock source for RNG peripheral
   RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
   RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
   HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param)
{
   //Endless loop
   while(1)
   {
      BSP_LED_On(LED1);
      osDelayTask(100);
      BSP_LED_Off(LED1);
      osDelayTask(900);
   }
}

void buttonTask(void *param)
{
	while(1)
	{
		if(BSP_PB_GetState(BUTTON_USER) == SET)
		{
			osDelayTask(100);

			if(BSP_PB_GetState(BUTTON_USER) == SET)
			{
				TRACE_INFO("Confirming update...\r\n");
				//TODO: uncomment when using flag mechanism
				//Signal to bootloader that app is healthy
				mailBoxSetUpdateConfirmed(TRUE);
				mailBoxClearBootCounter();
			}
		}
	}
}

/**
 * @brief Load user RSA public key from "res.c".
 * "res.c" is a C array file system representing the demo folder "resources/keys".
 **/

error_t updateUserLoadKeys(void)
{
   error_t error;

   //Get server's private key
   error = resGetData(UPDT_PUBLIC_SIGN_KEY, (const uint8_t **) &pemUpdtSignPublicKey,
      &pemUpdtSignPublicKeyLen);
   //Any error to report?
   if(error)
      return error;

   //Successful process
   return NO_ERROR;
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void)
{
	error_t error;
	OsTaskId taskId;
	OsTaskParameters taskParams;
	NetInterface *interface;
	NetSettings netSettings;
	MacAddr macAddr;
#if (APP_USE_DHCP_CLIENT == DISABLED)
	Ipv4Addr ipv4Addr;
#endif
#if (APP_USE_SLAAC == DISABLED)
	Ipv6Addr ipv6Addr;
#endif

	//HAL library initialization
	HAL_Init();
	//Configure the system clock
	SystemClock_Config();
	//Setup MPU
	MPU_Config();

	//Enable instruction cache
	//HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
	//HAL_ICACHE_Enable();

	//Initialize kernel
	osInitKernel();
	//Configure debug UART
	debugInit(115200);

	//Start-up message
	TRACE_INFO("\r\n");
	TRACE_INFO("**************************************************\r\n");
	TRACE_INFO("*** CycloneBOOT IAP Single-Bank HTTP Server Demo ***\r\n");
	TRACE_INFO("**************************************************\r\n");
	TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
	TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
	TRACE_INFO("Target: STM32H563\r\n");
	TRACE_INFO("Version: %s\r\n",APP_VERSION_STRING);
	//TODO: uncomment when using the counter mechanism
	//TRACE_INFO("Boot count: %d\r\n",mailBoxGetBootCounter());

	if (mailBoxIsUpdateConfirmed()) {
	    TRACE_INFO("Firmware status: CONFIRMED\n");
	} else {
	    TRACE_INFO("Firmware status: CANDIDATE\n");
	}
	TRACE_INFO("\r\n");

	//LED configuration
	BSP_LED_Init(LED1);
	BSP_LED_Init(LED2);
	BSP_LED_Init(LED3);

	//Clear LEDs
	BSP_LED_Off(LED1);
	BSP_LED_Off(LED2);
	BSP_LED_Off(LED3);

	//Initialize user button
	BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

	//Initialize hardware cryptographic accelerator
	error = stm32h5xxCryptoInit();
	//Any error to report?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to initialize hardware crypto accelerator!\r\n");
	}

	//Get default settings
    netGetDefaultSettings(&netSettings);
    //Network interfaces
    netSettings.interfaces = netInterfaces;
    netSettings.numInterfaces = arraysize(netInterfaces);

	//TCP/IP stack initialization
	error = netInit(&netContext, &netSettings);
	//Any error to report?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to initialize TCP/IP stack!\r\n");
	}

	//Start TCP/IP stack
	error = netStart(&netContext);
	//Any error to report?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to start TCP/IP stack!\r\n");
	}

	//Configure the first Ethernet interface
	interface = &netInterfaces[0];

	//Set interface name
	netSetInterfaceName(interface, APP_IF_NAME);
	//Set host name
	netSetHostname(interface, APP_HOST_NAME);
	//Set host MAC address
	macStringToAddr(APP_MAC_ADDR, &macAddr);
	netSetMacAddr(interface, &macAddr);
	//Select the relevant network adapter
	netSetDriver(interface, &stm32h5xxEthDriver);
	netSetPhyDriver(interface, &lan8742PhyDriver);

	//Initialize network interface
	error = netConfigInterface(interface);
	//Any error to report?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to configure interface %s!\r\n", interface->name);
	}

#if (IPV4_SUPPORT == ENABLED)
#if (APP_USE_DHCP_CLIENT == ENABLED)
	//Get default settings
	dhcpClientGetDefaultSettings(&dhcpClientSettings);
	//Set the network interface to be configured by DHCP
	dhcpClientSettings.interface = interface;
	//Disable rapid commit option
	dhcpClientSettings.rapidCommit = FALSE;

	//DHCP client initialization
	error = dhcpClientInit(&dhcpClientContext, &dhcpClientSettings);
	//Failed to initialize DHCP client?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to initialize DHCP client!\r\n");
	}

	//Start DHCP client
	error = dhcpClientStart(&dhcpClientContext);
	//Failed to start DHCP client?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to start DHCP client!\r\n");
	}
#else
	//Set IPv4 host address
	ipv4StringToAddr(APP_IPV4_HOST_ADDR, &ipv4Addr);
	ipv4SetHostAddr(interface, ipv4Addr);

	//Set subnet mask
	ipv4StringToAddr(APP_IPV4_SUBNET_MASK, &ipv4Addr);
	ipv4SetSubnetMask(interface, ipv4Addr);

	//Set default gateway
	ipv4StringToAddr(APP_IPV4_DEFAULT_GATEWAY, &ipv4Addr);
	ipv4SetDefaultGateway(interface, ipv4Addr);

	//Set primary and secondary DNS servers
	ipv4StringToAddr(APP_IPV4_PRIMARY_DNS, &ipv4Addr);
	ipv4SetDnsServer(interface, 0, ipv4Addr);
	ipv4StringToAddr(APP_IPV4_SECONDARY_DNS, &ipv4Addr);
	ipv4SetDnsServer(interface, 1, ipv4Addr);
#endif
#endif

#if (IPV6_SUPPORT == ENABLED)
#if (APP_USE_SLAAC == ENABLED)
	//Get default settings
	slaacGetDefaultSettings(&slaacSettings);
	//Set the network interface to be configured
	slaacSettings.interface = interface;

	//SLAAC initialization
	error = slaacInit(&slaacContext, &slaacSettings);
	//Failed to initialize SLAAC?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to initialize SLAAC!\r\n");
	}

	//Start IPv6 address autoconfiguration process
	error = slaacStart(&slaacContext);
	//Failed to start SLAAC process?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to start SLAAC!\r\n");
	}
#else
	//Set link-local address
	ipv6StringToAddr(APP_IPV6_LINK_LOCAL_ADDR, &ipv6Addr);
	ipv6SetLinkLocalAddr(interface, &ipv6Addr);

	//Set IPv6 prefix
	ipv6StringToAddr(APP_IPV6_PREFIX, &ipv6Addr);
	ipv6SetPrefix(interface, 0, &ipv6Addr, APP_IPV6_PREFIX_LENGTH);

	//Set global address
	ipv6StringToAddr(APP_IPV6_GLOBAL_ADDR, &ipv6Addr);
	ipv6SetGlobalAddr(interface, 0, &ipv6Addr);

	//Set default router
	ipv6StringToAddr(APP_IPV6_ROUTER, &ipv6Addr);
	ipv6SetDefaultRouter(interface, 0, &ipv6Addr);

	//Set primary and secondary DNS servers
	ipv6StringToAddr(APP_IPV6_PRIMARY_DNS, &ipv6Addr);
	ipv6SetDnsServer(interface, 0, &ipv6Addr);
	ipv6StringToAddr(APP_IPV6_SECONDARY_DNS, &ipv6Addr);
	ipv6SetDnsServer(interface, 1, &ipv6Addr);
#endif
#endif

	//Get default settings
	httpServerGetDefaultSettings(&httpServerSettings);
	//Bind HTTP server to the desired interface
	httpServerSettings.interface = &netInterfaces[0];
	//Listen to port 80
	httpServerSettings.port = HTTP_PORT;
	//Client connections
	httpServerSettings.maxConnections = APP_HTTP_MAX_CONNECTIONS;
	httpServerSettings.connections = httpConnections;
	//Specify the server's root directory
	strcpy(httpServerSettings.rootDirectory, "/www/");
	//Set default home page
	strcpy(httpServerSettings.defaultDocument, "index.html");
	//Callback functions
	httpServerSettings.requestCallback = httpServerRequestCallback;
	httpServerSettings.uriNotFoundCallback = httpServerUriNotFoundCallback;

	//HTTP server initialization
	error = httpServerInit(&httpServerContext, &httpServerSettings);
	//Failed to initialize HTTP server?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to initialize HTTP server!\r\n");
	}

	//Start HTTP server
	error = httpServerStart(&httpServerContext);
	//Failed to start HTTP server?
	if(error)
	{
		//Debug message
		TRACE_ERROR("Failed to start HTTP server!\r\n");
	}

	//Load Update public signature key
	error = updateUserLoadKeys();
	if (error)
	{
		//Debug message
		TRACE_ERROR("Failed to load Update user keys!\r\n");
		return error;
	}

	//Create Update mutex
	if(!osCreateMutex(&updateMutex))
	{
		//Debug message
		TRACE_ERROR("Failed to create Update mutex!\r\n");
	}


	//Set LED task parameters
	taskParams = OS_TASK_DEFAULT_PARAMS;
	taskParams.stackSize = 200;
	taskParams.priority = OS_TASK_PRIORITY_NORMAL;

	//Create a task to blink the LED
	taskId = osCreateTask("LED", ledTask, NULL, &taskParams);
	//Failed to create the task?
	if(taskId == OS_INVALID_TASK_ID)
	{
		//Debug message
		TRACE_ERROR("Failed to create task!\r\n");
	}

	//Set User button task parameters
	taskParams = OS_TASK_DEFAULT_PARAMS;
	taskParams.stackSize = 200;
	taskParams.priority = OS_TASK_PRIORITY_NORMAL;

	//Create a task to read user button state
	taskId = osCreateTask("BUTTON", buttonTask, NULL, &taskParams);

	//Failed to create the task?
	if(taskId == OS_INVALID_TASK_ID)
	{
		//Debug message
		TRACE_ERROR("Failed to create task!\r\n");
	}

	//Start the execution of tasks
	osStartKernel();

	//This function should never return
	return 0;
}

cboot_error_t mbx_set_nvm_flag(uint32_t flag, bool_t value) {
	FLASH_EraseInitTypeDef EraseInitStruct = {0};
	uint32_t sector_error;

	uint32_t sector_start = get_edata_sector(BOOT_MBX_NVM_START_ADDR);
	uint32_t sector_end = get_edata_sector(BOOT_MBX_NVM_END_ADDR);
	uint32_t nb_sectors = sector_end - sector_start + 1;

	//Erase the zone to clear any existing data
	EraseInitStruct.TypeErase = FLASH_TYPEERASE_SECTORS;
	EraseInitStruct.Banks = get_edata_bank(BOOT_MBX_NVM_START_ADDR);
	EraseInitStruct.Sector = sector_start;
	EraseInitStruct.NbSectors = nb_sectors;

	HAL_FLASH_Unlock();

	if (HAL_FLASHEx_Erase(&EraseInitStruct, &sector_error) != HAL_OK)
	{
		/*
		      Error occurred while sector erase.
		      User can add here some code to deal with this error.
		      SectorError will contain the faulty sector and then to know the code error on this sector,
		      user can call function 'HAL_FLASH_GetError()'
		 */
		TRACE_ERROR("Failed to erase sectors...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	uint16_t boot_magic[1] = {BOOT_MBX_NVM_MAGIC};

	if(HAL_FLASH_Program(FLASH_TYPEPROGRAM_HALFWORD_EDATA, BOOT_MBX_NVM_START_ADDR, (uint32_t)&boot_magic[0]) != HAL_OK)
	{
		TRACE_ERROR("Failed to write magic value...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	// Write the flag value
	uint32_t flagValue = value ? flag : 0xFFFFFFFF;
	if(HAL_FLASH_Program(FLASH_TYPEPROGRAM_HALFWORD_EDATA, BOOT_MBX_NVM_START_ADDR + 4, (uint32_t)&flagValue) != HAL_OK)
	{
		TRACE_ERROR("Failed to write flag value in NVM...\r\n");
		return CBOOT_ERROR_FAILURE;
	}

	HAL_FLASH_Lock();

	return CBOOT_NO_ERROR;
}

bool_t mbx_get_nvm_flag(uint32_t flag) {
	uint16_t storedMagic = *(volatile uint16_t*) BOOT_MBX_NVM_START_ADDR;
	uint16_t storedFlag  = *(volatile uint16_t*) (BOOT_MBX_NVM_START_ADDR + 4);
	uint16_t nvmFlag = BOOT_MBX_NVM_FLAG_UPDATE_CONFIRMED;

	// Validate storage
	if (storedMagic != BOOT_MBX_NVM_MAGIC) {
		return FALSE;
	}

	if(flag == UPDATE_CONFIRMED) {
		flag = nvmFlag;
	}

	return (storedFlag == flag) ? TRUE : FALSE;
}

/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
/**
  * @brief  Gets the sector of a given address
  * @param  Addr: Address of the FLASH Memory
  * @retval The sector of a given address
  */
uint32_t get_edata_sector(uint32_t address)
{
  uint32_t sector = 0;
  uint32_t edataSectorSize = 0;

  /*
    (FLASH_EDATA_SIZE/2) is the size of high-cycle area of flash BANK1.
    Flash high-cycle area have 8 sectors in each Bank.
  */
  edataSectorSize = (FLASH_EDATA_SIZE / 2) / 8;
  /* Check if the address is located in the FLASH high-cycle data area of BANK1 */
  if((address >= FLASH_EDATA_BASE) && (address < FLASH_EDATA_BASE + (FLASH_EDATA_SIZE / 2)))
  {
    sector = (address & ~FLASH_EDATA_BASE) / edataSectorSize;
    sector += 120;
  }
  /* Check if the address is located in the FLASH high-cycle data area of BANK2 */
  else if ((address >= FLASH_EDATA_BASE + (FLASH_EDATA_SIZE / 2)) && (address < FLASH_EDATA_BASE + FLASH_EDATA_SIZE))
  {
    sector = ((address & ~FLASH_EDATA_BASE) - (FLASH_EDATA_SIZE / 2)) / edataSectorSize;
    sector += 120;
  }
  else
  {
    sector = 0xFFFFFFFF; /* Address out of range */
  }

  return sector;
}

/**
  * @brief  Gets the bank of a given address in EDATA area
  * @param  Addr: Address of A given address in EDATA area
  * @retval The bank of a given address in EDATA area
  */
uint32_t get_edata_bank(uint32_t address)
{
  uint32_t bank = 0;

  /* (FLASH_EDATA_SIZE/2) is the size of high-cycle area of flash BANK1 */
  if((address >= FLASH_EDATA_BASE) && (address < FLASH_EDATA_BASE + (FLASH_EDATA_SIZE/2)))
  {
    bank = FLASH_BANK_1;
  }
  else if ((address >= FLASH_EDATA_BASE + (FLASH_EDATA_SIZE/2)) && (address < FLASH_EDATA_BASE + FLASH_EDATA_SIZE))
  {
    bank = FLASH_BANK_2;
  }
  else
  {
    bank = 0xFFFFFFFF; /* Address out of range */
  }
  return bank;
}
