/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

/**
 * Configuring Dual Bank mode (using ST-Link): STM32_Programmer_CLI.exe -c port=SWD index=0 -ob SWAP_BANK=0 NSBOOTADD=0x80000
 *
 * Flashing the firmware :
 *
 * - Initial Firmware Addresss - 0x08000000
 * - Use ST -Link or equivalent to flash the initial firmware application (iap_demo_1_0_0.bin) to the MCU internal flash :
 *   E.g., STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\iap_demo_1_0_0.bin 0x08000000 -halt -rst
 *
 * Uploading the firmware update :
 *
 *  - The trace outputs are redirected to USART. Exact configuration is found in debug.c file
 *  - Connect Ethernet port to the board
 *  - Once the application is running, from USART trace find the IP address allocated to the device (reset the MCU if necessary after flashing the firmware)
 *  - Using a web browser navigate to the IP address. E.g., http://192.168.1.42
 *  - Upload the iap_demo_2_0_0.img file using the HTML UI
 *  - After a short while, the MCU should reboot and the HTML UI should reflect the updated version
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h573i_discovery.h"
#include "stm32h573i_discovery_lcd.h"
#include "stm32_lcd.h"
#include "path.h"
#include "date_time.h"
#include "resource_manager.h"
#include "second_stage/boot.h"
#include "drivers/flash/internal/stm32h5xx_flash_driver.h"
#include "hardware/stm32h5xx/stm32h5xx_crypto.h"
#include "debug.h"

//#define RUNTIME_PUBLIC_SIGN_KEY "keys/ec-secp256k1-pub-key.pem"
#define RUNTIME_PUBLIC_SIGN_KEY "keys/my_rsa_pub_key.pem"

//Global variables
uint_t lcdLine = 0;
uint_t lcdColumn = 0;
uint8_t ledState = 0;
systime_t ledDelay = 0;
systime_t ledTimestamp = 0;

//Runtime signature public key declaration
const uint8_t* pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;

error_t bootUserLoadKeys(void);


/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/

void lcdSetCursor(uint_t line, uint_t column) {
    lcdLine = MIN(line, 12);
    lcdColumn = MIN(column, 18);
}


/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/

void lcdPutChar(char_t c) {
    if (c == '\r') {
        lcdColumn = 0;
    } else if (c == '\n') {
        lcdColumn = 0;
        lcdLine++;
    } else if (lcdLine < 12 && lcdColumn < 18) {
        //Display current character
        UTIL_LCD_DisplayChar(lcdColumn * 13 + 4, lcdLine * 20 + 2, c);

        //Advance the cursor position
        if (++lcdColumn >= 18) {
            lcdColumn = 0;
            lcdLine++;
        }
    }
}

/**
 * @brief System clock configuration
 **/

void SystemClock_Config(void) {
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};
    RCC_CRSInitTypeDef RCC_CRSInitStruct = {0};

    //Configure the main internal regulator output voltage
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

    //Wait for the voltage scaling ready flag to be set
    while (!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {
    }

    //Configure PLL with HSE as source
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI48;
    RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS_DIGITAL;
    RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSE;
    RCC_OscInitStruct.PLL.PLLM = 5;
    RCC_OscInitStruct.PLL.PLLN = 100;
    RCC_OscInitStruct.PLL.PLLP = 2;
    RCC_OscInitStruct.PLL.PLLQ = 2;
    RCC_OscInitStruct.PLL.PLLR = 2;
    RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_2;
    RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
    RCC_OscInitStruct.PLL.PLLFRACN = 0;
    HAL_RCC_OscConfig(&RCC_OscInitStruct);

    //Select PLL as system clock source and configure bus clocks dividers
    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
                                  RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
    RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
    HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

    //Select clock source for RNG peripheral
    RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
    RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
    HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);

    //Configure CRS
    RCC_CRSInitStruct.Prescaler = RCC_CRS_SYNC_DIV1;
    RCC_CRSInitStruct.Source = RCC_CRS_SYNC_SOURCE_USB;
    RCC_CRSInitStruct.Polarity = RCC_CRS_SYNC_POLARITY_RISING;
    RCC_CRSInitStruct.ReloadValue = __HAL_RCC_CRS_RELOADVALUE_CALCULATE(48000000, 1000);
    RCC_CRSInitStruct.ErrorLimitValue = 34;
    RCC_CRSInitStruct.HSI48CalibrationValue = 32;
    HAL_RCCEx_CRSConfig(&RCC_CRSInitStruct);

    //Enable CRS clock
    __HAL_RCC_CRS_CLK_ENABLE();
}

/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param) {
    //Endless loop
    if((uint32_t) (systemTicks - ledTimestamp) >= ledDelay)
    {
        //Toggle LED state
        if(ledState == 0)
        {
            BSP_LED_On(LED1);
            ledState = 1;
            ledDelay = 100;
        }
        else
        {
            BSP_LED_Off(LED1);
            ledState = 0;
            ledDelay = 900;
        }

        ledTimestamp = systemTicks;
    }
}

/**
 * @brief MPU configuration
 */
void MPU_Config(void)
{
    HAL_MPU_Disable();

    // configure the MPU to prohibit access to after the .bss
    // this will trigger an exception if the stack grows too large
    MPU_Region_InitTypeDef MPU_InitStruct;
    MPU_InitStruct.Enable = MPU_REGION_ENABLE;
    MPU_InitStruct.Number = MPU_REGION_NUMBER0;
    MPU_InitStruct.BaseAddress = 0x08fff000;
    MPU_InitStruct.LimitAddress = 0x08ffffff;
    MPU_InitStruct.AttributesIndex = MPU_ATTRIBUTES_NUMBER0;
    MPU_InitStruct.AccessPermission = MPU_REGION_ALL_RO;
    MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;
    MPU_InitStruct.IsShareable = MPU_ACCESS_OUTER_SHAREABLE;
    HAL_MPU_ConfigRegion(&MPU_InitStruct);

    HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void) {
    OsTaskId taskId;
    OsTaskParameters taskParams;

    BootSettings bootSettings;
    BootContext bootContext;

    error_t error;
    cboot_error_t cerror;

    //HAL library initialization
    HAL_Init();
    //Configure the system clock
    SystemClock_Config();

    //Enable instruction cache
    HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
    HAL_ICACHE_Enable();

    MPU_Config();

    //Initialize kernel
    osInitKernel();
    //Configure debug UART
    debugInit(115200);

    //Start-up message
    TRACE_INFO("\r\n");
    TRACE_INFO("*********************************************\r\n");
    TRACE_INFO("*** CycloneBOOT 2nd Stage Bootloader Demo ***\r\n");
    TRACE_INFO("*********************************************\r\n");
    TRACE_INFO("Copyright: 2010-2025 Oryx Embedded SARL\r\n");
    TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
    TRACE_INFO("Target: STM32H573\r\n");
    TRACE_INFO("\r\n");

    //LED configuration
    BSP_LED_Init(LED1);
    BSP_LED_Init(LED2);
    BSP_LED_Init(LED3);
    BSP_LED_Init(LED4);

    //Clear LEDs
    BSP_LED_Off(LED1);
    BSP_LED_Off(LED2);
    BSP_LED_Off(LED3);
    BSP_LED_Off(LED4);

    //Initialize user button
    BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

    //Initialize LCD display
    BSP_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
    UTIL_LCD_SetFuncDriver(&LCD_Driver);
    UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_BLUE);
    UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_WHITE);
    UTIL_LCD_SetFont(&Font20);
    BSP_LCD_DisplayOn(0);

    //Clear LCD display
    UTIL_LCD_Clear(UTIL_LCD_COLOR_BLUE);

    //Welcome message
    lcdSetCursor(0, 0);
    printf("2nd Stage BL\r\n");

    //Set task parameters
    taskParams = OS_TASK_DEFAULT_PARAMS;
    taskParams.stackSize = 200;
    taskParams.priority = OS_TASK_PRIORITY_NORMAL;

    //Create a task to blink the LED
    taskId = osCreateTask("LED", ledTask, NULL, &taskParams);
    //Failed to create the task?
    if (taskId == OS_INVALID_TASK_ID) {
        //Debug message
        TRACE_ERROR("Failed to create task!\r\n");
    }

    //Initialize hardware cryptographic accelerator
    error = stm32h5xxCryptoInit();
    //Any error to report?
    if(error)
    {
        //Debug message
        TRACE_ERROR("Failed to initialize hardware crypto accelerator!\r\n");
    }

    bootUserLoadKeys();

    //Begining of handling exception
    do
    {
        cerror = CBOOT_NO_ERROR;

        //Get default booloader user settings
        bootGetDefaultSettings(&bootSettings);

        //User settings primary memory configuration
        bootSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
        bootSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
        bootSettings.memories[0].driver = &stm32h5xxFlashDriver;
        bootSettings.memories[0].nbSlots = 3;
        //User settings primary memory slot 0 configuration
        bootSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
        bootSettings.memories[0].slots[0].cType = SLOT_CONTENT_BINARY;
        bootSettings.memories[0].slots[0].memParent = &bootSettings.memories[0];
        bootSettings.memories[0].slots[0].addr = 0x08020000;
        bootSettings.memories[0].slots[0].size = 0x80000;
        //User settings primary memory slot 1 configuration
        bootSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
        bootSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
        bootSettings.memories[0].slots[1].memParent = &bootSettings.memories[0];
        bootSettings.memories[0].slots[1].addr = 0x080C0000;
        bootSettings.memories[0].slots[1].size = 0x80000;
        //User settings primary memory slot 2 configuration
        bootSettings.memories[0].slots[2].type = SLOT_TYPE_DIRECT;
        bootSettings.memories[0].slots[2].cType = SLOT_CONTENT_UPDATE|SLOT_CONTENT_BACKUP;
        bootSettings.memories[0].slots[2].memParent = &bootSettings.memories[0];
        bootSettings.memories[0].slots[2].addr = 0x08160000;
        bootSettings.memories[0].slots[2].size = 0x80000;

        //Runtime signature check information
        bootSettings.verifySettings.signAlgo = VERIFY_SIGN_RSA;
        bootSettings.verifySettings.signHashAlgo = SHA256_HASH_ALGO;
        bootSettings.verifySettings.signKey = (const char *)pemUpdtSignPublicKey;
        bootSettings.verifySettings.signKeyLen = pemUpdtSignPublicKeyLen;

        //Initialize bootloader
        cerror = bootInit(&bootContext, &bootSettings);
        //Is any error?
        if(cerror)
            break;
    }while(0);

    //Is any error?
    if(cerror)
    {
        //Debug message
        TRACE_ERROR("Bootloader configuration failed!\r\n");
        while(1);
    }

    while(1)
    {
        ledTask(&taskParams);
        cerror = bootFsm(&bootContext);
        //Is any error?
        if(cerror)
        {
            //Debug message
            TRACE_ERROR("Bootloader failure!\r\n");
        }
    }

    //This function should never return
    return 0;
}

/**
 * @brief Load user ECDSA/RSA public key from "res.c".
 * "res.c" is a C array file system representing the custom user assets.
 **/

error_t bootUserLoadKeys(void)
{
    error_t error;

    //Get server's private key
    error = resGetData(RUNTIME_PUBLIC_SIGN_KEY, &pemUpdtSignPublicKey,
       &pemUpdtSignPublicKeyLen);
    //Any error to report?
    if(error)
        return error;

    //Successful process
    return NO_ERROR;
}
