/**
 * @file main.c
 * @brief Main routine
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

/**
 *
 * Before flashing the initial firmware :
 *
 * The current file defines a pre-processor macro UPDATE_SERVER_NAME. This indicates the IP address of the machine that runs the
 * bundled firmware update server. If you are running the server in the same computer as host, update the macro with the machines IP address
 *
 * Flashing the firmware :
 *
 * - 2nd stage bootloader address - 0x08000000
 * - Initial firmware address     - 0x08020000
 *
 * - Use ST-Link or equivalent to flash the bootloader (bootloader.bin) and initial firmware application (iap_demo_bootable.bin) to the MCU internal flash :
 *   E.g.,
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\2nd_stage_bl\bootloader.bin 0x08000000 -halt -rst
 *      STM32_Programmer_CLI.exe -c port=SWD index=0 -d ..\iap_demo_bootable.bin 0x08020000 -halt -rst
 *
 * Performing the firmware update :
 *
 *  - The trace outputs are redirected to USART. Exact configuration is found in debug.c file
 *  - Connect USB cable to the board (make sure to use USER USB port instead of ST-Link USB port)
 *  - Run the server/server.py script (e.g. python3 server.py) . This implements a basic web server with Python Flask micro-framework.
 *    Keep it running for the duration of the demo. (More information about Flask: https://flask.palletsprojects.com/en/stable/installation/)
 *  - Now, modify the src/version.h to simulate a firmware update. Modify the following macros: APP_MAJOR_VERSION and APP_VERSION_STRING
 *  - Using image builder tool, generate a firmware update image. The generated file will be served by the Flask web server
 *  - Reset the STM32 evaluation kit. Once the board has acquired an IP address, press the blue USER button. It'll initiate a HTTP(S) connection to the Flask server
 *  - Firmware update should now begin.
 *  - The serial terminal will show the device rebooting to install the update
 *
 **/

//Dependencies
#include <stdlib.h>
#include "stm32h5xx.h"
#include "stm32h5xx_hal.h"
#include "stm32h573i_discovery.h"
#include "stm32h573i_discovery_lcd.h"
#include "stm32_lcd.h"
#include "core/net.h"
#include "drivers/mac/stm32h5xx_eth_driver.h"
#include "drivers/phy/lan8742_driver.h"
#include "dhcp/dhcp_client.h"
#include "ipv6/slaac.h"
#include "http/http_client.h"
#include "tls.h"
#include "tls_cipher_suites.h"
#include "hardware/stm32h5xx/stm32h5xx_crypto.h"
#include "rng/trng.h"
#include "rng/hmac_drbg.h"
#include "resource_manager.h"
#include "debug.h"
#include "jsmn.h"
#include "core/mailbox.h"
#include "update/update.h"
#include "drivers/flash/internal/stm32h5xx_flash_driver.h"
#include "version.h"

//Ethernet interface configuration
#define APP_IF_NAME "eth0"
#define APP_HOST_NAME "https-client-demo"
#define APP_MAC_ADDR "00-AB-CD-EF-05-73"

#define APP_USE_DHCP_CLIENT ENABLED
#define APP_IPV4_HOST_ADDR "192.168.0.20"
#define APP_IPV4_SUBNET_MASK "255.255.255.0"
#define APP_IPV4_DEFAULT_GATEWAY "192.168.0.254"
#define APP_IPV4_PRIMARY_DNS "8.8.8.8"
#define APP_IPV4_SECONDARY_DNS "8.8.4.4"

#define APP_USE_SLAAC ENABLED
#define APP_IPV6_LINK_LOCAL_ADDR "fe80::573"
#define APP_IPV6_PREFIX "2001:db8::"
#define APP_IPV6_PREFIX_LENGTH 64
#define APP_IPV6_GLOBAL_ADDR "2001:db8::573"
#define APP_IPV6_ROUTER "fe80::1"
#define APP_IPV6_PRIMARY_DNS "2001:4860:4860::8888"
#define APP_IPV6_SECONDARY_DNS "2001:4860:4860::8844"

//Application configuration
#define UPDATE_SERVER_NAME "192.168.1.215"
#define UPDATE_SERVER_PORT 5000
#define CHECK_FOR_UPDATE_URI "/check-update"
#define FIRMWARE_UPDATE_URI "/firmware"

#define APP_SET_CIPHER_SUITES DISABLED
#define APP_SET_SERVER_NAME ENABLED
#define APP_SET_TRUSTED_CA_LIST ENABLED

//List of preferred ciphersuites
const uint16_t cipherSuites[] =
{
   TLS_CHACHA20_POLY1305_SHA256,
   TLS_AES_128_GCM_SHA256,
   TLS_AES_256_GCM_SHA384,
   TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256,
   TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256,
   TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
   TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
   TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384,
   TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384,
   TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
   TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
   TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
   TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
   TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
   TLS_DHE_RSA_WITH_AES_256_GCM_SHA384,
   TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
   TLS_DHE_RSA_WITH_AES_256_CBC_SHA,
   TLS_RSA_WITH_AES_128_GCM_SHA256,
   TLS_RSA_WITH_AES_256_GCM_SHA384,
   TLS_RSA_WITH_AES_128_CBC_SHA,
   TLS_RSA_WITH_AES_256_CBC_SHA,
   TLS_RSA_WITH_3DES_EDE_CBC_SHA
};

//List of trusted CA certificates
const char_t trustedCaList[] =
   "-----BEGIN CERTIFICATE-----"
   "MIICDzCCAbWgAwIBAgIUZxttrQ8T0QSfIFyvgChwuLOkMK8wCgYIKoZIzj0EAwIw"
   "XTELMAkGA1UEBhMCQVUxEzARBgNVBAgMClNvbWUtU3RhdGUxITAfBgNVBAoMGElu"
   "dGVybmV0IFdpZGdpdHMgUHR5IEx0ZDEWMBQGA1UEAwwNMTkyLjE2OC4xLjIxNTAe"
   "Fw0yNjAxMjcxMTA3MjVaFw0yNzAxMjcxMTA3MjVaMF0xCzAJBgNVBAYTAkFVMRMw"
   "EQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRnaXRzIFB0"
   "eSBMdGQxFjAUBgNVBAMMDTE5Mi4xNjguMS4yMTUwWTATBgcqhkjOPQIBBggqhkjO"
   "PQMBBwNCAARYASqtZTBKOl1axRkuYqRfttzLaHRx2voreUIQgh4TCvZrxTEgZRqS"
   "GiEwaEV69+viE/ZTs1tNmcVU7yu0y01oo1MwUTAdBgNVHQ4EFgQUqPpJrwzzKjqn"
   "k6avDSFWsoQ0FGIwHwYDVR0jBBgwFoAUqPpJrwzzKjqnk6avDSFWsoQ0FGIwDwYD"
   "VR0TAQH/BAUwAwEB/zAKBggqhkjOPQQDAgNIADBFAiBMlHpmSoNNFBtgmimtJ24+"
   "cFSFK6yEO3w6HMD2hAehCwIhAPWc8VFZgUcvpeASni+TgyMLUmIlNTSnPZL0+xLY"
   "OkD1"
   "-----END CERTIFICATE-----";

//Global variables
uint_t lcdLine = 0;
uint_t lcdColumn = 0;

DhcpClientSettings dhcpClientSettings;
DhcpClientContext dhcpClientContext;
SlaacSettings slaacSettings;
SlaacContext slaacContext;
NetContext netContext;
NetInterface netInterfaces[1];
HmacDrbgContext hmacDrbgContext;
uint8_t seed[48];

static int jsoneq(const char *json, jsmntok_t *tok, const char *s);
error_t updateUserLoadKeys(void);

volatile uint8_t updateAvailable = FALSE;
volatile uint8_t updateCompleted = FALSE;
uint32_t expectedUpdateSize = 0;

// Update library structs
UpdateContext updateContext;
UpdateSettings updateSettings;

//Update signature public key resource path
#define UPDT_PUBLIC_SIGN_KEY "keys/my_rsa_pub_key.pem"

const uint8_t *pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;
//Buffer to put data
uint8_t data_buffer[128];
//CycloneBOOT mutex handler
OsMutex updateMutex;

/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/
void lcdSetCursor(uint_t line, uint_t column)
{
   lcdLine = MIN(line, 12);
   lcdColumn = MIN(column, 18);
}

/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/
void lcdPutChar(char_t c)
{
   if(c == '\r')
   {
      lcdColumn = 0;
   }
   else if(c == '\n')
   {
      lcdColumn = 0;
      lcdLine++;
   }
   else if(lcdLine < 12 && lcdColumn < 18)
   {
      //Display current character
      UTIL_LCD_DisplayChar(lcdColumn * 13 + 4, lcdLine * 20 + 2, c);

      //Advance the cursor position
      if(++lcdColumn >= 18)
      {
         lcdColumn = 0;
         lcdLine++;
      }
   }
}


/**
 * @brief System clock configuration
 **/
void SystemClock_Config(void)
{
   RCC_OscInitTypeDef RCC_OscInitStruct = {0};
   RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
   RCC_PeriphCLKInitTypeDef RCC_PeriphClkInitStruct = {0};
   RCC_CRSInitTypeDef RCC_CRSInitStruct = {0};

   //Configure the main internal regulator output voltage
   __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

   //Wait for the voltage scaling ready flag to be set
   while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY))
   {
   }

   //Configure PLL with HSE as source
   RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI48;
   RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS_DIGITAL;
   RCC_OscInitStruct.HSI48State = RCC_HSI48_ON;
   RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
   RCC_OscInitStruct.PLL.PLLSource = RCC_PLL1_SOURCE_HSE;
   RCC_OscInitStruct.PLL.PLLM = 5;
   RCC_OscInitStruct.PLL.PLLN = 100;
   RCC_OscInitStruct.PLL.PLLP = 2;
   RCC_OscInitStruct.PLL.PLLQ = 2;
   RCC_OscInitStruct.PLL.PLLR = 2;
   RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1_VCIRANGE_2;
   RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1_VCORANGE_WIDE;
   RCC_OscInitStruct.PLL.PLLFRACN = 0;
   HAL_RCC_OscConfig(&RCC_OscInitStruct);

   //Select PLL as system clock source and configure bus clocks dividers
   RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK |
      RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_PCLK3;
   RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
   RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
   RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV1;
   RCC_ClkInitStruct.APB3CLKDivider = RCC_HCLK_DIV1;
   HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5);

   //Select clock source for RNG peripheral
   RCC_PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_RNG;
   RCC_PeriphClkInitStruct.RngClockSelection = RCC_RNGCLKSOURCE_HSI48;
   HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphClkInitStruct);

   //Configure CRS
   RCC_CRSInitStruct.Prescaler = RCC_CRS_SYNC_DIV1;
   RCC_CRSInitStruct.Source = RCC_CRS_SYNC_SOURCE_USB;
   RCC_CRSInitStruct.Polarity = RCC_CRS_SYNC_POLARITY_RISING;
   RCC_CRSInitStruct.ReloadValue = __HAL_RCC_CRS_RELOADVALUE_CALCULATE(48000000, 1000);
   RCC_CRSInitStruct.ErrorLimitValue = 34;
   RCC_CRSInitStruct.HSI48CalibrationValue = 32;
   HAL_RCCEx_CRSConfig(&RCC_CRSInitStruct);

   //Enable CRS clock
   __HAL_RCC_CRS_CLK_ENABLE();
}


/**
 * @brief TLS initialization callback
 * @param[in] context Pointer to the HTTP client context
 * @param[in] tlsContext Pointer to the TLS context
 * @param[in] param User parameter
 * @return Error code
 **/

error_t httpClientTlsInitCallback(HttpClientContext *context,
   TlsContext *tlsContext, void *param)
{
   error_t error;

   //Debug message
   TRACE_INFO("HTTP Client: TLS initialization callback\r\n");

   //Set the PRNG algorithm to be used
   error = tlsSetPrng(tlsContext, HMAC_DRBG_PRNG_ALGO, &hmacDrbgContext);
   //Any error to report?
   if(error)
      return error;

#if (APP_SET_CIPHER_SUITES == ENABLED)
   //Preferred cipher suite list
   error = tlsSetCipherSuites(tlsContext, cipherSuites, arraysize(cipherSuites));
   //Any error to report?
   if(error)
      return error;
#endif

#if (APP_SET_SERVER_NAME == ENABLED)
   //Set the fully qualified domain name of the server
   error = tlsSetServerName(tlsContext, UPDATE_SERVER_NAME);
   //Any error to report?
   if(error)
      return error;
#endif

#if (APP_SET_TRUSTED_CA_LIST == ENABLED)
   //Import the list of trusted CA certificates
   error = tlsSetTrustedCaList(tlsContext, trustedCaList, strlen(trustedCaList));
   //Any error to report?
   if(error)
      return error;
#endif

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief HTTP client test routine
 * @return Error code
 **/

error_t httpClientCheckFWUpdate(void)
{
   HttpClientContext httpClientContext;

   error_t error;
   size_t length;
   uint_t status;
   const char_t *value;
   IpAddr ipAddr;
   char_t buffer[128];

   jsmn_parser parser;
   jsmntok_t tokens[32];
   char_t response[512];
   size_t response_len = 0;

   cboot_error_t cboot_error = CBOOT_NO_ERROR;

   //Initialize JSON parser
   jsmn_init(&parser);

   //Initialize HTTP client context
   httpClientInit(&httpClientContext);

   //Start of exception handling block
   do
   {
      //Debug message
      TRACE_INFO("\r\n\r\nResolving server name...\r\n");

      //Resolve HTTP server name
      error = getHostByName(NULL, UPDATE_SERVER_NAME, &ipAddr, 0);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to resolve server name!\r\n");
         break;
      }

#if (UPDATE_SERVER_PORT == 5000)
      //Register TLS initialization callback
      error = httpClientRegisterTlsInitCallback(&httpClientContext,
         httpClientTlsInitCallback, NULL);
      //Any error to report?
      if(error)
         break;
#endif

      //Select HTTP protocol version
      error = httpClientSetVersion(&httpClientContext, HTTP_VERSION_1_1);
      //Any error to report?
      if(error)
         break;

      //Set timeout value for blocking operations
      error = httpClientSetTimeout(&httpClientContext, 20000);
      //Any error to report?
      if(error)
         break;

      //Debug message
      TRACE_INFO("Connecting to HTTP server %s...\r\n",
         ipAddrToString(&ipAddr, NULL));

      //Connect to the HTTP server
      error = httpClientConnect(&httpClientContext, &ipAddr,
         UPDATE_SERVER_PORT);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to connect to HTTP server!\r\n");
         break;
      }

      //Create an HTTP request
      httpClientCreateRequest(&httpClientContext);
      httpClientSetMethod(&httpClientContext, "POST");
      httpClientSetUri(&httpClientContext, CHECK_FOR_UPDATE_URI);

      //Prepare HTTP request body
      const char_t checkForUpdate[] =
         "{\"device_id\":\"some-value\",\"fw_version\":\""
         APP_VERSION_STRING
         "\"}";

      char_t contentLengthStr[16];
      sprintf(contentLengthStr, "%u", strlen(checkForUpdate));

      //Add HTTP header fields
      httpClientAddHeaderField(&httpClientContext, "Host", UPDATE_SERVER_NAME);
      httpClientAddHeaderField(&httpClientContext, "User-Agent", "STM32H5-IoT-Device");
      httpClientAddHeaderField(&httpClientContext, "Content-Type", "application/json");
      httpClientAddHeaderField(&httpClientContext, "Content-Length", contentLengthStr);

      //Send HTTP request header
      error = httpClientWriteHeader(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to write HTTP request header!\r\n");
         break;
      }

      error = httpClientWriteBody(&httpClientContext, checkForUpdate, strlen(checkForUpdate),
         NULL, 0);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to write HTTP request body!\r\n");
         break;
      }

      //Receive HTTP response header
      error = httpClientReadHeader(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to read HTTP response header!\r\n");
         break;
      }

      //Retrieve HTTP status code
      status = httpClientGetStatus(&httpClientContext);
      //Debug message
      TRACE_INFO("HTTP status code: %u\r\n", status);

      //Retrieve the value of the Content-Type header field
      value = httpClientGetHeaderField(&httpClientContext, "Content-Type");

      //Header field found?
      if(value != NULL)
      {
         //Debug message
         TRACE_INFO("Content-Type header field value: %s\r\n", value);
      }
      else
      {
         //Debug message
         TRACE_INFO("Content-Type header field not found!\r\n");
      }

      //Receive HTTP response body
      while(!error)
      {
         //Read data
         error = httpClientReadBody(&httpClientContext, buffer,
            sizeof(buffer) - 1, &length, 0);

         // Check status code
         if (!error && length > 0) {
            if (response_len + length < sizeof(response)) {
               memcpy(response + response_len, buffer, length);
               response_len += length;
            } else {
               TRACE_INFO("Response buffer overflow\r\n");
               error = ERROR_BUFFER_OVERFLOW;
               break;
            }
         }
      }

      response[response_len] = '\0';
      TRACE_INFO("\r\nFull JSON response:\r\n%s\r\n", response);

      int count = jsmn_parse(&parser, response, response_len, tokens, 32);
      if (count<0) {
         TRACE_INFO("Could not parse JSON from server.\r\n");
      } else {
         for (int i = 1; i < count; i++) {
            if (jsoneq(response, &tokens[i], "update_available") == 0)
            {
               updateAvailable = (response[tokens[i + 1].start] == 't');
               i++;
            }
            else if (jsoneq(response, &tokens[i], "size") == 0) {
               jsmntok_t *tok = &tokens[i+1];
               char temp[16];
               size_t len = tok->end - tok->start;

               if(len < sizeof(temp))
               {
                  memcpy(temp, response + tok->start, len);
                  temp[len] = '\0';
                  expectedUpdateSize = atoi(temp);
                  TRACE_INFO("Expected firmware size: %lu bytes\r\n", expectedUpdateSize);
               }

               i++;
            }
            else {
               // just for sanity
            }
         }
      }


      //Terminate the HTTP response body with a CRLF
      TRACE_INFO("\r\n");

      //Any error to report?
      if(error != ERROR_END_OF_STREAM)
         break;

      //Close HTTP response body
      error = httpClientCloseBody(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to read HTTP response trailer!\r\n");
         break;
      }

      //Gracefully disconnect from the HTTP server
      httpClientDisconnect(&httpClientContext);

      //Debug message
      TRACE_INFO("Connection closed\r\n");

      if (updateAvailable) {
         TRACE_INFO("Update available!\r\n");

         //Get default Update settings
         updateGetDefaultSettings(&updateSettings);

#if (UPDATE_ANTI_ROLLBACK_SUPPORT == ENABLED)
         updateSettings.appVersion = APP_VERSION;
#endif
         //User update settings security configuration
         updateSettings.imageInCrypto.verifySettings.verifyMethod = VERIFY_METHOD_SIGNATURE;
         updateSettings.imageInCrypto.verifySettings.signAlgo = VERIFY_SIGN_RSA;
         updateSettings.imageInCrypto.verifySettings.signHashAlgo = SHA256_HASH_ALGO;
         updateSettings.imageInCrypto.verifySettings.signKey = (const char_t *) pemUpdtSignPublicKey;
         updateSettings.imageInCrypto.verifySettings.signKeyLen = pemUpdtSignPublicKeyLen;
         updateSettings.imageInCrypto.cipherAlgo = AES_CIPHER_ALGO;
         updateSettings.imageInCrypto.cipherMode = CIPHER_MODE_CBC;
         updateSettings.imageInCrypto.cipherKey = "aa3ff7d43cc015682c7dfd00de9379e7";
         updateSettings.imageInCrypto.cipherKeyLen = 32;

         //User update settings primary memory configuration
         updateSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
         updateSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
         updateSettings.memories[0].driver = &stm32h5xxFlashDriver;
         updateSettings.memories[0].nbSlots = 3;
         //User settings primary memory slot 0 configuration
         updateSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
         updateSettings.memories[0].slots[0].cType = SLOT_CONTENT_APP;
         updateSettings.memories[0].slots[0].memParent = &updateSettings.memories[0];
         updateSettings.memories[0].slots[0].addr = 0x08020000;
         updateSettings.memories[0].slots[0].size = 0x80000; //0x9E000;//0xA0000;
         //User settings primary memory slot 1 configuration
         updateSettings.memories[0].slots[1].type = SLOT_TYPE_DIRECT;
         updateSettings.memories[0].slots[1].cType = SLOT_CONTENT_UPDATE | SLOT_CONTENT_BACKUP;
         updateSettings.memories[0].slots[1].memParent = &updateSettings.memories[0];
         updateSettings.memories[0].slots[1].addr = 0x080C0000;
         updateSettings.memories[0].slots[1].size = 0x80000; //0x9E000;//0xA0000;
         //User settings primary memory slot 2 configuration
         updateSettings.memories[0].slots[2].type = SLOT_TYPE_DIRECT;
         updateSettings.memories[0].slots[2].cType = SLOT_CONTENT_UPDATE | SLOT_CONTENT_BACKUP;
         updateSettings.memories[0].slots[2].memParent = &updateSettings.memories[0];
         updateSettings.memories[0].slots[2].addr = 0x08160000;
         updateSettings.memories[0].slots[2].size = 0x80000; //0x9E000;//0xA0000;

         //Initialize IAP Application context
         cboot_error = updateInit(&updateContext, &updateSettings);
         //Is any error?
         if (cboot_error != CBOOT_NO_ERROR) {
            //Debug message
            TRACE_ERROR("Failed to initialize update library!\r\n");
         }
      } else {
         TRACE_INFO("Update not available!\r\n");
      }

      //End of exception handling block
   } while(0);

   //Release HTTP client context
   httpClientDeinit(&httpClientContext);

   //Return status code
   return error;
}

/**
 * @brief HTTP client test routine
 * @return Error code
 **/

error_t httpClientReceiveFWUpdate(void)
{
   HttpClientContext httpClientContext;

   error_t error;
   size_t length;
   uint_t status;
   IpAddr ipAddr;
   char_t buffer[512];
   size_t totalBytes = 0;

   cboot_error_t cboot_error = CBOOT_NO_ERROR;

   //Initialize HTTP client context
   httpClientInit(&httpClientContext);

   //Start of exception handling block
   do
   {
      //Debug message
      TRACE_INFO("\r\n\r\nResolving server name...\r\n");

      //Resolve HTTP server name
      error = getHostByName(NULL, UPDATE_SERVER_NAME, &ipAddr, 0);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to resolve server name!\r\n");
         break;
      }

#if (UPDATE_SERVER_PORT == 5000)
      //Register TLS initialization callback
      error = httpClientRegisterTlsInitCallback(&httpClientContext,
         httpClientTlsInitCallback, NULL);
      //Any error to report?
      if(error)
         break;
#endif

      //Select HTTP protocol version
      error = httpClientSetVersion(&httpClientContext, HTTP_VERSION_1_1);
      //Any error to report?
      if(error)
         break;

      //Set timeout value for blocking operations
      error = httpClientSetTimeout(&httpClientContext, 20000);
      //Any error to report?
      if(error)
         break;

      //Debug message
      TRACE_INFO("Connecting to HTTP server %s...\r\n",
         ipAddrToString(&ipAddr, NULL));

      //Connect to the HTTP server
      error = httpClientConnect(&httpClientContext, &ipAddr,
         UPDATE_SERVER_PORT);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to connect to HTTP server!\r\n");
         break;
      }

      //Create an HTTP request
      httpClientCreateRequest(&httpClientContext);
      httpClientSetMethod(&httpClientContext, "GET");
      httpClientSetUri(&httpClientContext, FIRMWARE_UPDATE_URI);

      //Add HTTP header fields
      httpClientAddHeaderField(&httpClientContext, "Host", UPDATE_SERVER_NAME);
      httpClientAddHeaderField(&httpClientContext, "User-Agent", "STM32H5-IoT-Device");

      //Send HTTP request header
      error = httpClientWriteHeader(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to write HTTP request header!\r\n");
         break;
      }

      //Receive HTTP response header
      error = httpClientReadHeader(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to read HTTP response header!\r\n");
         break;
      }

      //Retrieve HTTP status code
      status = httpClientGetStatus(&httpClientContext);
      //Debug message
      TRACE_INFO("HTTP status code: %u\r\n", status);

      //Receive HTTP response body
      while(1)
      {
         //Read data
         error = httpClientReadBody(&httpClientContext, buffer, sizeof(buffer), &length, 0);

         if(!error && length > 0)
         {
            totalBytes += length;

            TRACE_INFO("\rBytes read: %u / %lu", totalBytes, expectedUpdateSize);
            cboot_error = updateProcess(&updateContext,buffer,length);
            //Is any error?
            if(cboot_error != CBOOT_NO_ERROR)
            {
               //Debug message
               TRACE_ERROR("Failed to update firmware!\r\n");
               break;
            }
         }
         else if (error == ERROR_END_OF_STREAM)
         {
            TRACE_INFO("\rBytes read: %u / %lu - Complete!\r\n", totalBytes, expectedUpdateSize);
            error = NO_ERROR; // End of data

            cboot_error = updateFinalize(&updateContext);
            if(cboot_error != CBOOT_NO_ERROR)
            {
               //Debug message
               TRACE_ERROR("Failed to finalize firmware update!\r\n");
            }

            updateCompleted = TRUE;
            break;
         }
         else {
            TRACE_INFO("Failed to read HTTP response body!\r\n");
         }
      }

      //Close HTTP response body
      error = httpClientCloseBody(&httpClientContext);
      //Any error to report?
      if(error)
      {
         //Debug message
         TRACE_INFO("Failed to read HTTP response trailer!\r\n");
         break;
      }

      //Gracefully disconnect from the HTTP server
      httpClientDisconnect(&httpClientContext);

      //Debug message
      TRACE_INFO("Connection closed\r\n");
      TRACE_INFO("Total firmware size: %u bytes\r\n", totalBytes);

      if (updateCompleted == TRUE) {
         cboot_error = updateReboot(&updateContext);
         if(cboot_error != CBOOT_NO_ERROR)
         {
            //Debug message
            TRACE_ERROR("Failed to reboot!\r\n");
         }
      }

      //End of exception handling block
   } while(0);

   //Release HTTP client context
   httpClientDeinit(&httpClientContext);

   //Return status code
   return error;
}


/**
 * @brief User task
 * @param[in] param Unused parameter
 **/

void userTask(void *param)
{
   char_t buffer[40];
#if (IPV4_SUPPORT == ENABLED)
   Ipv4Addr ipv4Addr;
#endif
#if (IPV6_SUPPORT == ENABLED)
   Ipv6Addr ipv6Addr;
#endif

   //Point to the network interface
   NetInterface *interface = &netInterfaces[0];

   //Initialize LCD display
   lcdSetCursor(2, 0);
   printf("IPv4 Addr\r\n");
   lcdSetCursor(5, 0);
   printf("IPv6 Link-Local\r\nAddr\r\n");
   lcdSetCursor(9, 0);
   printf("IPv6 Global Addr\r\n");

   //Endless loop
   while(1)
   {
#if (IPV4_SUPPORT == ENABLED)
      //Display IPv4 host address
      lcdSetCursor(3, 0);
      ipv4GetHostAddr(interface, &ipv4Addr);
      printf("%-16s\r\n", ipv4AddrToString(ipv4Addr, buffer));
#endif

#if (IPV6_SUPPORT == ENABLED)
      //Display IPv6 link-local address
      lcdSetCursor(7, 0);
      ipv6GetLinkLocalAddr(interface, &ipv6Addr);
      printf("%-36s\r\n", ipv6AddrToString(&ipv6Addr, buffer));

      //Display IPv6 global address
      lcdSetCursor(10, 0);
      ipv6GetGlobalAddr(interface, 0, &ipv6Addr);
      printf("%-36s\r\n", ipv6AddrToString(&ipv6Addr, buffer));
#endif

      //User button pressed?
      if(BSP_PB_GetState(BUTTON_USER))
      {
         //HTTP request to update server
         error_t error = httpClientCheckFWUpdate();

         if (!error && updateAvailable) {
            httpClientReceiveFWUpdate();
         }

         //Wait for the user button to be released
         while(BSP_PB_GetState(BUTTON_USER));
      }

      //Loop delay
      osDelayTask(100);
   }
}


/**
 * @brief LED task
 * @param[in] param Unused parameter
 **/

void ledTask(void *param)
{
   //Endless loop
   while(1)
   {
      BSP_LED_On(LED1);
      osDelayTask(100);
      BSP_LED_Off(LED1);
      osDelayTask(900);
   }
}


/**
 * @brief Main entry point
 * @return Unused value
 **/

int_t main(void)
{
   error_t error;
   OsTaskId taskId;
   OsTaskParameters taskParams;
   NetInterface *interface;
   NetSettings netSettings;
   MacAddr macAddr;
#if (APP_USE_DHCP_CLIENT == DISABLED)
   Ipv4Addr ipv4Addr;
#endif
#if (APP_USE_SLAAC == DISABLED)
   Ipv6Addr ipv6Addr;
#endif

   //HAL library initialization
   HAL_Init();
   //Configure the system clock
   SystemClock_Config();

   //Enable instruction cache
   HAL_ICACHE_ConfigAssociativityMode(ICACHE_2WAYS);
   HAL_ICACHE_Enable();

   //Initialize kernel
   osInitKernel();
   //Configure debug UART
   debugInit(115200);

   //Start-up message
   TRACE_INFO("\r\n");
   TRACE_INFO("*****************************************************\r\n");
   TRACE_INFO("*** CycloneBOOT IAP Single-Bank HTTPS Client Demo ***\r\n");
   TRACE_INFO("*****************************************************\r\n");
   TRACE_INFO("Version: %s\r\n", APP_VERSION_STRING);
   TRACE_INFO("Copyright: 2010-2026 Oryx Embedded SARL\r\n");
   TRACE_INFO("Compiled: %s %s\r\n", __DATE__, __TIME__);
   TRACE_INFO("Target: STM32H573\r\n");
   TRACE_INFO("\r\n");

   if (mailBoxIsUpdateConfirmed()) {
      TRACE_INFO("Firmware status: CONFIRMED\n");
   } else {
      TRACE_INFO("Firmware status: CANDIDATE\n");
   }
   TRACE_INFO("\r\n");

   //LED configuration
   BSP_LED_Init(LED1);
   BSP_LED_Init(LED2);
   BSP_LED_Init(LED3);
   BSP_LED_Init(LED4);

   //Clear LEDs
   BSP_LED_Off(LED1);
   BSP_LED_Off(LED2);
   BSP_LED_Off(LED3);
   BSP_LED_Off(LED4);

   //Initialize user button
   BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);

   //Initialize LCD display
   BSP_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
   UTIL_LCD_SetFuncDriver(&LCD_Driver);
   UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_BLUE);
   UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_WHITE);
   UTIL_LCD_SetFont(&Font20);
   BSP_LCD_DisplayOn(0);

   //Clear LCD display
   UTIL_LCD_Clear(UTIL_LCD_COLOR_BLUE);

   //Welcome message
   lcdSetCursor(0, 0);
   printf("HTTPS OTA Demo\r\n");

   //Load IAP public signature key
   error = updateUserLoadKeys();
   if (error) {
      //Debug message
      TRACE_ERROR("Failed to load IAP user keys!\r\n");
      return error;
   }

   //Initialize hardware cryptographic accelerator
   error = stm32h5xxCryptoInit();
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize hardware crypto accelerator!\r\n");
   }

   //Generate a random seed
   error = trngGetRandomData(seed, sizeof(seed));
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to generate random data!\r\n");
   }

   //PRNG initialization
   error = hmacDrbgInit(&hmacDrbgContext, SHA256_HASH_ALGO);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize PRNG!\r\n");
   }

   //Properly seed the PRNG
   error = hmacDrbgSeed(&hmacDrbgContext, seed, sizeof(seed));
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to seed PRNG!\r\n");
   }

   //Get default settings
   netGetDefaultSettings(&netSettings);
   //Network interfaces
   netSettings.interfaces = netInterfaces;
   netSettings.numInterfaces = arraysize(netInterfaces);

   //TCP/IP stack initialization
   error = netInit(&netContext, &netSettings);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize TCP/IP stack!\r\n");
   }

   
   //Start TCP/IP stack
   error = netStart(&netContext);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to start TCP/IP stack!\r\n");
   }

   //Configure the first Ethernet interface
   interface = &netInterfaces[0];

   //Set interface name
   netSetInterfaceName(interface, APP_IF_NAME);
   //Set host name
   netSetHostname(interface, APP_HOST_NAME);
   //Set host MAC address
   macStringToAddr(APP_MAC_ADDR, &macAddr);
   netSetMacAddr(interface, &macAddr);
   //Select the relevant network adapter
   netSetDriver(interface, &stm32h5xxEthDriver);
   netSetPhyDriver(interface, &lan8742PhyDriver);

   //Initialize network interface
   error = netConfigInterface(interface);
   //Any error to report?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to configure interface %s!\r\n", interface->name);
   }

#if (IPV4_SUPPORT == ENABLED)
#if (APP_USE_DHCP_CLIENT == ENABLED)
   //Get default settings
   dhcpClientGetDefaultSettings(&dhcpClientSettings);
   //Set the network interface to be configured by DHCP
   dhcpClientSettings.interface = interface;
   //Disable rapid commit option
   dhcpClientSettings.rapidCommit = FALSE;

   //DHCP client initialization
   error = dhcpClientInit(&dhcpClientContext, &dhcpClientSettings);
   //Failed to initialize DHCP client?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize DHCP client!\r\n");
   }

   //Start DHCP client
   error = dhcpClientStart(&dhcpClientContext);
   //Failed to start DHCP client?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to start DHCP client!\r\n");
   }
#else
   //Set IPv4 host address
   ipv4StringToAddr(APP_IPV4_HOST_ADDR, &ipv4Addr);
   ipv4SetHostAddr(interface, ipv4Addr);

   //Set subnet mask
   ipv4StringToAddr(APP_IPV4_SUBNET_MASK, &ipv4Addr);
   ipv4SetSubnetMask(interface, ipv4Addr);

   //Set default gateway
   ipv4StringToAddr(APP_IPV4_DEFAULT_GATEWAY, &ipv4Addr);
   ipv4SetDefaultGateway(interface, ipv4Addr);

   //Set primary and secondary DNS servers
   ipv4StringToAddr(APP_IPV4_PRIMARY_DNS, &ipv4Addr);
   ipv4SetDnsServer(interface, 0, ipv4Addr);
   ipv4StringToAddr(APP_IPV4_SECONDARY_DNS, &ipv4Addr);
   ipv4SetDnsServer(interface, 1, ipv4Addr);
#endif
#endif

#if (IPV6_SUPPORT == ENABLED)
#if (APP_USE_SLAAC == ENABLED)
   //Get default settings
   slaacGetDefaultSettings(&slaacSettings);
   //Set the network interface to be configured
   slaacSettings.interface = interface;

   //SLAAC initialization
   error = slaacInit(&slaacContext, &slaacSettings);
   //Failed to initialize SLAAC?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to initialize SLAAC!\r\n");
   }

   //Start IPv6 address autoconfiguration process
   error = slaacStart(&slaacContext);
   //Failed to start SLAAC process?
   if(error)
   {
      //Debug message
      TRACE_ERROR("Failed to start SLAAC!\r\n");
   }
#else
   //Set link-local address
   ipv6StringToAddr(APP_IPV6_LINK_LOCAL_ADDR, &ipv6Addr);
   ipv6SetLinkLocalAddr(interface, &ipv6Addr);

   //Set IPv6 prefix
   ipv6StringToAddr(APP_IPV6_PREFIX, &ipv6Addr);
   ipv6SetPrefix(interface, 0, &ipv6Addr, APP_IPV6_PREFIX_LENGTH);

   //Set global address
   ipv6StringToAddr(APP_IPV6_GLOBAL_ADDR, &ipv6Addr);
   ipv6SetGlobalAddr(interface, 0, &ipv6Addr);

   //Set default router
   ipv6StringToAddr(APP_IPV6_ROUTER, &ipv6Addr);
   ipv6SetDefaultRouter(interface, 0, &ipv6Addr);

   //Set primary and secondary DNS servers
   ipv6StringToAddr(APP_IPV6_PRIMARY_DNS, &ipv6Addr);
   ipv6SetDnsServer(interface, 0, &ipv6Addr);
   ipv6StringToAddr(APP_IPV6_SECONDARY_DNS, &ipv6Addr);
   ipv6SetDnsServer(interface, 1, &ipv6Addr);
#endif
#endif

   //Create IAP mutex
   if (!osCreateMutex(&updateMutex)) {
      //Debug message
      TRACE_ERROR("Failed to create IAP mutex!\r\n");
   }

   //Set task parameters
   taskParams = OS_TASK_DEFAULT_PARAMS;
   taskParams.stackSize = 1500;
   taskParams.priority = OS_TASK_PRIORITY_NORMAL;

   //Create user task
   taskId = osCreateTask("User", userTask, NULL, &taskParams);
   //Failed to create the task?
   if(taskId == OS_INVALID_TASK_ID)
   {
      //Debug message
      TRACE_ERROR("Failed to create task!\r\n");
   }

   //Set task parameters
   taskParams = OS_TASK_DEFAULT_PARAMS;
   taskParams.stackSize = 200;
   taskParams.priority = OS_TASK_PRIORITY_NORMAL;

   //Create a task to blink the LED
   taskId = osCreateTask("LED", ledTask, NULL, &taskParams);
   //Failed to create the task?
   if(taskId == OS_INVALID_TASK_ID)
   {
      //Debug message
      TRACE_ERROR("Failed to create task!\r\n");
   }

   //Start the execution of tasks
   osStartKernel();

   //This function should never return
   return 0;
}

// Helpers
static int jsoneq(const char *json, jsmntok_t *tok, const char *s)
{
   if (tok->type == JSMN_STRING &&
       (int)strlen(s) == tok->end - tok->start &&
       strncmp(json + tok->start, s, tok->end - tok->start) == 0)
   {
      return 0;
   }
   return -1;
}

/**
 * @brief Load user RSA public key from "res.c".
 * "res.c" is a C array file system representing the demo folder "resources/keys".
 **/

error_t updateUserLoadKeys(void) {
   error_t error;

   //Get server's public key
   error = resGetData(UPDT_PUBLIC_SIGN_KEY, (const uint8_t **) &pemUpdtSignPublicKey,
                      &pemUpdtSignPublicKeyLen);
   //Any error to report?
   if (error)
      return error;

   //Successful process
   return NO_ERROR;
}