import serial
import time
import sys
import os

def send_firmware(port, baudrate, firmware_path, chunk_size=64, delay=0.001):
    """
    Send a binary firmware file over USB CDC.

    :param port: Serial port (e.g., COM3 or /dev/ttyACM0)
    :param baudrate: Serial baud rate (not critical for USB CDC)
    :param firmware_path: Path to binary firmware file
    :param chunk_size: Number of bytes per write
    :param delay: Delay between writes (seconds)
    """
    if not os.path.exists(firmware_path):
        print("Firmware file not found!")
        return

    ser = serial.Serial(port, baudrate, timeout=1)
    print(f"Sending {firmware_path} to {port} at {baudrate} baud")

    with open(firmware_path, "rb") as f:
        while True:
            chunk = f.read(chunk_size)
            if not chunk:
                break
            ser.write(chunk)
            ser.flush()
            time.sleep(delay)  # small delay to avoid overwhelming MCU

    print("Firmware transmission complete.")
    ser.close()


if __name__ == "__main__":
    if len(sys.argv) != 4:
        print("Usage: python send.py <PORT> <BAUDRATE> <FIRMWARE_FILE>")
        sys.exit(1)

    port = sys.argv[1]
    baudrate = int(sys.argv[2])
    firmware_file = sys.argv[3]

    send_firmware(port, baudrate, firmware_file)
