/**
 * @file helper.c
 * @brief Generic module description
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#include <stdint.h>
#include "stm32h750b_discovery.h"
#include "stm32h750b_discovery_qspi.h"
#include "helper.h"

/**
 * @brief  Flash Memory XiP mode activated ?
 **/

static bool_t flashXiPModeActivated = 0;

/**
 * @brief Initialize flash memory.
 * @return Status code
 **/

error_t flashInit(void)
{
    uint8_t status;

    //Beginning of exceptions handling
    do
    {
        //Deinitialize QSPI external flash memory
        status = BSP_QSPI_DeInit();
        if(status != QSPI_OK)
            break;

        //Initialize QSPI external flash memory
        status = BSP_QSPI_Init();
        if(status != QSPI_OK)
            break;
    }while(0);

    //Is there any error?
    if(status != QSPI_OK)
    {
        //Debug message
        TRACE_INFO("Failed to initialize memory!\r\n");
        return ERROR_FAILURE;
    }

    flashXiPModeActivated = 0;

    //Successful process
    return NO_ERROR;
}


/**
 * @brief Initialize flash memory.
 * @return Status code
 **/

error_t flashActivateXiPMode(bool_t activateXiPMode)
{
    error_t error;
    uint8_t status;

    //Activate XiP mode?
    if(activateXiPMode)
    {
        //Is XiP mode not yet activated?
        if(!flashXiPModeActivated)
        {
            //Activate QSPi external flash memory XiP (eXecute In Place) mode
            status = BSP_QSPI_EnableMemoryMappedMode();
            if(status != QSPI_OK)
            {
                //Debug message
                TRACE_INFO("Failed to setup QSPI in memory mapped mode!\r\n");
                return ERROR_FAILURE;
            }

            //Set flash memory XiP mode activated
            flashXiPModeActivated = 1;
        }
    }
    else
    {
        //Is XiP mode already activated?
        if(flashXiPModeActivated)
        {
            //Initialize flash memory
            error = flashInit();
            //Is any error?
            if(error)
                return error;
            else
                //Set flash memory XiP mode deactivated
                    flashXiPModeActivated = 0;
        }
    }

    //Successful process
    return NO_ERROR;
}
