/**
 * @file main.c
 * @brief Generic module description
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#include <stdio.h>
#include "main.h"
#include "lcd.h"
#include "stm32h750b_discovery.h"
#include "stm32h750b_discovery_lcd.h"
#include "memory.h"
#include "core/cboot_error.h"
#include "resource_manager.h"
#include "debug.h"
#include "helper.h"
#include "uart_user.h"
#include "core/mailbox.h"
#include "drivers/flash/external/mt25tl01g_flash_driver.h"
#include "update/update.h"

#define SDRAM_DEVICE_ADDR  ((uint32_t)0xD0000000)

//LCD frame buffers
#define LCD_FRAME_BUFFER_LAYER0 0xD0400000
#define LCD_FRAME_BUFFER_LAYER1 0xD0480000

//Update signature public key resource path
#define UPDT_PUBLIC_SIGN_KEY "keys/rsa_public_key.pem"

typedef void (*pFunction)(void);

UART_HandleTypeDef huart3;
pFunction JumpToApplication;
uint_t lcdLine = 0;
uint_t lcdColumn = 0;
UpdateSettings updateSettings;
UpdateContext updateContext;
OsMutex updateMutex;
bool_t updateFlag = FALSE;

// Update signature public key declaration
uint8_t *pemUpdtSignPublicKey;
size_t pemUpdtSignPublicKeyLen;

/* Private function prototypes -----------------------------------------------*/
void SystemClock_Config(void);
static void MPU_Config(void);
static void MX_GPIO_Init(void);

/**
 * @brief Set cursor location
 * @param[in] line Line number
 * @param[in] column Column number
 **/

void lcdSetCursor(uint_t line, uint_t column) {
  lcdLine = MIN(line, 11);
  lcdColumn = MIN(column, 30);
}


/**
 * @brief Write a character to the LCD display
 * @param[in] c Character to be written
 **/

void lcdPutChar(char_t c) {
  if (c == '\r') {
    lcdColumn = 0;
  } else if (c == '\n') {
    lcdColumn = 0;
    lcdLine++;
  } else if (lcdLine < 11 && lcdColumn < 30) {
    //Display current character
    BSP_LCD_DisplayChar(lcdColumn * 16, lcdLine * 24, c);

    //Advance the cursor position
    if (++lcdColumn >= 30) {
      lcdColumn = 0;
      lcdLine++;
    }
  }
}

// User defined functions
bool_t bootIsUpdateAvailable(void) {
  TRACE_INFO("Checking if update is available...\r\n");
  return updateFlag;
}

cboot_error_t bootEnableXiPMode(void) {
  error_t error;
  error = flashActivateXiPMode(1);
  if (error)
    return CBOOT_ERROR_FAILURE;
  return CBOOT_NO_ERROR;
}

/**
 * @brief Load user RSA public key from "res.c".
 * "res.c" is a C array file system representing the demo folder "resources/keys".
 **/

error_t updateUserLoadKeys(void) {
  error_t error;

  //Get server's private key
  error = resGetData(UPDT_PUBLIC_SIGN_KEY, (const uint8_t **) &pemUpdtSignPublicKey,
                     &pemUpdtSignPublicKeyLen);
  //Any error to report?
  if (error)
    return error;

  //Successful process
  return NO_ERROR;
}

__attribute__((naked, noreturn) ) void bootAppImageAsm(uint32_t sp, uint32_t rh) {
  __asm("MSR  MSP, r0");
  __asm("BX   r1");
}

/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void) {
  error_t error;
  Image current_image = {0};
  bool_t goToBootloader = FALSE;

  /* Enable I-Cache */
  SCB_EnableICache();
  /* Enable D-Cache */
  SCB_EnableDCache();

  /* MCU Configuration--------------------------------------------------------*/
  /* Reset of all peripherals, Initializes the Flash interface and the Systick. */
  HAL_Init();

  /* Configure the system clock */
  SystemClock_Config();

  /* MPU Configuration--------------------------------------------------------*/
  MPU_Config();

  /* Initialize all configured peripherals */
  MX_GPIO_Init();

  /* configure debug trace */
  //debugInit(115200);

  TRACE_INFO("STM32H750B BSP\r\n");
  TRACE_INFO("CMake Reference Bootloader\r\n");

  BSP_LED_Init(LED_GREEN);
  BSP_LED_Init(LED_RED);

  //Initialize LCD display
  BSP_LCD_Init();
  BSP_LCD_LayerDefaultInit(0, LCD_FRAME_BUFFER_LAYER0);
  BSP_LCD_SelectLayer(0);
  BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
  BSP_LCD_SetTextColor(LCD_COLOR_BLACK);
  BSP_LCD_SetFont(&Font20);
  BSP_LCD_DisplayOn();

  //Clear LCD
  BSP_LCD_Clear(LCD_COLOR_WHITE);

  //Welcome message
  lcdSetCursor(1, 8);
  printf("STM32H750B BSP\r\n");
  lcdSetCursor(2, 2);
  printf("CMake Reference Bootloader\r\n");

  error = updateUserLoadKeys();
  if (error) {
    TRACE_INFO("Error: failed to load user keys.\n");
  }

  //Get default Update settings
  updateGetDefaultSettings(&updateSettings);

  // Update library user settings
#if UPDATE_ANTI_ROLLBACK_SUPPORT == ENABLED
  updateSettings.appVersion = APP_VERSION;
#endif
  //User update settings security configuration
  updateSettings.imageInCrypto.verifySettings.verifyMethod = VERIFY_METHOD_SIGNATURE;
  updateSettings.imageInCrypto.verifySettings.signAlgo = VERIFY_SIGN_RSA;
  updateSettings.imageInCrypto.verifySettings.signHashAlgo = SHA256_HASH_ALGO;
  updateSettings.imageInCrypto.verifySettings.signKey = (const char_t *) pemUpdtSignPublicKey;
  updateSettings.imageInCrypto.verifySettings.signKeyLen = pemUpdtSignPublicKeyLen;
  updateSettings.imageInCrypto.cipherAlgo = AES_CIPHER_ALGO;
  updateSettings.imageInCrypto.cipherMode = CIPHER_MODE_CBC;
  updateSettings.imageInCrypto.cipherKey = (const uint8_t *) "aa3ff7d43cc015682c7dfd00de9379e7";
  updateSettings.imageInCrypto.cipherKeyLen = 32;

  //User update settings primary memory configuration
  //User settings primary memory configuration
  updateSettings.memories[0].memoryType = MEMORY_TYPE_FLASH;
  updateSettings.memories[0].memoryRole = MEMORY_ROLE_PRIMARY;
  updateSettings.memories[0].driver = &mt25tl01gFlashDriver;
  updateSettings.memories[0].nbSlots = 1;
  //User settings primary memory slot 0 configuration
  updateSettings.memories[0].slots[0].type = SLOT_TYPE_DIRECT;
  updateSettings.memories[0].slots[0].cType = SLOT_CONTENT_APP;
  updateSettings.memories[0].slots[0].memParent = &updateSettings.memories[0];
  updateSettings.memories[0].slots[0].addr = APP_START_ADDRESS;
  updateSettings.memories[0].slots[0].size = 0x7D000;

  //Point to the slot where current running image resides
  current_image.activeSlot = &updateSettings.memories[0].slots[0];

  // Initialize the flash using normal QSPI mode (no memory mapping yet)
  flashInit();

  HAL_Delay(1000); // Give a little time for the button press

  // Step 1: Check if an update is explicitly requested
  bool_t updateRequested = mailBoxIsUpdateRequested();
  volatile bool_t updateAvailable = bootIsUpdateAvailable();

  if (updateRequested || updateAvailable) {
    // Clear the update requested flag
    if (updateRequested)
      mailBoxSetUpdateRequested(FALSE);

    goToBootloader = TRUE;
  } else {
    // Step 2: Validate integrity of the current application image
    if (imageCheckIntegrity(&current_image) != CBOOT_NO_ERROR) {
      goToBootloader = TRUE;
    } else {
#if BOOT_XIP_SUPPORT == ENABLED
      // TODO: need re-try logic here
      // TODO: if we are already in XiP mode, we will skip this
      // Step 3: enable XiP mode
      if (bootEnableXiPMode() != CBOOT_NO_ERROR) {
        goToBootloader = TRUE;
        TRACE_ERROR("Booting XIP mode failed\r\n");
      } else {
        // Step 4: Jump to application
        //TODO: perform any necessary clean up operations here (e.g, cache, systick,etc.), before jumping to the app
        BSP_LED_DeInit(LED_GREEN);
        BSP_LED_DeInit(LED_RED);
        BSP_LCD_DeInit();

        SCB_DisableDCache();
        SCB_DisableICache();

        TRACE_INFO("Jumping to application...\r\n");
        // Calculate start address
        uint32_t appStartAddr = current_image.activeSlot->addr + mcuGetVtorOffset();
        mcuJumpToApplication(appStartAddr);
      }
#else
      // Step 3bis: Jump to application
      TRACE_INFO("Jumping to application...\r\n");
      mcuJumpToApplication(APP_START_ADDRESS);
#endif
    }
  }

  if (goToBootloader)
    TRACE_INFO("continuing to bootloader...\r\n");
  /* Infinite loop */
  while (1) {
    uartUserTask(NULL);
    //BSP_LED_Toggle(LED_GREEN);
    //HAL_Delay(250);
  }
}

/**
  * @brief  System Clock Configuration
  *         The system Clock is configured as follow :
  *            System Clock source            = PLL (HSE)
  *            SYSCLK(Hz)                     = 400000000 (Cortex-M7 CPU Clock)
  *            HCLK(Hz)                       = 200000000 (Cortex-M4 CPU, Bus matrix Clocks)
  *            AHB Prescaler                  = 2
  *            D1 APB3 Prescaler              = 2 (APB3 Clock  100MHz)
  *            D2 APB1 Prescaler              = 2 (APB1 Clock  100MHz)
  *            D2 APB2 Prescaler              = 2 (APB2 Clock  100MHz)
  *            D3 APB4 Prescaler              = 2 (APB4 Clock  100MHz)
  *            HSE Frequency(Hz)              = 25000000
  *            PLL_M                          = 5
  *            PLL_N                          = 160
  *            PLL_P                          = 2
  *            PLL_Q                          = 4
  *            PLL_R                          = 2
  *            VDD(V)                         = 3.3
  *            Flash Latency(WS)              = 4
  * @param  None
  * @retval None
  */
void SystemClock_Config(void) {
  RCC_ClkInitTypeDef RCC_ClkInitStruct;
  RCC_OscInitTypeDef RCC_OscInitStruct;
  HAL_StatusTypeDef ret = HAL_OK;

  /* The voltage scaling allows optimizing the power consumption when the device is
     clocked below the maximum system frequency, to update the voltage scaling value
     regarding system frequency refer to product datasheet.  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

  while (!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {
  }

  /* Enable HSE Oscillator and activate PLL with HSE as source */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
  RCC_OscInitStruct.HSEState = RCC_HSE_ON;
  RCC_OscInitStruct.HSIState = RCC_HSI_OFF;
  RCC_OscInitStruct.CSIState = RCC_CSI_OFF;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;

  RCC_OscInitStruct.PLL.PLLM = 5;
  RCC_OscInitStruct.PLL.PLLN = 160;
  RCC_OscInitStruct.PLL.PLLFRACN = 0;
  RCC_OscInitStruct.PLL.PLLP = 2;
  RCC_OscInitStruct.PLL.PLLR = 2;
  RCC_OscInitStruct.PLL.PLLQ = 4;

  RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1VCOWIDE;
  RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1VCIRANGE_2;
  ret = HAL_RCC_OscConfig(&RCC_OscInitStruct);
  if (ret != HAL_OK) {
    Error_Handler();
  }

  /* Select PLL as system clock source and configure  bus clocks dividers */
  RCC_ClkInitStruct.ClockType = (
    RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_D1PCLK1 | RCC_CLOCKTYPE_PCLK1 |
    RCC_CLOCKTYPE_PCLK2 | RCC_CLOCKTYPE_D3PCLK1);

  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.SYSCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_HCLK_DIV2;
  RCC_ClkInitStruct.APB3CLKDivider = RCC_APB3_DIV2;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_APB1_DIV2;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_APB2_DIV2;
  RCC_ClkInitStruct.APB4CLKDivider = RCC_APB4_DIV2;
  ret = HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4);
  if (ret != HAL_OK) {
    Error_Handler();
  }

  /*
   Note : The activation of the I/O Compensation Cell is recommended with communication  interfaces
           (GPIO, SPI, FMC, QSPI ...)  when  operating at  high frequencies(please refer to product datasheet)
           The I/O Compensation Cell activation  procedure requires :
         - The activation of the CSI clock
         - The activation of the SYSCFG clock
         - Enabling the I/O Compensation Cell : setting bit[0] of register SYSCFG_CCCSR
  */

  /*activate CSI clock mondatory for I/O Compensation Cell*/
  __HAL_RCC_CSI_ENABLE();

  /* Enable SYSCFG clock mondatory for I/O Compensation Cell */
  __HAL_RCC_SYSCFG_CLK_ENABLE() ;

  /* Enables the I/O Compensation Cell */
  HAL_EnableCompensationCell();
}

/**
  * @brief GPIO Initialization Function
  * @param None
  * @retval None
  */
static void MX_GPIO_Init(void) {
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* GPIO Ports Clock Enable */
  __HAL_RCC_GPIOC_CLK_ENABLE();
  __HAL_RCC_GPIOI_CLK_ENABLE();
  __HAL_RCC_GPIOH_CLK_ENABLE();
  __HAL_RCC_GPIOB_CLK_ENABLE();
  __HAL_RCC_GPIOJ_CLK_ENABLE();

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(LED2_GPIO_Port, LED2_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(LED1_GPIO_Port, LED1_Pin, GPIO_PIN_RESET);

  /*Configure GPIO pin : USER_BUTTON_Pin */
  GPIO_InitStruct.Pin = USER_BUTTON_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_IT_RISING;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(USER_BUTTON_GPIO_Port, &GPIO_InitStruct);

  /*Configure GPIO pin : LED2_Pin */
  GPIO_InitStruct.Pin = LED2_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(LED2_GPIO_Port, &GPIO_InitStruct);

  /*Configure GPIO pin : LED1_Pin */
  GPIO_InitStruct.Pin = LED1_Pin;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(LED1_GPIO_Port, &GPIO_InitStruct);

  /* EXTI interrupt init*/
  HAL_NVIC_SetPriority(USER_BUTTON_EXTI_IRQn, 7, 0);
  HAL_NVIC_EnableIRQ(USER_BUTTON_EXTI_IRQn);
}

void HAL_GPIO_EXTI_Callback(uint16_t GPIO_Pin) {
  if (GPIO_Pin == GPIO_PIN_13) {
    TRACE_INFO("Button Pressed.\r\n");
    updateFlag = TRUE;
  }
}

/**
 * @brief MPU configuration
 **/

void MPU_Config(void) {
  MPU_Region_InitTypeDef MPU_InitStruct;

  //Disable MPU
  HAL_MPU_Disable();

  //DTCM RAM
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER0;
  MPU_InitStruct.BaseAddress = 0x20000000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_128KB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  //AXI SRAM
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER1;
  MPU_InitStruct.BaseAddress = 0x24000000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_512KB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  //AHB SRAM
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER2;
  MPU_InitStruct.BaseAddress = 0x30000000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_256KB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  //AHB SRAM (no cache)
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER3;
  MPU_InitStruct.BaseAddress = 0x30040000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_32KB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_NOT_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  //SDRAM
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER4;
  MPU_InitStruct.BaseAddress = 0xD0000000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_8MB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_NOT_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  //LCD frame buffer
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER5;
  MPU_InitStruct.BaseAddress = 0xD0400000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_4MB;
  MPU_InitStruct.SubRegionDisable = 0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  /* Configure the MPU QSPI flash */
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.BaseAddress = 0x90000000;
  MPU_InitStruct.Size = MPU_REGION_SIZE_128MB;
  MPU_InitStruct.AccessPermission = MPU_REGION_FULL_ACCESS;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_CACHEABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_NOT_SHAREABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER2;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.SubRegionDisable = 0x0;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_ENABLE;

  //Enable MPU
  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

/**
  * @brief  This function is executed in case of error occurrence.
  * @retval None
  */
void Error_Handler(void) {
  /* USER CODE BEGIN Error_Handler_Debug */
  /* User can add his own implementation to report the HAL error return state */
  __disable_irq();
  while (1) {
  }
  /* USER CODE END Error_Handler_Debug */
}
#ifdef USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line) {
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
}
#endif /* USE_FULL_ASSERT */
