/**
 * @file sc598_crypto_trng.c
 * @brief ADSP-SC598 true random number generator
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include <sys/platform.h>
#include <sys/adi_core.h>
#include <drivers/crypto/adi_trng.h>
#include "core/crypto.h"
#include "hardware/sc598/sc598_crypto.h"
#include "hardware/sc598/sc598_crypto_trng.h"
#include "debug.h"

//Check crypto library configuration
#if (SC598_CRYPTO_TRNG_SUPPORT == ENABLED)

//Global variables
static uint8_t trngMemory[ADI_TRNG_MEMORY_SIZE];
static ADI_TRNG_HANDLE trngHandle;


/**
 * @brief TRNG module initialization
 * @return Error code
 **/

error_t trngInit(void)
{
   ADI_TRNG_RESULT res;

   //Initialize TRNG peripheral
   res = adi_TRNG_Open(&trngHandle, trngMemory, sizeof(trngMemory));

   //Return status code
   return (res == ADI_TRNG_SUCCESS) ? NO_ERROR : ERROR_FAILURE;
}


/**
 * @brief Get random data from the TRNG module
 * @param[out] data Buffer where to store random data
 * @param[in] length Number of random bytes to generate
 **/

error_t trngGetRandomData(uint8_t *data, size_t length)
{
   size_t i;
   uint32_t value;
   ADI_TRNG_RESULT res;

   //Initialize status code
   res = ADI_TRNG_SUCCESS;

   //Acquire exclusive access to the TRNG module
   osAcquireMutex(&sc598CryptoMutex);

   //Generate random data
   for(i = 0; i < length; i++)
   {
      //Generate a new 32-bit random value when necessary
      if((i % 4) == 0)
      {
         //Get 32-bit random value
         res = adi_TRNG_Read_Output(trngHandle, &value, ADI_TRNG_32_BITS);
         //Check status code
         if(res != ADI_TRNG_SUCCESS)
         {
            break;
         }
      }

      //Copy random byte
      data[i] = value & 0xFF;
      //Shift the 32-bit random value
      value >>= 8;
   }

   //Release exclusive access to the TRNG module
   osReleaseMutex(&sc598CryptoMutex);

   //Return status code
   return (res == ADI_TRNG_SUCCESS) ? NO_ERROR : ERROR_FAILURE;
}

#endif
