/**
 * @file shake.c
 * @brief SHAKE128 and SHAKE256 extendable-output functions
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneCRYPTO Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL CRYPTO_TRACE_LEVEL

//Dependencies
#include "core/crypto.h"
#include "xof/shake.h"

//Check crypto library configuration
#if (SHAKE_SUPPORT == ENABLED)

//SHAKE128 object identifier (2.16.840.1.101.3.4.2.11)
const uint8_t SHAKE128_OID[9] = {0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x0B};
//SHAKE256 object identifier (2.16.840.1.101.3.4.2.12)
const uint8_t SHAKE256_OID[9] = {0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x0C};

//Common interface for XOF algorithms (SHAKE128)
const XofAlgo shake128XofAlgo =
{
   "SHAKE128",
   SHAKE128_OID,
   sizeof(SHAKE128_OID),
   sizeof(ShakeContext),
   (XofAlgoCompute) shake128Compute,
   (XofAlgoInit) shake128Init,
   (XofAlgoAbsorb) shakeAbsorb,
   (XofAlgoFinal) shakeFinal,
   (XofAlgoSqueeze) shakeSqueeze
};

//Common interface for XOF algorithms (SHAKE256)
const XofAlgo shake256XofAlgo =
{
   "SHAKE256",
   SHAKE256_OID,
   sizeof(SHAKE256_OID),
   sizeof(ShakeContext),
   (XofAlgoCompute) shake256Compute,
   (XofAlgoInit) shake256Init,
   (XofAlgoAbsorb) shakeAbsorb,
   (XofAlgoFinal) shakeFinal,
   (XofAlgoSqueeze) shakeSqueeze
};


/**
 * @brief Digest a message using SHAKE128 or SHAKE256
 * @param[in] strength Number of bits of security (128 for SHAKE128 and
 *   256 for SHAKE256)
 * @param[in] input Pointer to the input data
 * @param[in] inputLen Length of the input data
 * @param[out] output Pointer to the output data
 * @param[in] outputLen Expected length of the output data
 * @return Error code
 **/

error_t shakeCompute(uint_t strength, const void *input, size_t inputLen,
   uint8_t *output, size_t outputLen)
{
   error_t error;
#if (CRYPTO_STATIC_MEM_SUPPORT == DISABLED)
   ShakeContext *context;
#else
   ShakeContext context[1];
#endif

   //Check parameters
   if(input == NULL && inputLen != 0)
      return ERROR_INVALID_PARAMETER;

   if(output == NULL && outputLen != 0)
      return ERROR_INVALID_PARAMETER;

#if (CRYPTO_STATIC_MEM_SUPPORT == DISABLED)
   //Allocate a memory buffer to hold the SHAKE context
   context = cryptoAllocMem(sizeof(ShakeContext));
   //Failed to allocate memory?
   if(context == NULL)
      return ERROR_OUT_OF_MEMORY;
#endif

   //Initialize the SHAKE context
   error = shakeInit(context, strength);

   //Check status code
   if(!error)
   {
      //Absorb input data
      shakeAbsorb(context, input, inputLen);
      //Finish absorbing phase
      shakeFinal(context);
      //Extract data from the squeezing phase
      shakeSqueeze(context, output, outputLen);
   }

#if (CRYPTO_STATIC_MEM_SUPPORT == DISABLED)
   //Free previously allocated memory
   cryptoFreeMem(context);
#endif

   //Return status code
   return error;
}


/**
 * @brief Digest a message using SHAKE128
 * @param[in] input Pointer to the input data
 * @param[in] inputLen Length of the input data
 * @param[out] output Pointer to the output data
 * @param[in] outputLen Expected length of the output data
 * @return Error code
 **/

error_t shake128Compute(const void *input, size_t inputLen, uint8_t *output,
   size_t outputLen)
{
   //Digest the message using SHAKE128
   return shakeCompute(128, input, inputLen, output, outputLen);
}


/**
 * @brief Digest a message using SHAKE256
 * @param[in] input Pointer to the input data
 * @param[in] inputLen Length of the input data
 * @param[out] output Pointer to the output data
 * @param[in] outputLen Expected length of the output data
 * @return Error code
 **/

error_t shake256Compute(const void *input, size_t inputLen, uint8_t *output,
   size_t outputLen)
{
   //Digest the message using SHAKE256
   return shakeCompute(256, input, inputLen, output, outputLen);
}


/**
 * @brief Initialize SHAKE context
 * @param[in] context Pointer to the SHAKE context to initialize
 * @param[in] strength Number of bits of security (128 for SHAKE128 and
 *   256 for SHAKE256)
 * @return Error code
 **/

error_t shakeInit(ShakeContext *context, uint_t strength)
{
   error_t error;

   //SHAKE128 and SHAKE256 provides respectively 128 and 256 bits of security
   if(strength == 128 || strength == 256)
   {
      //Initialize Keccak context
      error = keccakInit(&context->keccakContext, 2 * strength);
   }
   else
   {
      //Report an error
      error = ERROR_INVALID_PARAMETER;
   }

   //Return status code
   return error;
}


/**
 * @brief Initialize SHAKE128 context
 * @param[in] context Pointer to the SHAKE context to initialize
 * @return Error code
 **/

error_t shake128Init(ShakeContext *context)
{
   //Initialize SHAKE128 context
   return shakeInit(context, 128);
}


/**
 * @brief Initialize SHAKE256 context
 * @param[in] context Pointer to the SHAKE context to initialize
 * @return Error code
 **/

error_t shake256Init(ShakeContext *context)
{
   //Initialize SHAKE256 context
   return shakeInit(context, 256);
}


/**
 * @brief Absorb data
 * @param[in] context Pointer to the SHAKE context
 * @param[in] input Pointer to the buffer being hashed
 * @param[in] length Length of the buffer
 **/

void shakeAbsorb(ShakeContext *context, const void *input, size_t length)
{
   //Absorb the input data
   keccakAbsorb(&context->keccakContext, input, length);
}


/**
 * @brief Finish absorbing phase
 * @param[in] context Pointer to the SHAKE context
 **/

void shakeFinal(ShakeContext *context)
{
   //Finish absorbing phase (padding byte is 0x1F for XOFs)
   keccakFinal(&context->keccakContext, KECCAK_SHAKE_PAD);
}


/**
 * @brief Extract data from the squeezing phase
 * @param[in] context Pointer to the SHAKE context
 * @param[out] output Output string
 * @param[in] length Desired output length, in bytes
 **/

void shakeSqueeze(ShakeContext *context, uint8_t *output, size_t length)
{
   //Extract data from the squeezing phase
   keccakSqueeze(&context->keccakContext, output, length);
}

#endif
