/**
 * @file gpy111_driver.h
 * @brief GPY111 Gigabit Ethernet PHY driver
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#ifndef _GPY111_DRIVER_H
#define _GPY111_DRIVER_H

//Dependencies
#include "core/nic.h"

//PHY address
#ifndef GPY111_PHY_ADDR
   #define GPY111_PHY_ADDR 0
#elif (GPY111_PHY_ADDR < 0 || GPY111_PHY_ADDR > 31)
   #error GPY111_PHY_ADDR parameter is not valid
#endif

//GPY111 PHY registers
#define GPY111_CTRL                        0x00
#define GPY111_STAT                        0x01
#define GPY111_PHYID1                      0x02
#define GPY111_PHYID2                      0x03
#define GPY111_AN_ADV                      0x04
#define GPY111_AN_LPA                      0x05
#define GPY111_AN_EXP                      0x06
#define GPY111_AN_NPTX                     0x07
#define GPY111_AN_NPRX                     0x08
#define GPY111_GCTRL                       0x09
#define GPY111_GSTAT                       0x0A
#define GPY111_RES11                       0x0B
#define GPY111_RES12                       0x0C
#define GPY111_MMDCTRL                     0x0D
#define GPY111_MMDDATA                     0x0E
#define GPY111_XSTAT                       0x0F
#define GPY111_PHYPHYPERF                  0x10
#define GPY111_PHYSTAT1                    0x11
#define GPY111_PHYSTAT2                    0x12
#define GPY111_PHYCTL1                     0x13
#define GPY111_PHYCTL2                     0x14
#define GPY111_ERRCNT                      0x15
#define GPY111_EECTRL                      0x16
#define GPY111_MIICTRL                     0x17
#define GPY111_MIISTAT                     0x18
#define GPY111_IMASK                       0x19
#define GPY111_ISTAT                       0x1A
#define GPY111_LED                         0x1B
#define GPY111_TPGCTRL                     0x1C
#define GPY111_TPGDATA                     0x1D
#define GPY111_FWV                         0x1E
#define GPY111_RES1F                       0x1F

//GPY111 MMD registers
#define GPY111_EEE_CTRL1                   0x03, 0x0000
#define GPY111_EEE_STAT1                   0x03, 0x0001
#define GPY111_EEE_CAP                     0x03, 0x0014
#define GPY111_EEE_WAKERR                  0x03, 0x0016
#define GPY111_ANEGEEE_AN_ADV              0x07, 0x003C
#define GPY111_EEE_AN_LPADV                0x07, 0x003D
#define GPY111_EEPROM                      0x1E, 0x0000
#define GPY111_LEDCH                       0x1F, 0x01E0
#define GPY111_LEDCL                       0x1F, 0x01E1
#define GPY111_LED0H                       0x1F, 0x01E2
#define GPY111_LED0L                       0x1F, 0x01E3
#define GPY111_LED1H                       0x1F, 0x01E4
#define GPY111_LED1L                       0x1F, 0x01E5
#define GPY111_LED2H                       0x1F, 0x01E6
#define GPY111_LED2L                       0x1F, 0x01E7
#define GPY111_EEE_RXERR_LINK_FAIL_H       0x1F, 0x01EA
#define GPY111_EEE_RXERR_LINK_FAIL_L       0x1F, 0x01EB
#define GPY111_MII2CTRL                    0x1F, 0x01EC
#define GPY111_LEG_LPI_CFG0                0x1F, 0x01ED
#define GPY111_LEG_LPI_CFG1                0x1F, 0x01EE
#define GPY111_WOLCTRL                     0x1F, 0x0781
#define GPY111_WOLAD0                      0x1F, 0x0783
#define GPY111_WOLAD1                      0x1F, 0x0784
#define GPY111_WOLAD2                      0x1F, 0x0785
#define GPY111_WOLAD3                      0x1F, 0x0786
#define GPY111_WOLAD4                      0x1F, 0x0787
#define GPY111_WOLAD5                      0x1F, 0x0788
#define GPY111_WOLPW0                      0x1F, 0x0789
#define GPY111_WOLPW1                      0x1F, 0x078A
#define GPY111_WOLPW2                      0x1F, 0x078B
#define GPY111_WOLPW3                      0x1F, 0x078C
#define GPY111_WOLPW4                      0x1F, 0x078D
#define GPY111_WOLPW5                      0x1F, 0x078E
#define GPY111_LEG_LPI_CFG2                0x1F, 0x0EB5
#define GPY111_LEG_LPI_CFG3                0x1F, 0x0EB7

//Control register
#define GPY111_CTRL_RST                    0x8000
#define GPY111_CTRL_LB                     0x4000
#define GPY111_CTRL_SSL                    0x2000
#define GPY111_CTRL_ANEN                   0x1000
#define GPY111_CTRL_PD                     0x0800
#define GPY111_CTRL_ISOL                   0x0400
#define GPY111_CTRL_ANRS                   0x0200
#define GPY111_CTRL_DPLX                   0x0100
#define GPY111_CTRL_COL                    0x0080
#define GPY111_CTRL_SSM                    0x0040

//Status register
#define GPY111_STAT_CBT4                   0x8000
#define GPY111_STAT_CBTXF                  0x4000
#define GPY111_STAT_CBTXH                  0x2000
#define GPY111_STAT_XBTF                   0x1000
#define GPY111_STAT_XBTH                   0x0800
#define GPY111_STAT_CBT2F                  0x0400
#define GPY111_STAT_CBT2H                  0x0200
#define GPY111_STAT_EXT                    0x0100
#define GPY111_STAT_MFPS                   0x0040
#define GPY111_STAT_ANOK                   0x0020
#define GPY111_STAT_RF                     0x0010
#define GPY111_STAT_ANAB                   0x0008
#define GPY111_STAT_LS                     0x0004
#define GPY111_STAT_JD                     0x0002
#define GPY111_STAT_XCAP                   0x0001

//PHY Identifier 1 register
#define GPY111_PHYID1_OUI_MSB              0xFFFF
#define GPY111_PHYID1_OUI_MSB_DEFAULT      0x0000

//PHY Identifier 2 register
#define GPY111_PHYID2_OUI_LSB              0xFC00
#define GPY111_PHYID2_OUI_LSB_DEFAULT      0x0000
#define GPY111_PHYID2_LDN                  0x03F0
#define GPY111_PHYID2_LDN_DEFAULT          0x0000
#define GPY111_PHYID2_LDRN                 0x000F

//Auto-Negotiation Advertisement register
#define GPY111_AN_ADV_NP                   0x8000
#define GPY111_AN_ADV_RF                   0x2000
#define GPY111_AN_ADV_TAF                  0x1FE0
#define GPY111_AN_ADV_TAF_XBT_HDX          0x0020
#define GPY111_AN_ADV_TAF_XBT_FDX          0x0040
#define GPY111_AN_ADV_TAF_DBT_HDX          0x0080
#define GPY111_AN_ADV_TAF_DBT_FDX          0x0100
#define GPY111_AN_ADV_TAF_DBT4             0x0200
#define GPY111_AN_ADV_TAF_PS_SYM           0x0400
#define GPY111_AN_ADV_TAF_PS_ASYM          0x0800
#define GPY111_AN_ADV_TAF_RES              0x1000
#define GPY111_AN_ADV_SF                   0x001F
#define GPY111_AN_ADV_SF_DEFAULT           0x0001

//Auto-Negotiation Link-Partner Ability register
#define GPY111_AN_LPA_NP                   0x8000
#define GPY111_AN_LPA_ACK                  0x4000
#define GPY111_AN_LPA_RF                   0x2000
#define GPY111_AN_LPA_TAF                  0x1FE0
#define GPY111_AN_LPA_TAF_XBT_HDX          0x0020
#define GPY111_AN_LPA_TAF_XBT_FDX          0x0040
#define GPY111_AN_LPA_TAF_DBT_HDX          0x0080
#define GPY111_AN_LPA_TAF_DBT_FDX          0x0100
#define GPY111_AN_LPA_TAF_DBT4             0x0200
#define GPY111_AN_LPA_TAF_PS_SYM           0x0400
#define GPY111_AN_LPA_TAF_PS_ASYM          0x0800
#define GPY111_AN_LPA_TAF_RES              0x1000
#define GPY111_AN_LPA_SF                   0x001F
#define GPY111_AN_LPA_SF_DEFAULT           0x0001

//Auto-Negotiation Expansion register
#define GPY111_AN_EXP_RESD                 0xFFE0
#define GPY111_AN_EXP_PDF                  0x0010
#define GPY111_AN_EXP_LPNPC                0x0008
#define GPY111_AN_EXP_NPC                  0x0004
#define GPY111_AN_EXP_PR                   0x0002
#define GPY111_AN_EXP_LPANC                0x0001

//Auto-Negotiation Next-Page Transmit register
#define GPY111_AN_NPTX_NP                  0x8000
#define GPY111_AN_NPTX_MP                  0x2000
#define GPY111_AN_NPTX_ACK2                0x1000
#define GPY111_AN_NPTX_TOGG                0x0800
#define GPY111_AN_NPTX_MCF                 0x07FF

//Auto-Negotiation Link-Partner Received Next-Page register
#define GPY111_AN_NPRX_NP                  0x8000
#define GPY111_AN_NPRX_ACK                 0x4000
#define GPY111_AN_NPRX_MP                  0x2000
#define GPY111_AN_NPRX_ACK2                0x1000
#define GPY111_AN_NPRX_TOGG                0x0800
#define GPY111_AN_NPRX_MCF                 0x07FF

//Gigabit Control register
#define GPY111_GCTRL_TM                    0xE000
#define GPY111_GCTRL_MSEN                  0x1000
#define GPY111_GCTRL_MS                    0x0800
#define GPY111_GCTRL_MSPT                  0x0400
#define GPY111_GCTRL_MBTFD                 0x0200
#define GPY111_GCTRL_MBTHD                 0x0100

//Gigabit Status register
#define GPY111_GSTAT_MSFAULT               0x8000
#define GPY111_GSTAT_MSRES                 0x4000
#define GPY111_GSTAT_LRXSTAT               0x2000
#define GPY111_GSTAT_RRXSTAT               0x1000
#define GPY111_GSTAT_MBTFD                 0x0800
#define GPY111_GSTAT_MBTHD                 0x0400
#define GPY111_GSTAT_IEC                   0x00FF

//MMD Access Control register
#define GPY111_MMDCTRL_ACTYPE              0xC000
#define GPY111_MMDCTRL_ACTYPE_ADDR         0x0000
#define GPY111_MMDCTRL_ACTYPE_DATA         0x4000
#define GPY111_MMDCTRL_ACTYPE_DATA_PI      0x8000
#define GPY111_MMDCTRL_ACTYPE_DATA_PIWR    0xC000
#define GPY111_MMDCTRL_RESH                0x3F00
#define GPY111_MMDCTRL_RESL                0x00E0
#define GPY111_MMDCTRL_DEVAD               0x001F

//MMD Access Data register
#define GPY111_MMDDATA_ADDR_DATA           0xFFFF

//Extended Status register
#define GPY111_XSTAT_MBXF                  0x8000
#define GPY111_XSTAT_MBXH                  0x4000
#define GPY111_XSTAT_MBTF                  0x2000
#define GPY111_XSTAT_MBTH                  0x1000
#define GPY111_XSTAT_RESH                  0x0F00
#define GPY111_XSTAT_RESL                  0x00FF

//Physical Layer Performance Status register
#define GPY111_PHYPHYPERF_FREQ             0xFF00
#define GPY111_PHYPHYPERF_SNR              0x00F0
#define GPY111_PHYPHYPERF_LEN              0x000F

//Physical Layer Status 1 register
#define GPY111_PHYSTAT1_RESH               0xFE00
#define GPY111_PHYSTAT1_LSADS              0x0100
#define GPY111_PHYSTAT1_POLD               0x0080
#define GPY111_PHYSTAT1_POLC               0x0040
#define GPY111_PHYSTAT1_POLB               0x0020
#define GPY111_PHYSTAT1_POLA               0x0010
#define GPY111_PHYSTAT1_MDICD              0x0008
#define GPY111_PHYSTAT1_MDIAB              0x0004
#define GPY111_PHYSTAT1_RESL               0x0003

//Physical Layer Status 2 register
#define GPY111_PHYSTAT2_RESD               0x8000
#define GPY111_PHYSTAT2_SKEWD              0x7000
#define GPY111_PHYSTAT2_RESC               0x0800
#define GPY111_PHYSTAT2_SKEWC              0x0700
#define GPY111_PHYSTAT2_RESB               0x0080
#define GPY111_PHYSTAT2_SKEWB              0x0070
#define GPY111_PHYSTAT2_RESA               0x0008
#define GPY111_PHYSTAT2_SKEWA              0x0007

//Physical Layer Control 1 register
#define GPY111_PHYCTL1_TLOOP               0xE000
#define GPY111_PHYCTL1_TXOFF               0x1000
#define GPY111_PHYCTL1_TXADJ               0x0F00
#define GPY111_PHYCTL1_POLD                0x0080
#define GPY111_PHYCTL1_POLC                0x0040
#define GPY111_PHYCTL1_POLB                0x0020
#define GPY111_PHYCTL1_POLA                0x0010
#define GPY111_PHYCTL1_MDICD               0x0008
#define GPY111_PHYCTL1_MDIAB               0x0004
#define GPY111_PHYCTL1_TXEEE10             0x0002
#define GPY111_PHYCTL1_AMDIX               0x0001

//Physical Layer Control 2 register
#define GPY111_PHYCTL2_LSADS               0xC000
#define GPY111_PHYCTL2_LSADS_OFF           0x0000
#define GPY111_PHYCTL2_LSADS_ADS2          0x4000
#define GPY111_PHYCTL2_LSADS_ADS3          0x8000
#define GPY111_PHYCTL2_LSADS_ADS4          0xC000
#define GPY111_PHYCTL2_RESH                0x3800
#define GPY111_PHYCTL2_CLKSEL              0x0400
#define GPY111_PHYCTL2_CLKSEL_CLK25M       0x0000
#define GPY111_PHYCTL2_CLKSEL_CLK125M      0x0400
#define GPY111_PHYCTL2_SDETP               0x0200
#define GPY111_PHYCTL2_SDETP_LOWACTIVE     0x0000
#define GPY111_PHYCTL2_SDETP_HIGHACTIVE    0x0200
#define GPY111_PHYCTL2_STICKY              0x0100
#define GPY111_PHYCTL2_RESL                0x00F0
#define GPY111_PHYCTL2_ADCR                0x0008
#define GPY111_PHYCTL2_ADCR_DEFAULT        0x0000
#define GPY111_PHYCTL2_ADCR_BOOST          0x0008
#define GPY111_PHYCTL2_PSCL                0x0004
#define GPY111_PHYCTL2_ANPD                0x0002
#define GPY111_PHYCTL2_LPI                 0x0001

//Error Counter register
#define GPY111_ERRCNT_SEL                  0x0F00
#define GPY111_ERRCNT_SEL_RXERR            0x0000
#define GPY111_ERRCNT_SEL_RXACT            0x0100
#define GPY111_ERRCNT_SEL_ESDERR           0x0200
#define GPY111_ERRCNT_SEL_SSDERR           0x0300
#define GPY111_ERRCNT_SEL_TXERR            0x0400
#define GPY111_ERRCNT_SEL_TXACT            0x0500
#define GPY111_ERRCNT_SEL_COL              0x0600
#define GPY111_ERRCNT_COUNT                0x00FF

//EEPROM Control register
#define GPY111_EECTRL_EESCAN               0x8000
#define GPY111_EECTRL_EEAF                 0x4000
#define GPY111_EECTRL_CSRDET               0x2000
#define GPY111_EECTRL_EEDET                0x1000
#define GPY111_EECTRL_SIZE                 0x0F00
#define GPY111_EECTRL_SIZE_SIZE1K          0x0000
#define GPY111_EECTRL_SIZE_SIZE2K          0x0100
#define GPY111_EECTRL_SIZE_SIZE4K          0x0200
#define GPY111_EECTRL_SIZE_SIZE8K          0x0300
#define GPY111_EECTRL_SIZE_SIZE16K         0x0400
#define GPY111_EECTRL_SIZE_SIZE32K         0x0500
#define GPY111_EECTRL_SIZE_SIZE64K         0x0600
#define GPY111_EECTRL_SIZE_SIZE128K        0x0700
#define GPY111_EECTRL_SIZE_SIZE256K        0x0800
#define GPY111_EECTRL_SIZE_SIZE512K        0x0900
#define GPY111_EECTRL_SIZE_SIZE1024K       0x0A00
#define GPY111_EECTRL_ADRMODE              0x0080
#define GPY111_EECTRL_ADRMODE_MODE11       0x0000
#define GPY111_EECTRL_ADRMODE_MODE16       0x0080
#define GPY111_EECTRL_DADR                 0x0070
#define GPY111_EECTRL_SPEED                0x000C
#define GPY111_EECTRL_SPEED_FRQ_100KHZ     0x0000
#define GPY111_EECTRL_SPEED_FRQ_400KHZ     0x0004
#define GPY111_EECTRL_SPEED_FRQ_1_0MHZ     0x0008
#define GPY111_EECTRL_SPEED_FRQ_3_4MHZ     0x000C
#define GPY111_EECTRL_RDWR                 0x0002
#define GPY111_EECTRL_EXEC                 0x0001

//Media-Independent Interface Control register
#define GPY111_MIICTRL_RXCOFF              0x8000
#define GPY111_MIICTRL_RXSKEW              0x7000
#define GPY111_MIICTRL_RXSKEW_SKEW_0N0     0x0000
#define GPY111_MIICTRL_RXSKEW_SKEW_0N5     0x1000
#define GPY111_MIICTRL_RXSKEW_SKEW_1N0     0x2000
#define GPY111_MIICTRL_RXSKEW_SKEW_1N5     0x3000
#define GPY111_MIICTRL_RXSKEW_SKEW_2N0     0x4000
#define GPY111_MIICTRL_RXSKEW_SKEW_2N5     0x5000
#define GPY111_MIICTRL_RXSKEW_SKEW_3N0     0x6000
#define GPY111_MIICTRL_RXSKEW_SKEW_3N5     0x7000
#define GPY111_MIICTRL_V25_33              0x0800
#define GPY111_MIICTRL_TXSKEW              0x0700
#define GPY111_MIICTRL_TXSKEW_SKEW_0N0     0x0000
#define GPY111_MIICTRL_TXSKEW_SKEW_0N5     0x0100
#define GPY111_MIICTRL_TXSKEW_SKEW_1N0     0x0200
#define GPY111_MIICTRL_TXSKEW_SKEW_1N5     0x0300
#define GPY111_MIICTRL_TXSKEW_SKEW_2N0     0x0400
#define GPY111_MIICTRL_TXSKEW_SKEW_2N5     0x0500
#define GPY111_MIICTRL_TXSKEW_SKEW_3N0     0x0600
#define GPY111_MIICTRL_TXSKEW_SKEW_3N5     0x0700
#define GPY111_MIICTRL_CRS                 0x00C0
#define GPY111_MIICTRL_FLOW                0x0030
#define GPY111_MIICTRL_FLOW_COPPER         0x0000
#define GPY111_MIICTRL_FLOW_CONVERTER      0x0030
#define GPY111_MIICTRL_MODE                0x000F
#define GPY111_MIICTRL_MODE_RGMII          0x0000
#define GPY111_MIICTRL_MODE_SGMII          0x0001
#define GPY111_MIICTRL_MODE_RMII           0x0002
#define GPY111_MIICTRL_MODE_RTBI           0x0003
#define GPY111_MIICTRL_MODE_GMII           0x0004
#define GPY111_MIICTRL_MODE_TBI            0x0005
#define GPY111_MIICTRL_MODE_SGMIINC        0x0006
#define GPY111_MIICTRL_MODE_TEST           0x000F
#define GPY111_MIICTRL_MODE_CONV_X2T1000   0x0000
#define GPY111_MIICTRL_MODE_CONV_X2T1000A  0x0001

//Media-Independent Interface Status register
#define GPY111_MIISTAT_RESH                0xFF00
#define GPY111_MIISTAT_PHY                 0x00C0
#define GPY111_MIISTAT_PHY_TP              0x0000
#define GPY111_MIISTAT_PHY_FIBER           0x0040
#define GPY111_MIISTAT_PHY_MII2            0x0080
#define GPY111_MIISTAT_PHY_SGMII           0x00C0
#define GPY111_MIISTAT_PS                  0x0030
#define GPY111_MIISTAT_PS_NONE             0x0000
#define GPY111_MIISTAT_PS_TX               0x0010
#define GPY111_MIISTAT_PS_RX               0x0020
#define GPY111_MIISTAT_PS_TXRX             0x0030
#define GPY111_MIISTAT_DPX                 0x0008
#define GPY111_MIISTAT_EEE                 0x0004
#define GPY111_MIISTAT_EEE_OFF             0x0000
#define GPY111_MIISTAT_EEE_ON              0x0004
#define GPY111_MIISTAT_SPEED               0x0003
#define GPY111_MIISTAT_SPEED_TEN           0x0000
#define GPY111_MIISTAT_SPEED_FAST          0x0001
#define GPY111_MIISTAT_SPEED_GIGA          0x0002
#define GPY111_MIISTAT_SPEED_RES           0x0003

//Interrupt Mask register
#define GPY111_IMASK_WOL                   0x8000
#define GPY111_IMASK_MSRE                  0x4000
#define GPY111_IMASK_NPRX                  0x2000
#define GPY111_IMASK_NPTX                  0x1000
#define GPY111_IMASK_ANE                   0x0800
#define GPY111_IMASK_ANC                   0x0400
#define GPY111_IMASK_RESH                  0x0300
#define GPY111_IMASK_RESL                  0x00C0
#define GPY111_IMASK_ADSC                  0x0020
#define GPY111_IMASK_MDIPC                 0x0010
#define GPY111_IMASK_MDIXC                 0x0008
#define GPY111_IMASK_DXMC                  0x0004
#define GPY111_IMASK_LSPC                  0x0002
#define GPY111_IMASK_LSTC                  0x0001

//Interrupt Status register
#define GPY111_ISTAT_WOL                   0x8000
#define GPY111_ISTAT_MSRE                  0x4000
#define GPY111_ISTAT_NPRX                  0x2000
#define GPY111_ISTAT_NPTX                  0x1000
#define GPY111_ISTAT_ANE                   0x0800
#define GPY111_ISTAT_ANC                   0x0400
#define GPY111_ISTAT_RESH                  0x0300
#define GPY111_ISTAT_RESL                  0x00C0
#define GPY111_ISTAT_ADSC                  0x0020
#define GPY111_ISTAT_MDIPC                 0x0010
#define GPY111_ISTAT_MDIXC                 0x0008
#define GPY111_ISTAT_DXMC                  0x0004
#define GPY111_ISTAT_LSPC                  0x0002
#define GPY111_ISTAT_LSTC                  0x0001

//LED Control register
#define GPY111_LED_RESH                    0xF000
#define GPY111_LED_LED3EN                  0x0800
#define GPY111_LED_LED2EN                  0x0400
#define GPY111_LED_LED1EN                  0x0200
#define GPY111_LED_LED0EN                  0x0100
#define GPY111_LED_RESL                    0x00F0
#define GPY111_LED_LED3DA                  0x0008
#define GPY111_LED_LED3DA_OFF              0x0000
#define GPY111_LED_LED3DA_ON               0x0008
#define GPY111_LED_LED2DA                  0x0004
#define GPY111_LED_LED2DA_OFF              0x0000
#define GPY111_LED_LED2DA_ON               0x0004
#define GPY111_LED_LED1DA                  0x0002
#define GPY111_LED_LED1DA_OFF              0x0000
#define GPY111_LED_LED1DA_ON               0x0002
#define GPY111_LED_LED0DA                  0x0001
#define GPY111_LED_LED0DA_OFF              0x0000
#define GPY111_LED_LED0DA_ON               0x0001

//Test-Packet Generator Control register
#define GPY111_TPGCTRL_RESH1               0xC000
#define GPY111_TPGCTRL_MODE                0x2000
#define GPY111_TPGCTRL_MODE_BURST          0x0000
#define GPY111_TPGCTRL_MODE_SINGLE         0x2000
#define GPY111_TPGCTRL_RESH0               0x1000
#define GPY111_TPGCTRL_IPGL                0x0C00
#define GPY111_TPGCTRL_IPGL_BT48           0x0000
#define GPY111_TPGCTRL_IPGL_BT96           0x0400
#define GPY111_TPGCTRL_IPGL_BT960          0x0800
#define GPY111_TPGCTRL_IPGL_BT9600         0x0C00
#define GPY111_TPGCTRL_TYPE                0x0300
#define GPY111_TPGCTRL_TYPE_RANDOM         0x0000
#define GPY111_TPGCTRL_TYPE_BYTEINC        0x0100
#define GPY111_TPGCTRL_TYPE_PREDEF         0x0200
#define GPY111_TPGCTRL_RESL1               0x0080
#define GPY111_TPGCTRL_SIZE                0x0070
#define GPY111_TPGCTRL_SIZE_B64            0x0000
#define GPY111_TPGCTRL_SIZE_B128           0x0010
#define GPY111_TPGCTRL_SIZE_B256           0x0020
#define GPY111_TPGCTRL_SIZE_B512           0x0030
#define GPY111_TPGCTRL_SIZE_B1024          0x0040
#define GPY111_TPGCTRL_SIZE_B1518          0x0050
#define GPY111_TPGCTRL_SIZE_B9600          0x0060
#define GPY111_TPGCTRL_RESL0               0x000C
#define GPY111_TPGCTRL_START               0x0002
#define GPY111_TPGCTRL_EN                  0x0001

//Test-Packet Generator Data register
#define GPY111_TPGDATA_DA                  0xF000
#define GPY111_TPGDATA_SA                  0x0F00
#define GPY111_TPGDATA_DATA                0x00FF

//Firmware Version register
#define GPY111_FWV_REL                     0x8000
#define GPY111_FWV_REL_TEST                0x0000
#define GPY111_FWV_REL_RELEASE             0x8000
#define GPY111_FWV_MAJOR                   0x7F00
#define GPY111_FWV_MINOR                   0x00FF

//EEE Control 1 register
#define GPY111_EEE_CTRL1_RXCKST            0x0400

//EEE Status 1 register
#define GPY111_EEE_STAT1_TXLPI_RCVD        0x0800
#define GPY111_EEE_STAT1_TXLPI_IND         0x0200
#define GPY111_EEE_STAT1_RXLPI_IND         0x0100
#define GPY111_EEE_STAT1_TXCKST            0x0040

//EEE Capability register
#define GPY111_EEE_CAP_EEE_10GBKR          0x0040
#define GPY111_EEE_CAP_EEE_10GBKX4         0x0020
#define GPY111_EEE_CAP_EEE_1000BKX         0x0010
#define GPY111_EEE_CAP_EEE_10GBT           0x0008
#define GPY111_EEE_CAP_EEE_1000BT          0x0004
#define GPY111_EEE_CAP_EEE_100BTX          0x0002

//EEE Wake Time Fault Count register
#define GPY111_EEE_WAKERR_ERRCNT           0xFFFF

//EEE Auto-Negotiation Advertisement register
#define GPY111_ANEGEEE_AN_ADV_EEE_10GBKR   0x0040
#define GPY111_ANEGEEE_AN_ADV_EEE_10GBKX4  0x0020
#define GPY111_ANEGEEE_AN_ADV_EEE_1000BKX  0x0010
#define GPY111_ANEGEEE_AN_ADV_EEE_10GBT    0x0008
#define GPY111_ANEGEEE_AN_ADV_EEE_1000BT   0x0004
#define GPY111_ANEGEEE_AN_ADV_EEE_100BTX   0x0002

//EEE Auto-Negotiation Link-Partner Advertisement register
#define GPY111_EEE_AN_LPADV_EEE_10GBKR     0x0040
#define GPY111_EEE_AN_LPADV_EEE_10GBKX4    0x0020
#define GPY111_EEE_AN_LPADV_EEE_1000BKX    0x0010
#define GPY111_EEE_AN_LPADV_EEE_10GBT      0x0008
#define GPY111_EEE_AN_LPADV_EEE_1000BT     0x0004
#define GPY111_EEE_AN_LPADV_EEE_100BTX     0x0002

//EEPROM Content register
#define GPY111_EEPROM_DATA                 0x00FF

//LED Configuration H register
#define GPY111_LEDCH_FBF                   0x00C0
#define GPY111_LEDCH_FBF_F02HZ             0x0000
#define GPY111_LEDCH_FBF_F04HZ             0x0040
#define GPY111_LEDCH_FBF_F08HZ             0x0080
#define GPY111_LEDCH_FBF_F16HZ             0x00C0
#define GPY111_LEDCH_SBF                   0x0030
#define GPY111_LEDCH_SBF_F02HZ             0x0000
#define GPY111_LEDCH_SBF_F04HZ             0x0010
#define GPY111_LEDCH_SBF_F08HZ             0x0020
#define GPY111_LEDCH_SBF_F16HZ             0x0030
#define GPY111_LEDCH_NACS                  0x0007
#define GPY111_LEDCH_NACS_NONE             0x0000
#define GPY111_LEDCH_NACS_LINK             0x0001
#define GPY111_LEDCH_NACS_PDOWN            0x0002
#define GPY111_LEDCH_NACS_EEE              0x0003
#define GPY111_LEDCH_NACS_ANEG             0x0004
#define GPY111_LEDCH_NACS_ABIST            0x0005
#define GPY111_LEDCH_NACS_CDIAG            0x0006
#define GPY111_LEDCH_NACS_TEST             0x0007

//LED Configuration L register
#define GPY111_LEDCL_SCAN                  0x0070
#define GPY111_LEDCL_SCAN_NONE             0x0000
#define GPY111_LEDCL_SCAN_LINK             0x0010
#define GPY111_LEDCL_SCAN_PDOWN            0x0020
#define GPY111_LEDCL_SCAN_EEE              0x0030
#define GPY111_LEDCL_SCAN_ANEG             0x0040
#define GPY111_LEDCL_SCAN_ABIST            0x0050
#define GPY111_LEDCL_SCAN_CDIAG            0x0060
#define GPY111_LEDCL_SCAN_TEST             0x0070
#define GPY111_LEDCL_CBLINK                0x0007
#define GPY111_LEDCL_CBLINK_NONE           0x0000
#define GPY111_LEDCL_CBLINK_LINK           0x0001
#define GPY111_LEDCL_CBLINK_PDOWN          0x0002
#define GPY111_LEDCL_CBLINK_EEE            0x0003
#define GPY111_LEDCL_CBLINK_ANEG           0x0004
#define GPY111_LEDCL_CBLINK_ABIST          0x0005
#define GPY111_LEDCL_CBLINK_CDIAG          0x0006
#define GPY111_LEDCL_CBLINK_TEST           0x0007

//Configuration for LED Pin 0 H register
#define GPY111_LED0H_CON                   0x00F0
#define GPY111_LED0H_CON_NONE              0x0000
#define GPY111_LED0H_CON_LINK10            0x0010
#define GPY111_LED0H_CON_LINK100           0x0020
#define GPY111_LED0H_CON_LINK10X           0x0030
#define GPY111_LED0H_CON_LINK1000          0x0040
#define GPY111_LED0H_CON_LINK10_0          0x0050
#define GPY111_LED0H_CON_LINK100X          0x0060
#define GPY111_LED0H_CON_LINK10XX          0x0070
#define GPY111_LED0H_CON_PDOWN             0x0080
#define GPY111_LED0H_CON_EEE               0x0090
#define GPY111_LED0H_CON_ANEG              0x00A0
#define GPY111_LED0H_CON_ABIST             0x00B0
#define GPY111_LED0H_CON_CDIAG             0x00C0
#define GPY111_LED0H_CON_COPPER            0x00D0
#define GPY111_LED0H_CON_FIBER             0x00E0
#define GPY111_LED0H_BLINKF                0x000F
#define GPY111_LED0H_BLINKF_NONE           0x0000
#define GPY111_LED0H_BLINKF_LINK10         0x0001
#define GPY111_LED0H_BLINKF_LINK100        0x0002
#define GPY111_LED0H_BLINKF_LINK10X        0x0003
#define GPY111_LED0H_BLINKF_LINK1000       0x0004
#define GPY111_LED0H_BLINKF_LINK10_0       0x0005
#define GPY111_LED0H_BLINKF_LINK100X       0x0006
#define GPY111_LED0H_BLINKF_LINK10XX       0x0007
#define GPY111_LED0H_BLINKF_PDOWN          0x0008
#define GPY111_LED0H_BLINKF_EEE            0x0009
#define GPY111_LED0H_BLINKF_ANEG           0x000A
#define GPY111_LED0H_BLINKF_ABIST          0x000B
#define GPY111_LED0H_BLINKF_CDIAG          0x000C

//Configuration for LED Pin 0 L register
#define GPY111_LED0L_BLINKS                0x00F0
#define GPY111_LED0L_BLINKS_NONE           0x0000
#define GPY111_LED0L_BLINKS_LINK10         0x0010
#define GPY111_LED0L_BLINKS_LINK100        0x0020
#define GPY111_LED0L_BLINKS_LINK10X        0x0030
#define GPY111_LED0L_BLINKS_LINK1000       0x0040
#define GPY111_LED0L_BLINKS_LINK10_0       0x0050
#define GPY111_LED0L_BLINKS_LINK100X       0x0060
#define GPY111_LED0L_BLINKS_LINK10XX       0x0070
#define GPY111_LED0L_BLINKS_PDOWN          0x0080
#define GPY111_LED0L_BLINKS_EEE            0x0090
#define GPY111_LED0L_BLINKS_ANEG           0x00A0
#define GPY111_LED0L_BLINKS_ABIST          0x00B0
#define GPY111_LED0L_BLINKS_CDIAG          0x00C0
#define GPY111_LED0L_PULSE                 0x000F
#define GPY111_LED0L_PULSE_NONE            0x0000
#define GPY111_LED0L_PULSE_TXACT           0x0001
#define GPY111_LED0L_PULSE_RXACT           0x0002
#define GPY111_LED0L_PULSE_COL             0x0004

//Configuration for LED Pin 1 H register
#define GPY111_LED1H_CON                   0x00F0
#define GPY111_LED1H_CON_NONE              0x0000
#define GPY111_LED1H_CON_LINK10            0x0010
#define GPY111_LED1H_CON_LINK100           0x0020
#define GPY111_LED1H_CON_LINK10X           0x0030
#define GPY111_LED1H_CON_LINK1000          0x0040
#define GPY111_LED1H_CON_LINK10_0          0x0050
#define GPY111_LED1H_CON_LINK100X          0x0060
#define GPY111_LED1H_CON_LINK10XX          0x0070
#define GPY111_LED1H_CON_PDOWN             0x0080
#define GPY111_LED1H_CON_EEE               0x0090
#define GPY111_LED1H_CON_ANEG              0x00A0
#define GPY111_LED1H_CON_ABIST             0x00B0
#define GPY111_LED1H_CON_CDIAG             0x00C0
#define GPY111_LED1H_CON_COPPER            0x00D0
#define GPY111_LED1H_CON_FIBER             0x00E0
#define GPY111_LED1H_BLINKF                0x000F
#define GPY111_LED1H_BLINKF_NONE           0x0000
#define GPY111_LED1H_BLINKF_LINK10         0x0001
#define GPY111_LED1H_BLINKF_LINK100        0x0002
#define GPY111_LED1H_BLINKF_LINK10X        0x0003
#define GPY111_LED1H_BLINKF_LINK1000       0x0004
#define GPY111_LED1H_BLINKF_LINK10_0       0x0005
#define GPY111_LED1H_BLINKF_LINK100X       0x0006
#define GPY111_LED1H_BLINKF_LINK10XX       0x0007
#define GPY111_LED1H_BLINKF_PDOWN          0x0008
#define GPY111_LED1H_BLINKF_EEE            0x0009
#define GPY111_LED1H_BLINKF_ANEG           0x000A
#define GPY111_LED1H_BLINKF_ABIST          0x000B
#define GPY111_LED1H_BLINKF_CDIAG          0x000C

//Configuration for LED Pin 1 L register
#define GPY111_LED1L_BLINKS                0x00F0
#define GPY111_LED1L_BLINKS_NONE           0x0000
#define GPY111_LED1L_BLINKS_LINK10         0x0010
#define GPY111_LED1L_BLINKS_LINK100        0x0020
#define GPY111_LED1L_BLINKS_LINK10X        0x0030
#define GPY111_LED1L_BLINKS_LINK1000       0x0040
#define GPY111_LED1L_BLINKS_LINK10_0       0x0050
#define GPY111_LED1L_BLINKS_LINK100X       0x0060
#define GPY111_LED1L_BLINKS_LINK10XX       0x0070
#define GPY111_LED1L_BLINKS_PDOWN          0x0080
#define GPY111_LED1L_BLINKS_EEE            0x0090
#define GPY111_LED1L_BLINKS_ANEG           0x00A0
#define GPY111_LED1L_BLINKS_ABIST          0x00B0
#define GPY111_LED1L_BLINKS_CDIAG          0x00C0
#define GPY111_LED1L_PULSE                 0x000F
#define GPY111_LED1L_PULSE_NONE            0x0000
#define GPY111_LED1L_PULSE_TXACT           0x0001
#define GPY111_LED1L_PULSE_RXACT           0x0002
#define GPY111_LED1L_PULSE_COL             0x0004

//Configuration for LED Pin 2 H register
#define GPY111_LED2H_CON                   0x00F0
#define GPY111_LED2H_CON_NONE              0x0000
#define GPY111_LED2H_CON_LINK10            0x0010
#define GPY111_LED2H_CON_LINK100           0x0020
#define GPY111_LED2H_CON_LINK10X           0x0030
#define GPY111_LED2H_CON_LINK1000          0x0040
#define GPY111_LED2H_CON_LINK10_0          0x0050
#define GPY111_LED2H_CON_LINK100X          0x0060
#define GPY111_LED2H_CON_LINK10XX          0x0070
#define GPY111_LED2H_CON_PDOWN             0x0080
#define GPY111_LED2H_CON_EEE               0x0090
#define GPY111_LED2H_CON_ANEG              0x00A0
#define GPY111_LED2H_CON_ABIST             0x00B0
#define GPY111_LED2H_CON_CDIAG             0x00C0
#define GPY111_LED2H_CON_COPPER            0x00D0
#define GPY111_LED2H_CON_FIBER             0x00E0
#define GPY111_LED2H_BLINKF                0x000F
#define GPY111_LED2H_BLINKF_NONE           0x0000
#define GPY111_LED2H_BLINKF_LINK10         0x0001
#define GPY111_LED2H_BLINKF_LINK100        0x0002
#define GPY111_LED2H_BLINKF_LINK10X        0x0003
#define GPY111_LED2H_BLINKF_LINK1000       0x0004
#define GPY111_LED2H_BLINKF_LINK10_0       0x0005
#define GPY111_LED2H_BLINKF_LINK100X       0x0006
#define GPY111_LED2H_BLINKF_LINK10XX       0x0007
#define GPY111_LED2H_BLINKF_PDOWN          0x0008
#define GPY111_LED2H_BLINKF_EEE            0x0009
#define GPY111_LED2H_BLINKF_ANEG           0x000A
#define GPY111_LED2H_BLINKF_ABIST          0x000B
#define GPY111_LED2H_BLINKF_CDIAG          0x000C

//Configuration for LED Pin 2 L register
#define GPY111_LED2L_BLINKS                0x00F0
#define GPY111_LED2L_BLINKS_NONE           0x0000
#define GPY111_LED2L_BLINKS_LINK10         0x0010
#define GPY111_LED2L_BLINKS_LINK100        0x0020
#define GPY111_LED2L_BLINKS_LINK10X        0x0030
#define GPY111_LED2L_BLINKS_LINK1000       0x0040
#define GPY111_LED2L_BLINKS_LINK10_0       0x0050
#define GPY111_LED2L_BLINKS_LINK100X       0x0060
#define GPY111_LED2L_BLINKS_LINK10XX       0x0070
#define GPY111_LED2L_BLINKS_PDOWN          0x0080
#define GPY111_LED2L_BLINKS_EEE            0x0090
#define GPY111_LED2L_BLINKS_ANEG           0x00A0
#define GPY111_LED2L_BLINKS_ABIST          0x00B0
#define GPY111_LED2L_BLINKS_CDIAG          0x00C0
#define GPY111_LED2L_PULSE                 0x000F
#define GPY111_LED2L_PULSE_NONE            0x0000
#define GPY111_LED2L_PULSE_TXACT           0x0001
#define GPY111_LED2L_PULSE_RXACT           0x0002
#define GPY111_LED2L_PULSE_COL             0x0004

//EEE Link-Fail Counter H register
#define GPY111_EEE_RXERR_LINK_FAIL_H_VAL   0x00FF

//EEE Link-Fail Counter L register
#define GPY111_EEE_RXERR_LINK_FAIL_L_VAL   0x00FF

//MII2 Control register
#define GPY111_MII2CTRL_RXSKEW             0x0070
#define GPY111_MII2CTRL_RXSKEW_SKEW_0N0    0x0000
#define GPY111_MII2CTRL_RXSKEW_SKEW_0N5    0x0010
#define GPY111_MII2CTRL_RXSKEW_SKEW_1N0    0x0020
#define GPY111_MII2CTRL_RXSKEW_SKEW_1N5    0x0030
#define GPY111_MII2CTRL_RXSKEW_SKEW_2N0    0x0040
#define GPY111_MII2CTRL_RXSKEW_SKEW_2N5    0x0050
#define GPY111_MII2CTRL_RXSKEW_SKEW_3N0    0x0060
#define GPY111_MII2CTRL_RXSKEW_SKEW_3N5    0x0070
#define GPY111_MII2CTRL_TXSKEW             0x0007
#define GPY111_MII2CTRL_TXSKEW_SKEW_0N0    0x0000
#define GPY111_MII2CTRL_TXSKEW_SKEW_0N5    0x0001
#define GPY111_MII2CTRL_TXSKEW_SKEW_1N0    0x0002
#define GPY111_MII2CTRL_TXSKEW_SKEW_1N5    0x0003
#define GPY111_MII2CTRL_TXSKEW_SKEW_2N0    0x0004
#define GPY111_MII2CTRL_TXSKEW_SKEW_2N5    0x0005
#define GPY111_MII2CTRL_TXSKEW_SKEW_3N0    0x0006
#define GPY111_MII2CTRL_TXSKEW_SKEW_3N5    0x0007

//Legacy LPI Configuration 0 register
#define GPY111_LEG_LPI_CFG0_HOLDOFF_100BT  0x00FF

//Legacy LPI Configuration 1 register
#define GPY111_LEG_LPI_CFG1_HOLDOFF_1000BT 0x00FF

//Wake-On-LAN Control register
#define GPY111_WOLCTRL_SPWD_EN             0x0004
#define GPY111_WOLCTRL_RES                 0x0002
#define GPY111_WOLCTRL_EN                  0x0001

//Wake-On-LAN Address Byte 0 register
#define GPY111_WOLAD0_VAL                  0x00FF

//Wake-On-LAN Address Byte 1 register
#define GPY111_WOLAD1_VAL                  0x00FF

//Wake-On-LAN Address Byte 2 register
#define GPY111_WOLAD2_VAL                  0x00FF

//Wake-On-LAN Address Byte 3 register
#define GPY111_WOLAD3_VAL                  0x00FF

//Wake-On-LAN Address Byte 4 register
#define GPY111_WOLAD4_VAL                  0x00FF

//Wake-On-LAN Address Byte 5 register
#define GPY111_WOLAD5_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 0 register
#define GPY111_WOLPW0_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 1 register
#define GPY111_WOLPW1_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 2 register
#define GPY111_WOLPW2_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 3 register
#define GPY111_WOLPW3_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 4 register
#define GPY111_WOLPW4_VAL                  0x00FF

//Wake-On-LAN SecureON Password Byte 5 register
#define GPY111_WOLPW5_VAL                  0x00FF

//Legacy LPI Configuration 2 register
#define GPY111_LEG_LPI_CFG2_IPG            0x00FF
#define GPY111_LEG_LPI_CFG2_IPG_DEFAULT    0x000E

//Legacy LPI Configuration 3 register
#define GPY111_LEG_LPI_CFG3_IDLE           0x00FF
#define GPY111_LEG_LPI_CFG3_IDLE_DEFAULT   0x0040

//C++ guard
#ifdef __cplusplus
extern "C" {
#endif

//GPY111 Ethernet PHY driver
extern const PhyDriver gpy111PhyDriver;

//GPY111 related functions
error_t gpy111Init(NetInterface *interface);
void gpy111InitHook(NetInterface *interface);

void gpy111Tick(NetInterface *interface);

void gpy111EnableIrq(NetInterface *interface);
void gpy111DisableIrq(NetInterface *interface);

void gpy111EventHandler(NetInterface *interface);

void gpy111WritePhyReg(NetInterface *interface, uint8_t address,
   uint16_t data);

uint16_t gpy111ReadPhyReg(NetInterface *interface, uint8_t address);

void gpy111DumpPhyReg(NetInterface *interface);

void gpy111WriteMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr, uint16_t data);

uint16_t gpy111ReadMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr);

//C++ guard
#ifdef __cplusplus
}
#endif

#endif
