/**
 * @file rtl8211f_driver.c
 * @brief RTL8211F Gigabit Ethernet PHY driver
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Open
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL NIC_TRACE_LEVEL

//Dependencies
#include "core/net.h"
#include "drivers/phy/rtl8211f_driver.h"
#include "debug.h"


/**
 * @brief RTL8211F Ethernet PHY driver
 **/

const PhyDriver rtl8211fPhyDriver =
{
   rtl8211fInit,
   rtl8211fTick,
   rtl8211fEnableIrq,
   rtl8211fDisableIrq,
   rtl8211fEventHandler
};


/**
 * @brief RTL8211F PHY transceiver initialization
 * @param[in] interface Underlying network interface
 * @return Error code
 **/

error_t rtl8211fInit(NetInterface *interface)
{
   //Debug message
   TRACE_INFO("Initializing RTL8211F...\r\n");

   //Undefined PHY address?
   if(interface->phyAddr >= 32)
   {
      //Use the default address
      interface->phyAddr = RTL8211F_PHY_ADDR;
   }

   //Initialize serial management interface
   if(interface->smiDriver != NULL)
   {
      interface->smiDriver->init();
   }

   //Initialize external interrupt line driver
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->init();
   }

   //Reset PHY transceiver
   rtl8211fWritePhyReg(interface, RTL8211F_BMCR, RTL8211F_BMCR_RESET);

   //Wait for the reset to complete
   while(rtl8211fReadPhyReg(interface, RTL8211F_BMCR) & RTL8211F_BMCR_RESET)
   {
   }

   //Dump PHY registers for debugging purpose
   rtl8211fDumpPhyReg(interface);

   //The PHY will generate interrupts when link status changes are detected
   rtl8211fWritePhyReg(interface, RTL8211F_INER, RTL8211F_INER_AN_COMPLETE |
      RTL8211F_INER_LINK_STATUS);

   //Perform custom configuration
   rtl8211fInitHook(interface);

   //Force the TCP/IP stack to poll the link state at startup
   interface->phyEvent = TRUE;
   //Notify the TCP/IP stack of the event
   osSetEvent(&interface->netContext->event);

   //Successful initialization
   return NO_ERROR;
}


/**
 * @brief RTL8211F custom configuration
 * @param[in] interface Underlying network interface
 **/

__weak_func void rtl8211fInitHook(NetInterface *interface)
{
}


/**
 * @brief RTL8211F timer handler
 * @param[in] interface Underlying network interface
 **/

void rtl8211fTick(NetInterface *interface)
{
   uint16_t value;
   bool_t linkState;

   //No external interrupt line driver?
   if(interface->extIntDriver == NULL)
   {
      //Read basic status register
      value = rtl8211fReadPhyReg(interface, RTL8211F_BMSR);
      //Retrieve current link state
      linkState = (value & RTL8211F_BMSR_LINK_STATUS) ? TRUE : FALSE;

      //Link up event?
      if(linkState && !interface->linkState)
      {
         //Set event flag
         interface->phyEvent = TRUE;
         //Notify the TCP/IP stack of the event
         osSetEvent(&interface->netContext->event);
      }
      //Link down event?
      else if(!linkState && interface->linkState)
      {
         //Set event flag
         interface->phyEvent = TRUE;
         //Notify the TCP/IP stack of the event
         osSetEvent(&interface->netContext->event);
      }
   }
}


/**
 * @brief Enable interrupts
 * @param[in] interface Underlying network interface
 **/

void rtl8211fEnableIrq(NetInterface *interface)
{
   //Enable PHY transceiver interrupts
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->enableIrq();
   }
}


/**
 * @brief Disable interrupts
 * @param[in] interface Underlying network interface
 **/

void rtl8211fDisableIrq(NetInterface *interface)
{
   //Disable PHY transceiver interrupts
   if(interface->extIntDriver != NULL)
   {
      interface->extIntDriver->disableIrq();
   }
}


/**
 * @brief RTL8211F event handler
 * @param[in] interface Underlying network interface
 **/

void rtl8211fEventHandler(NetInterface *interface)
{
   uint16_t status;

   //Read status register to acknowledge the interrupt
   status = rtl8211fReadPhyReg(interface, RTL8211F_INSR);

   //Link status change?
   if((status & (RTL8211F_INSR_AN_COMPLETE | RTL8211F_INSR_LINK_STATUS)) != 0)
   {
      //Any link failure condition is latched in the BMSR register. Reading
      //the register twice will always return the actual link status
      status = rtl8211fReadPhyReg(interface, RTL8211F_BMSR);
      status = rtl8211fReadPhyReg(interface, RTL8211F_BMSR);

      //Check link state
      if((status & RTL8211F_BMSR_LINK_STATUS) != 0)
      {
         //Read PHY status register
         status = rtl8211fReadPhyReg(interface, RTL8211F_PHYSR);

         //Check current speed
         switch(status & RTL8211F_PHYSR_SPEED)
         {
         //10BASE-T
         case RTL8211F_PHYSR_SPEED_10MBPS:
            interface->linkSpeed = NIC_LINK_SPEED_10MBPS;
            break;
         //100BASE-TX
         case RTL8211F_PHYSR_SPEED_100MBPS:
            interface->linkSpeed = NIC_LINK_SPEED_100MBPS;
            break;
         //1000BASE-T
         case RTL8211F_PHYSR_SPEED_1000MBPS:
            interface->linkSpeed = NIC_LINK_SPEED_1GBPS;
            break;
         //Unknown speed
         default:
            //Debug message
            TRACE_WARNING("Invalid speed\r\n");
            break;
         }

         //Check current duplex mode
         if((status & RTL8211F_PHYSR_DUPLEX) != 0)
         {
            interface->duplexMode = NIC_FULL_DUPLEX_MODE;
         }
         else
         {
            interface->duplexMode = NIC_HALF_DUPLEX_MODE;
         }

         //Update link state
         interface->linkState = TRUE;

         //Adjust MAC configuration parameters for proper operation
         interface->nicDriver->updateMacConfig(interface);
      }
      else
      {
         //Update link state
         interface->linkState = FALSE;
      }

      //Process link state change event
      nicNotifyLinkChange(interface);
   }
}


/**
 * @brief Write PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address PHY register address
 * @param[in] data Register value
 **/

void rtl8211fWritePhyReg(NetInterface *interface, uint8_t address,
   uint16_t data)
{
   //Write the specified PHY register
   if(interface->smiDriver != NULL)
   {
      interface->smiDriver->writePhyReg(SMI_OPCODE_WRITE,
         interface->phyAddr, address, data);
   }
   else
   {
      interface->nicDriver->writePhyReg(SMI_OPCODE_WRITE,
         interface->phyAddr, address, data);
   }
}


/**
 * @brief Read PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address PHY register address
 * @return Register value
 **/

uint16_t rtl8211fReadPhyReg(NetInterface *interface, uint8_t address)
{
   uint16_t data;

   //Read the specified PHY register
   if(interface->smiDriver != NULL)
   {
      data = interface->smiDriver->readPhyReg(SMI_OPCODE_READ,
         interface->phyAddr, address);
   }
   else
   {
      data = interface->nicDriver->readPhyReg(SMI_OPCODE_READ,
         interface->phyAddr, address);
   }

   //Return the value of the PHY register
   return data;
}


/**
 * @brief Dump PHY registers for debugging purpose
 * @param[in] interface Underlying network interface
 **/

void rtl8211fDumpPhyReg(NetInterface *interface)
{
   uint8_t i;

   //Loop through PHY registers
   for(i = 0; i < 32; i++)
   {
      //Display current PHY register
      TRACE_DEBUG("%02" PRIu8 ": 0x%04" PRIX16 "\r\n", i,
         rtl8211fReadPhyReg(interface, i));
   }

   //Terminate with a line feed
   TRACE_DEBUG("\r\n");
}


/**
 * @brief Write MMD register
 * @param[in] interface Underlying network interface
 * @param[in] devAddr Device address
 * @param[in] regAddr Register address
 * @param[in] data MMD register value
 **/

void rtl8211fWriteMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr, uint16_t data)
{
   //Select register operation
   rtl8211fWritePhyReg(interface, RTL8211F_MMDACR,
      RTL8211F_MMDACR_FUNC_ADDR | (devAddr & RTL8211F_MMDACR_DEVAD));

   //Write MMD register address
   rtl8211fWritePhyReg(interface, RTL8211F_MMDAADR, regAddr);

   //Select data operation
   rtl8211fWritePhyReg(interface, RTL8211F_MMDACR,
      RTL8211F_MMDACR_FUNC_DATA_NO_POST_INC | (devAddr & RTL8211F_MMDACR_DEVAD));

   //Write the content of the MMD register
   rtl8211fWritePhyReg(interface, RTL8211F_MMDAADR, data);
}


/**
 * @brief Read MMD register
 * @param[in] interface Underlying network interface
 * @param[in] devAddr Device address
 * @param[in] regAddr Register address
 * @return MMD register value
 **/

uint16_t rtl8211fReadMmdReg(NetInterface *interface, uint8_t devAddr,
   uint16_t regAddr)
{
   //Select register operation
   rtl8211fWritePhyReg(interface, RTL8211F_MMDACR,
      RTL8211F_MMDACR_FUNC_ADDR | (devAddr & RTL8211F_MMDACR_DEVAD));

   //Write MMD register address
   rtl8211fWritePhyReg(interface, RTL8211F_MMDAADR, regAddr);

   //Select data operation
   rtl8211fWritePhyReg(interface, RTL8211F_MMDACR,
      RTL8211F_MMDACR_FUNC_DATA_NO_POST_INC | (devAddr & RTL8211F_MMDACR_DEVAD));

   //Read the content of the MMD register
   return rtl8211fReadPhyReg(interface, RTL8211F_MMDAADR);
}
