/**
 * @file main.c
 * @brief CycloneBOOT compatible Update Image Builder
 *
 * @section License
 *
 * Copyright (C) 2021-2026 Oryx Embedded SARL. All rights reserved.
 * 
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 2.6.0
 **/

#include <stdio.h>
#include "cli.h"
#include "header.h"
#include "body.h"
#include "footer.h"
#include "utils.h"
#include "main.h"
#include "config/ImageBuilderConfig.h"


#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
HmacDrbgContext hmacDrbgContext = {0};
#endif

/**
 * Main entry point of the program.
 */
int main(int argc, char *argv[])
{
   // flags
   error_t status = NO_ERROR;
   uint8_t encrypted = 0;
   uint8_t keygen = 0;

   // structures
   ImageHeader header = {0};
   ImageBody body = {0};
   UpdateImage updateImage = {0};

   CipherInfo cipherInfo = {0};
   CheckDataInfo checkDataInfo = {0};
   struct builder_cli_configuration cli_config = {0};

   // buffers
   char check_data[CHECK_DATA_LENGTH] = {0};

   char iv[INIT_VECTOR_LENGTH];
   size_t ivSize = INIT_VECTOR_LENGTH;

   // Generate an initialization vector for cipher operations (AES-CBC)
   seedInitVector((uint8_t *)iv, INIT_VECTOR_LENGTH);

   // Get command-line options supplied by the user
   status = parse_options(argc, argv, &cli_config);

   if(status == ERROR_FAILURE)
   {
      printf("Something went wrong while parsing command line options.\n");
      return ERROR_FAILURE;
   }

   if(status == CLI_OK)
      return NO_ERROR;

   // Should the image be encrypted?
   if(cli_config.encryption_key != NULL)
   {
      encrypted = 1;
   }

   if(cli_config.keygen != 0)
   {
      keygen = 1;
   }

#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
   // Initialize Crypto stuff
   if(encrypted)
   {
      cipherInfo.hmacDrbgContext = &hmacDrbgContext;
      cipherInfo.prngAlgo = (PrngAlgo *)HMAC_DRBG_PRNG_ALGO;

      cipherInfo.cipherKey = cli_config.encryption_key;
      cipherInfo.cipherKeySize = cli_config.encryption_key_len;

      cipherInfo.iv = iv;
      cipherInfo.ivSize = ivSize;

      status = init_crypto(&cipherInfo);
      if(status != NO_ERROR)
      {
         printf("Something went wrong in init_crypto.\r\n");
         return ERROR_FAILURE;
      }
   }
   else
   {
      cipherInfo.hmacDrbgContext = &hmacDrbgContext;
      cipherInfo.prngAlgo = (PrngAlgo *)HMAC_DRBG_PRNG_ALGO;

      status = init_crypto(&cipherInfo);
      if(status != NO_ERROR)
      {
         printf("Something went wrong in init_crypto.\r\n");
         return ERROR_FAILURE;
      }
   }

   // Generate a keypair and return
   if(keygen)
   {
      status = generate_key_pair(&cli_config, &cipherInfo);
      if(status)
      {
         printf("Something went wrong in generate_key_pair.\r\n");
         return ERROR_FAILURE;
      }
      else
      {
         printf("Keypair generated successfully.\r\n");
         return NO_ERROR;
      }
   }
#endif

   // Associate update image type
   if(cli_config.update_type)
   {
      if(strcasecmp(cli_config.update_type, "app") == 0)
         header.imgType = IMAGE_TYPE_APP;
      if(strcasecmp(cli_config.update_type, "boot") == 0)
      {
         header.imgType = IMAGE_TYPE_BOOTLOADER;
      }
      else
      {
         header.imgType = IMAGE_TYPE_APP;
      }
   }

   // Make header
   status = headerMake(&header, &cli_config, cipherInfo);

   if(status != NO_ERROR)
   {
      printf("Something went wrong while making the header.\n");
      return ERROR_FAILURE;
   }
   // Make body
   status = bodyMake(&header, &body, cipherInfo);
   if(status != NO_ERROR)
   {
      printf("Something went wrong while making the body.\n");
      return ERROR_FAILURE;
   }
   // Make footer
   // Determine which check data mechanism to use based on user-supplied parameters
   // simple integrity?
   if(cli_config.integrity_algo != NULL)
   {
      checkDataInfo.integrity = 1;
      checkDataInfo.integrity_algo = cli_config.integrity_algo;
   }


#if defined(VARIANT_OPEN) || defined(VARIANT_EVAL) || defined(VARIANT_ULTIMATE)
   // authentication required?
   if(cli_config.authentication_algo != NULL)
   {
      checkDataInfo.authentication = 1;
      checkDataInfo.auth_algo = cli_config.authentication_algo;
      checkDataInfo.authKey = (char *)cli_config.authentication_key;
      checkDataInfo.authKeySize = strlen((char *)cli_config.authentication_key);
   }
   // signature required ?
   if(cli_config.signature_algo != NULL)
   {
      checkDataInfo.signature = 1;
      checkDataInfo.sign_algo = cli_config.signature_algo;
      checkDataInfo.signKey = cli_config.signature_key;
      checkDataInfo.signKeySize = strlen(cli_config.signature_key);
      checkDataInfo.signHashAlgo = SHA256_HASH_ALGO;
   }
#endif

   // Make the footer (the check data section mainly), based on the image verification method chosen
   status = footerMake(&header, &body, &cipherInfo, &checkDataInfo, check_data);
   if(status != NO_ERROR)
   {
      printf("Something went wrong while making the footer.\n");
      return ERROR_FAILURE;
   }

   updateImage.header = &header;
   updateImage.body = &body;

   // Now write the whole image to a file in the disk.
   write_image_to_file(&updateImage, &cipherInfo, cli_config.output);

   // Free allocated resources
   if(cli_config.encryption_key_hex)
      free(cli_config.encryption_key);

   return EXIT_SUCCESS;
}
