/**
 * @file lan8740.c
 * @brief LAN8740 Ethernet PHY transceiver
 *
 * @section License
 *
 * Copyright (C) 2010-2014 Oryx Embedded. All rights reserved.
 *
 * This file is part of CycloneTCP Open.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @author Oryx Embedded (www.oryx-embedded.com)
 * @version 1.4.2
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL NIC_TRACE_LEVEL

//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   #include <p32xxxx.h>
#endif

//Dependencies
#include "tcp_ip_stack.h"
#include "lan8740.h"
#include "debug.h"


/**
 * @brief LAN8740 Ethernet PHY driver
 **/

const PhyDriver lan8740PhyDriver =
{
   lan8740Init,
   lan8740Tick,
   lan8740EnableIrq,
   lan8740DisableIrq,
   lan8740EventHandler,
};


/**
 * @brief LAN8740 PHY transceiver initialization
 * @param[in] interface Underlying network interface
 * @return Error code
 **/

error_t lan8740Init(NetInterface *interface)
{
//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   //Debug message
   TRACE_INFO("Initializing LAN8740...\r\n");

   //Remap INT4 to RPC13
   INT4R = 7;
   //Configure edge polarity for INT4 interrupt (falling edge)
   INTCONCLR = _INTCON_INT4EP_MASK;
   //Set interrupt priority
   IPC5CLR = _IPC5_INT4IP_MASK;
   IPC5SET = (2 << _IPC5_INT4IP_POSITION);
   //Set interrupt subpriority
   IPC5CLR = _IPC5_INT4IS_MASK;
   IPC5SET = (0 << _IPC5_INT4IS_POSITION);
#endif

   //Reset PHY transceiver (soft reset)
   lan8740WritePhyReg(interface, LAN8740_PHY_REG_BMCR, BMCR_RESET);
   //Wait for the reset to complete
   while(lan8740ReadPhyReg(interface, LAN8740_PHY_REG_BMCR) & BMCR_RESET);

   //The PHY will generate interrupts when link status changes are detected
   lan8740WritePhyReg(interface, LAN8740_PHY_REG_IMR, IMR_AN_COMPLETE | IMR_LINK_DOWN);

   //Dump PHY registers for debugging purpose
   lan8740DumpPhyReg(interface);

   //Successful initialization
   return NO_ERROR;
}


/**
 * @brief LAN8740 timer handler
 * @param[in] interface Underlying network interface
 **/

void lan8740Tick(NetInterface *interface)
{
}


/**
 * @brief Enable interrupts
 * @param[in] interface Underlying network interface
 **/

void lan8740EnableIrq(NetInterface *interface)
{
//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   //Enable PHY transceiver interrupts
   IEC0SET = _IEC0_INT4IE_MASK;
#endif
}


/**
 * @brief Disable interrupts
 * @param[in] interface Underlying network interface
 **/

void lan8740DisableIrq(NetInterface *interface)
{
//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   //Disable PHY transceiver interrupts
   IEC0CLR = _IEC0_INT4IE_MASK;
#endif
}


/**
 * @brief LAN8740 interrupt service routine
 **/

void lan8740IrqHandler(void)
{
//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   //Point to the structure describing the network interface
   NetInterface *interface = &netInterface[0];

   //Check interrupt status
   if(IFS0 & _IFS0_INT4IF_MASK)
   {
      //Clear interrupt flag
      IFS0CLR = _IFS0_INT4IF_MASK;
      //A PHY event is pending...
      interface->phyEvent = TRUE;
      //Notify the user that the link state has changed
      osEventSetFromIrq(interface->nicRxEvent);
   }
#endif
}


/**
 * @brief LAN8740 event handler
 * @param[in] interface Underlying network interface
 * @return TRUE if a link state change notification is received
 **/

bool_t lan8740EventHandler(NetInterface *interface)
{
//PIC32MZ EC Starter Kit?
#if defined(USE_PIC32MZ_EC_STARTER_KIT)
   uint16_t value;

   //Read status register to acknowledge the interrupt
   value = lan8740ReadPhyReg(interface, LAN8740_PHY_REG_ISR);

   //Link status change?
   if(value & (IMR_AN_COMPLETE | IMR_LINK_DOWN))
   {
      //Read basic status register
      value = lan8740ReadPhyReg(interface, LAN8740_PHY_REG_BMSR);

      //Link is up?
      if(value & BMSR_LINK_STATUS)
      {
         //Read PHY special control/status register
         value = lan8740ReadPhyReg(interface, LAN8740_PHY_REG_PSCSR);

         //Check current operation mode
         switch(value & PSCSR_HCDSPEED_MASK)
         {
         //10BASE-T
         case PSCSR_HCDSPEED_10BT:
            interface->speed100 = FALSE;
            interface->fullDuplex = FALSE;
            break;
         //10BASE-T full-duplex
         case PSCSR_HCDSPEED_10BT_FD:
            interface->speed100 = FALSE;
            interface->fullDuplex = TRUE;
            break;
         //100BASE-TX
         case PSCSR_HCDSPEED_100BTX:
            interface->speed100 = TRUE;
            interface->fullDuplex = FALSE;
            break;
         //100BASE-TX full-duplex
         case PSCSR_HCDSPEED_100BTX_FD:
            interface->speed100 = TRUE;
            interface->fullDuplex = TRUE;
            break;
         //Unknown operation mode
         default:
            //Debug message
            TRACE_WARNING("Invalid Duplex mode\r\n");
            break;
         }

         //Update link state
         interface->linkState = TRUE;
         //Display link state
         TRACE_INFO("Link is up (%s)...\r\n", interface->name);

         //Display actual speed and duplex mode
         TRACE_INFO("%s %s\r\n",
            interface->speed100 ? "100BASE-TX" : "10BASE-T",
            interface->fullDuplex ? "Full-Duplex" : "Half-Duplex");
      }
      else
      {
         //Update link state
         interface->linkState = FALSE;
         //Display link state
         TRACE_INFO("Link is down (%s)...\r\n", interface->name);
      }

      //Notify the user that the link state has changed
      return TRUE;
   }
   else
   {
      //No link state change...
      return FALSE;
   }
#endif
}


/**
 * @brief Write PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address Register address
 * @param[in] data Register value
 **/

void lan8740WritePhyReg(NetInterface *interface, uint8_t address, uint16_t data)
{
   //Write the specified PHY register
   interface->nicDriver->writePhyReg(LAN8740_PHY_ADDR, address, data);
}


/**
 * @brief Read PHY register
 * @param[in] interface Underlying network interface
 * @param[in] address PHY register address
 * @return Register value
 **/

uint16_t lan8740ReadPhyReg(NetInterface *interface, uint8_t address)
{
   //Read the specified PHY register
   return interface->nicDriver->readPhyReg(LAN8740_PHY_ADDR, address);
}


/**
 * @brief Dump PHY registers for debugging purpose
 * @param[in] interface Underlying network interface
 **/

void lan8740DumpPhyReg(NetInterface *interface)
{
   uint8_t i;

   //Loop through PHY registers
   for(i = 0; i < 32; i++)
   {
      //Display current PHY register
      TRACE_DEBUG("%02" PRIu8 ": 0x%04" PRIX16 "\r\n", i, lan8740ReadPhyReg(interface, i));
   }

   //Terminate with a line feed
   TRACE_DEBUG("\r\n");
}
