/**
 * @file ext_int.c
 * @brief External interrupt line driver
 *
 * @section License
 *
 * Copyright (C) 2010-2014 Oryx Embedded. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @author Oryx Embedded (www.oryx-embedded.com)
 * @version 1.4.2
 **/

#include <p32xxxx.h>
#include "tcp_ip_stack.h"
#include "enc624j600.h"
#include "ext_int_driver.h"
#include "debug.h"


/**
 * @brief External interrupt line driver
 **/

const ExtIntDriver extIntDriver =
{
   extIntInit,
   extIntEnableIrq,
   extIntDisableIrq
};


/**
 * @brief EXTI configuration
 * @return Error code
 **/

error_t extIntInit(void)
{
   //Configure edge polarity for INT2 interrupt (falling edge)
   INTCONCLR = _INTCON_INT2EP_MASK;
   //Set interrupt priority
   IPC2CLR = _IPC2_INT2IP_MASK;
   IPC2SET = (2 << _IPC2_INT2IP_POSITION);
   //Set interrupt subpriority
   IPC2CLR = _IPC2_INT2IS_MASK;
   IPC2SET = (0 << _IPC2_INT2IS_POSITION);

   //Successful processing
   return NO_ERROR;
}


/**
 * @brief Enable external interrupts
 **/

void extIntEnableIrq(void)
{
   //Enable interrupts
   IEC0SET = _IEC0_INT2IE_MASK;
}


/**
 * @brief Disable external interrupts
 **/

void extIntDisableIrq(void)
{
   //Disable interrupts
   IEC0CLR = _IEC0_INT2IE_MASK;
}


/**
 * @brief External interrupt handler
 **/

void extIntHandler(void)
{
   //Point to the structure describing the network interface
   NetInterface *interface = &netInterface[0];

   //Check interrupt status
   if(IFS0 & _IFS0_INT2IF_MASK)
   {
      //Clear interrupt flag
      IFS0CLR = _IFS0_INT2IF_MASK;
      //Call interrupt handler
      enc624j600IrqHandler(interface);
   }
}
